#! /usr/bin/env python3
# -*- coding: utf-8 -

# z3d_map_to_gpx.py
#
# D.Ahlgrimm    01.02.2022

import argparse
import os
import sys

VERSION="1.0"

HEADER="""<?xml version="1.0" encoding="UTF-8" standalone="yes" ?>
<gpx version="1.1" creator="z3d_map_to_gpx" xmlns="http://www.topografix.com/GPX/1/1" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd">
  <trk>
<!--    <desc>%s</desc> -->
    <trkseg>
"""

LINE="""      <trkpt lat="%s" lon="%s"><time>%s</time><extensions><speedkmh>%s</speedkmh></extensions></trkpt>\n"""

FOOTER="""    </trkseg>
  </trk>
</gpx>
"""


# ----------------------------------------------------------------------
# Liefert zu einer Koordinate in "Grad + Dezimalminuten"
# den entsprechenden Wert in "Dezimalgrad".
def ddm2dd(ddmf):
    if isinstance(ddmf, str):
        ddmf=float(ddmf)
    ddm = ddmf/100.0
    d = int(ddm)
    m = int((ddm-d)*100)
    s = round((((ddm-d)*100)-m)*60, 4)
    return round(d+m/60.0+s/3600.0, 6)


# ----------------------------------------------------------------------
# Setzt den ArgParser auf.
def setupArgParser():
    desc="%(prog)s v"+VERSION+" - Create gpx-file from all map-files in a given folder."
    usage='%(prog)s [-h] [-f FOLDER] [-o OUTPUT] [-d DESCRIPTION]'
    epilog='Examples:\n' \
         '  %(prog)s\n' \
         '     write ./z3d_trk.gpx from all *.map-files in .\n'      \
         '  %(prog)s -f /tmp/tour -o ~/abc -d "some description"\n' \
         '    write ~/abc.gpx for all *.map-files in /tmp/tour and set\n' \
         '    gpx-description to "some description"'

    parser=argparse.ArgumentParser(formatter_class=argparse.RawDescriptionHelpFormatter,
                                    description=desc, usage=usage, epilog=epilog)
    parser.add_argument('-f', dest='folder', type=str,
                help='source directory (default is ".")')
    parser.add_argument('-o', dest='output', default="z3d_trk.gpx", type=str,
                help='output gpx-filename (default is "z3d_trk")')
    parser.add_argument('-d', dest='desc', type=str,
                help='description/title in gpx-file (default is <input dirname>)')
    args=parser.parse_args()

    if args.folder and not os.path.isdir(args.folder):
        parser.error("folder not an existing directory!")
    return(args)


# ----------------------------------------------------------------------
# 
if __name__=="__main__":
    args=setupArgParser()

    if args.folder:
        args.folder=os.path.normpath(args.folder)
    else:
        args.folder=os.getcwd()
    if not args.output.endswith(".gpx"):
        args.output+=".gpx"
    if not args.desc:
        args.desc=os.path.basename(args.folder)

    #print("args.folder", args.folder)
    #print("args.output", args.output)
    #print("args.desc", args.desc)

    file_count=points_cnt=0
    fo=open(args.output, "wt")
    fo.write(HEADER%(args.desc,))
    for curdir, subdirlst, filenamelst in os.walk(args.folder):
        for fn in sorted(filenamelst):
            if fn.endswith(".map"):
                file_count+=1
                with open(os.path.join(args.folder, fn), 'r') as fl:
                    lines=fl.readlines()
                for ln in lines:
                    lns=ln.strip()
                    if lns.startswith("A,"):
                        elems=lns.split(",")
                        _, dt, tm, lat, _, lon, _, spd, *_=elems
                        dd, mm, yy=dt[0:2], dt[2:4], dt[4:6]
                        hh, mn, ss=tm[0:2], tm[2:4], tm[4:6]
                        dt="20%02d-%02d-%02dT%02d:%02d:%02dZ"%(int(yy), int(mm), int(dd), int(hh), int(mn), int(ss))
                        #print(dt, ddm2dd(lat), ddm2dd(lon), spd)
                        fo.write(LINE%(ddm2dd(lat), ddm2dd(lon), dt, spd))
                        points_cnt+=1
    fo.write(FOOTER)
    print("files parsed  :", file_count)
    print("points written:", points_cnt)
