#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ###########################################################
# Ein kleines GUI als Test der OSM_screen-Klasse.
# Also Anzeige der OpenStreetMap-Karte mit der Möglichkeit
# des scrollens, zoomens und einer Track-Anzeige.
#
# Detlev Ahlgrimm, 2017
#
#
# 08.04.2017  erste Version
# 10.04.2017  kompletter Umbau
# 14.04.2017  scrollMapToPoint() und Server-Wahl mittels SOURCE eingebaut
# 15.04.2017  Zoomlevel-Anpassung nach Track-Laden eingebaut
# 16.04.2017  Umbau wegen Umstellung auf wx.MemoryDC(), Auslagerung von Funktionen
# 17.04.2017  unabhängige Point-Listen und drawLinesForAP() zugefügt
#             Entfernungs-Anzeige bei AP-Sichtungs-Punkten
#

import wx               # python-wxWidgets-2.8.12.1-10.4.1.x86_64
import time
import math
import os
from lxml import etree  # python-lxml-3.3.5-2.1.4.x86_64
from datetime import datetime, tzinfo, timedelta
from time import mktime
from calendar import timegm

from OSMscr2 import OSMscr2

USE_DB=False
if USE_DB:
  from Database4wd        import Database
  DATABASE_FILE="/home/dede/daten/wardriving.sqlite"

VERSION="1.3a"

HOME_DIR=os.path.expanduser('~')
# http://wiki.openstreetmap.org/wiki/Slippy_map_tilenames#Tile_servers
# http://www.thunderforest.com/docs/apikeys/
SOURCE={
  "OSM"   : { "tile_url"  : "http://tile.openstreetmap.org/{z}/{x}/{y}.png",
              "tile_size" : 256,
              "max_zoom"  : 19,
              "cache_dir" : os.path.join(HOME_DIR, "wd", "osm")
            },
  "OCM"   : { "tile_url"  : "https://tile.thunderforest.com/cycle/{z}/{x}/{y}.png?apikey=....der.ist.nur.fuer.mich.......",
              "tile_size" : 256,
              "max_zoom"  : 18,
              "cache_dir" : os.path.join(HOME_DIR, "wd", "ocm")
            },
  "FAU"   : { "tile_url"  : "https://osm.rrze.fau.de/osmde/{z}/{x}/{y}.png",
              "tile_size" : 256,
              "max_zoom"  : 19,
              "cache_dir" : os.path.join(HOME_DIR, "wd", "fau")
            },
  "FAUHD" : { "tile_url"  : "https://osm.rrze.fau.de/osmhd/{z}/{x}/{y}.png",
              "tile_size" : 512,
              "max_zoom"  : 19,
              "cache_dir" : os.path.join(HOME_DIR, "wd", "fauhd")
            },
}
SERVER="OSM"


# Notation für Variablennamen:
#   variableXY  - Koordinaten in Pixel auf dem DC
#   variableTN  - TileNumber
#   variableLL  - Koordinaten in Lat/Lon

# ###########################################################
# Quelle: https://aboutsimon.com/blog/2013/06/06/Datetime-hell-Time-zone-aware-to-UNIX-timestamp.html
class UTC(tzinfo):
    """UTC"""
    def utcoffset(self, dt):
        return timedelta(0)
    def tzname(self, dt):
        return "UTC"
    def dst(self, dt):
        return timedelta(0)
# ###########################################################
# Liefert den UNIX-Timestamp für einen Timestamp-String in
# UTC.
# Erweitert mit: http://stackoverflow.com/a/33148723/3588613
def getTimestampStringAsUTC(strg):
  if "." in strg:
    datetime_obj = datetime.strptime(strg, '%Y-%m-%dT%H:%M:%S.%fZ')
  else:
    datetime_obj = datetime.strptime(strg, '%Y-%m-%dT%H:%M:%SZ')
  datetime_obj = datetime_obj.replace(tzinfo=UTC())
  timestamp = timegm(datetime_obj.timetuple())
  return(timestamp)


# ###########################################################
# eine kleine Klasse, um neue Dateien via Drag&Drop öffnen
# zu können.
class FileDrop(wx.FileDropTarget):
  def __init__(self, window):
    wx.FileDropTarget.__init__(self)
    self.window=window

  def OnDropFiles(self, x, y, filenames):
    if len(filenames)>1:
      wx.MessageBox('Es kann nur eine Datei geladen werden!', 'Fehler', wx.OK | wx.ICON_ERROR)
      return
    self.window.fileDropped(filenames[0])


# ######################################################################
# Das eigentliche Fenster mit der OSM-Ansicht.
class MapWindow(wx.Window):
  def __init__(self, parent, centerLL, zoom, border, title, size):
    wx.Window.__init__(self, parent)
    self.parent=parent
    self.centerLL=centerLL
    self.zoom=zoom
    self.border=border

    self.wardriving_mode=False
    self.wardriving_point_handle=None
    self.createMenu()

    self.Bind(wx.EVT_PAINT,         self.onPaint)
    self.Bind(wx.EVT_LEFT_DOWN,     self.onLeftDown)
    self.Bind(wx.EVT_LEFT_UP,       self.onLeftUp)
    self.Bind(wx.EVT_MOTION,        self.onMotion)
    self.Bind(wx.EVT_MOUSEWHEEL,    self.onMouseWheel)
    self.Bind(wx.EVT_SIZE,          self.onSize)
    self.Bind(wx.EVT_KEY_DOWN,      self.onKeyDown)
    self.Bind(wx.EVT_KEY_UP,        self.onKeyUp)
    self.Bind(wx.EVT_TIMER,         self.onTimer)
    self.Bind(wx.EVT_CONTEXT_MENU,  self.onContextMenu)

    self.timer_interval=100
    self.timer=wx.Timer(self)
    self.timer.Start(self.timer_interval)

    self.statusbar=self.parent.CreateStatusBar(5)
    self.statusbar.SetStatusWidths([50, 50, 100, -1, -1])
    self.dc_sizeXY=self.GetSize() # das ist schon die Größe abzüglich der Statusbar

    self.leftDown=False
    self.osm=OSMscr2(self.dc_sizeXY, SOURCE[SERVER], centerLL, self.zoom, self.border)
    self.pt1LL=None
    self.esc=False
    self.file_loaded=False
    self.refresh_needed=False
    self.was_moved=False
    self.superspeed=False
    self.initial_follow_pointLL=None
    self.autofollow_running=False
    self.has_image=False
    self.handle_for_ap_list=None
    self.ap_seen_at_list=None

    self.statusbar.SetStatusText("%d"%(self.zoom,), 0)

    self.filename_dt=FileDrop(self)
    self.SetDropTarget(self.filename_dt)

    if USE_DB:
      self.db=Database(DATABASE_FILE)
    else:
      self.point_list=[ ( (54.791682553658454,  9.442400336265564), 5,  wx.Brush("RED"),    (99, " Hackspace ")),
                        ( (54.80507860419667,   9.52488899230957),  7,  wx.Brush("GOLD"),   (99, " Zuhause\n ist es am schönsten ")),
                        ( (54.80737577256639,   9.513548612594604), 5,  wx.Brush("RED"),    (99, " lecker\n Bier-Versorger ")),
                      ]
      self.osm.loadPoints(self.point_list)
    self.pointMoveOver=list()

    wx.FutureCall(10, self.SetFocus)  # damit EVT_KEY_xx kommt


  # ###########################################################
  # Wird aus FileDrop aufgerufen, wenn ein DateiObjekt auf dem
  # Fenster "fallengelassen" wurde.
  def fileDropped(self, filename):
    self.lstsLL=self.loadGPX(filename)
    # Format von self.lstsLL:
    #   [ [((lat, lon), time), ((lat, lon), time), ...],   - erster Track + Timestamps
    #     [((lat, lon), time), ((lat, lon), time), ...],   - zweiter Track + Timestamps
    #      ... ]
    t0LL=list()             # time aus den Listen entfernen
    for trk in self.lstsLL:
      t1LL=list()
      for pLL, dummy in trk:
        t1LL.append(pLL)
      t0LL.append(t1LL)
    self.zoom, centerLL=self.osm.loadTracks(t0LL)
    self.osm.setZoom(self.zoom, (0, 0))
    self.fixTrackPenAfterZoom()
    self.osm.setCenter(centerLL)
    self.initial_follow_pointLL=t0LL[0][0]   # (lat, lon)
    self.statusbar.SetStatusText("%d"%(self.zoom,), 0)
    if len(self.lstsLL)>0:
      self.pt1LL=self.lstsLL[0][0]  # Punkte zur Geschwindigkeitsbestimmung initialisieren
      self.pt0LL=self.pt1LL         # Format ist: ((lat, lon), UNIX-Timestamp)
    self.Refresh()


  # ######################################################################
  # GPX-Datei laden, parsen und als Liste von Tracks, bestehend aus Listen
  # von Tupeln ((lat, lon), Timestamp), zurückgeben.
  def loadGPX(self, filename):
    with open(filename, "r") as fl:
      filedata=fl.read()
    root=etree.fromstring(filedata)
    ns=""
    if None in root.nsmap:
      ns="{"+root.nsmap[None]+"}"
    tracks=list()
    if etree.iselement(root) and root.tag==ns+"gpx":
      for trk in root:
        track=list()
        if etree.iselement(trk) and trk.tag==ns+"trk":
          for trkseg in trk:
            if etree.iselement(trkseg) and trkseg.tag==ns+"trkseg":
              for trkpt in trkseg:
                if etree.iselement(trkpt) and trkpt.tag==ns+"trkpt":
                  t=""
                  for e in trkpt:
                    if etree.iselement(e) and e.tag==ns+"time":
                      t=getTimestampStringAsUTC(e.text)
                      break
                  track.append(((float(trkpt.get("lat")), float(trkpt.get("lon"))), t))
        if len(track)>0:
          tracks.append(track)
    if len(tracks)>0:
      self.file_loaded=True
    return(tracks)


  # ######################################################################
  # Fensterinhalt darstellen.
  def onPaint(self, evt):
    #wx.BeginBusyCursor()
    self.refresh_needed=self.osm.drawMap()
    self.dc=wx.PaintDC(self)
    imgbuf, (sx, sy)=self.osm.getBitmap()
    self.dc.DrawBitmap(imgbuf, sx, sy)

    if self.ap_seen_at_list is not None:
      self.drawLinesForAP(self.ap_seen_at_list)

    if len(self.pointMoveOver)>0:
      balloon_lst=list()
      for (ptype, data), pos in self.pointMoveOver:
        if ptype==0:    # Punkt-Typ == AP-Koordinate
          txt=" %s \n %s \n channel: %d \n %s \n points: %d \n %s \n %s "%data[:7]
        elif ptype==1:  # Punkt-Typ == AP-Sichtungs-Koordinate
          txt=" dbm:%d \n %s \n Entfernung: %dm "%data
        elif ptype==99: # Testdaten
          txt=data
        else:
          txt=" ??? "
        balloon_lst.append((pos, txt))
        #self.showBalloonText(pos, txt)
        #break # eins langt
      if len(balloon_lst)>0:
        self.showBalloonTextList(balloon_lst)

    if self.initial_follow_pointLL is not None:
      self.drawFollowPoint(self.osm.getPixelForLatLon(self.initial_follow_pointLL))

    if self.autofollow_running:                         # wenn auto-follow läuft
      secs=self.pt1LL[1]-self.pt0LL[1]
      self.drawFollowPoint((self.dc_sizeXY[0]/2, self.dc_sizeXY[1]/2))
      if secs>0:
        txt="%6.2f km/h "%(self.haversine(self.pt0LL[0], self.pt1LL[0])/secs*3600,)
      else:
        txt=" ??.?? km/h "
      self.showBalloonText((self.dc_sizeXY[0]/2, self.dc_sizeXY[1]/2), txt)

    self.statusbar.SetStatusText(str(self.osm.getOpenRequests()), 1)
    #wx.EndBusyCursor()
    #wx.EndBusyCursor()  # doppelt hält besser - sonst bleibt manchmal der BusyCursor dauerhaft an


  # ######################################################################
  # Stellt für die Track-Anzeige den Startpunkt und bei Auto-Follow die
  # aktuelle Positionsmarkierung dar.
  def drawFollowPoint(self, posXY):
    self.dc.SetPen(wx.Pen("BLACK"))
    self.dc.SetBrush(wx.Brush("WHITE"))
    self.dc.DrawCirclePoint(posXY, 5)


  # ######################################################################
  # Zeichnet Linien von einer AP-Position zu allen seinen
  # Sichtungs-Punkten.
  def drawLinesForAP(self, data):
    base, lst=data
    self.dc.SetPen(wx.Pen("BLACK"))
    for pLL, xx, xx, (xx, (dbm, ts, dist)) in lst:
      x, y=self.osm.getPixelForLatLon(pLL)
      self.dc.DrawLine(base[0], base[1], x, y)


  # ######################################################################
  # Linke Maustaste geklickt - Map scrollen.
  def onLeftDown(self, evt):
    self.leftDown=True
    self.was_moved=False
    self.leftDownPosXY=evt.GetPosition()
    self.SetCursor(wx.StockCursor(wx.CURSOR_HAND))
    self.statusbar.SetStatusText("%dx%d"%(self.leftDownPosXY[0], self.leftDownPosXY[1]), 2)
    posLL=self.osm.getLatLonForPixel(self.leftDownPosXY)
    self.statusbar.SetStatusText("%14.12f, %14.12f"%(posLL[0], posLL[1]), 3)
    print "posLL", posLL

    self.pointMoveOver=self.osm.getDataForPoint(self.leftDownPosXY)

    if self.handle_for_ap_list is not None: # wenn eine alte Sichtungs-Liste entfern werden muss
      self.osm.unloadPoints(self.handle_for_ap_list)  # weg damit
      self.ap_seen_at_list=None
      self.handle_for_ap_list=None
      if len(self.pointMoveOver)==0:  # wenn nicht eh gleich ein Refresh erfolgt
        self.osm.forceRefresh()
        self.Refresh()                # hier anfordern

    if len(self.pointMoveOver)>0:
      for (ptype, data), pos in self.pointMoveOver:
        if ptype==0:  # wenn AP-Koordinate
          # data=(bssid, name, channel, crypto, calc_points, first_seen, last_seen, lat, lon)
          bssid=data[0]
          locs=self.db.getFullGPSpoints(bssid)  # locs=[(lat, lon, dbm), ...]
          tmp_lst=list()
          for lat, lon, dbm, ts in locs:
            dist=int(round(self.haversine((data[7], data[8]), (float(lat), float(lon)))*1000))
            tmp_lst.append(((float(lat), float(lon)), 3, wx.Brush("BLUE"), (1, (int(dbm), ts, dist))))
          self.ap_seen_at_list=(pos, tmp_lst)
          self.handle_for_ap_list=self.osm.loadPoints(tmp_lst)
          self.osm.forceRefresh()
          self.Refresh()
          break # einer langt - sonst wirds unübersichtlich


  # ######################################################################
  # Linke Maustaste losgelassen - Map scrollen beenden.
  def onLeftUp(self, evt):
    self.leftDown=False
    self.SetCursor(wx.StockCursor(wx.CURSOR_ARROW))
    if self.was_moved:
      self.osm.endMoveMap()
      self.Refresh()
      # der Refresh() wird hier gebraucht, um nach einem Verschiebevorgang
      # außerhalb der vorgeladenen Rand-Tiles die fehlenden Tiles nachzuladen.


  # ######################################################################
  # Mauscursor wird über das Fenster bewegt.
  def onMotion(self, evt):
    self.cur_mouse_posXY=evt.GetPosition()
    if self.leftDown:
      self.was_moved=True
      scroll_distXY=self.leftDownPosXY-evt.GetPosition()
      self.osm.doMoveMap(scroll_distXY)
      self.Refresh()
    else:
      old_len=len(self.pointMoveOver)
      self.pointMoveOver=self.osm.getDataForPoint(self.cur_mouse_posXY)
      if len(self.pointMoveOver)>0: # wenn Daten verfügbar sind
        self.Refresh()              # Sprechblase darstellen
      elif old_len>0:               # wenn vorher Daten verfügbar waren
        self.Refresh()              # die Sprechblase wieder löschen


  # ######################################################################
  # Gibt eine Sprechblase mit "text" bei der Position "(x, y)" aus.
  def showBalloonText(self, (x, y), text):
    self.dc.SetPen(wx.Pen("BLACK"))
    self.dc.SetBrush(wx.Brush("WHITE"))
    w, h=self.dc.GetTextExtent(text)
    d=5 # Abstand
    xo, yo=(x+d, y-d-h)
    if yo<0: yo=y+d
    if xo+w>self.dc_sizeXY[0]:  xo=x-d-w
    self.dc.DrawRectangle(xo, yo, w, h)
    self.dc.DrawText(text, xo, yo)


  # ######################################################################
  # Wie showBalloonText() - nur für mehrere nebeneinander.
  def showBalloonTextList(self, lst):
    maxcnt=5
    wsum=0    # Gesamtbreite
    xlst=[0]  # Startpositionen
    wlst=[]   # Breiten
    maxh=0    # Höhe
    for pos, text in lst:
      w, h=self.dc.GetTextExtent(text)
      wlst.append(w)
      wsum+=w
      xlst.append(wsum)
      maxh=max(maxh, h)
    self.dc.SetPen(wx.Pen("BLACK"))
    self.dc.SetBrush(wx.Brush("WHITE"))
    x, y=lst[0][0]
    d=5 # Abstand
    xo, yo=(x+d, y-d-maxh)
    if yo<0: yo=y+d
    if xo+wsum>self.dc_sizeXY[0]:  xo=x-d-wsum
    if xo<0:  xo=0
    idx=0
    for pos, text in lst:
      self.dc.DrawRectangle(xo+xlst[idx], yo, wlst[idx], maxh)
      self.dc.DrawText(text, xo+xlst[idx], yo)
      idx+=1

  # ######################################################################
  # Scrollrad betätigt -> Map ändern.
  def onMouseWheel(self, evt):
    zoom_old=self.zoom
    if evt.GetWheelRotation()>0:
      self.zoom=min(SOURCE[SERVER]["max_zoom"], self.zoom+1)
    else:
      self.zoom=max(0, self.zoom-1)
    if self.zoom!=zoom_old:
      self.ap_seen_at_list=None # erstmal quick&dirty
      self.osm.setZoom(self.zoom, evt.GetPosition())
      self.statusbar.SetStatusText("%d"%(self.zoom,), 0)
      self.Refresh()
    self.fixTrackPenAfterZoom()


  # ######################################################################
  # Setzt den wx.Pen zum Track-Zeichnen je nach Zoom-Level.
  def fixTrackPenAfterZoom(self):
    if self.zoom<14:  self.osm.setTrackPen(wx.Pen("MEDIUM VIOLET RED", width=4))
    else:             self.osm.setTrackPen(wx.Pen("MEDIUM VIOLET RED", width=2))


  # ######################################################################
  # Fenstergröße wurde geändert - Variablen anpassen.
  def onSize(self, evt):
    self.dc_sizeXY=self.GetSize()
    try:
      self.osm.setSize(self.dc_sizeXY, self.zoom, self.border)
    except:
      pass  # der allererste EVT_SIZE kommt, bevor self.osm existiert


  # ######################################################################
  # Eine Taste wurde betätigt - ggf. auto-follow starten.
  def onKeyDown(self, evt):
    if evt.GetKeyCode()==wx.WXK_CONTROL:    self.ctrlIsDown=True
    if evt.GetKeyCode()==wx.WXK_ESCAPE:     self.esc=True
    if evt.GetKeyCode()==ord("A"):          self.autoFollow()
    evt.Skip(True)


  # ######################################################################
  # Eine Taste wurde losgelassen.
  def onKeyUp(self, evt):
    if evt.GetKeyCode()==wx.WXK_CONTROL:  self.ctrlIsDown=False
    if evt.GetKeyCode()==wx.WXK_ESCAPE:   self.esc=False
    if evt.GetKeyCode()==ord("S"):        self.superspeed=not self.superspeed
    if evt.GetKeyCode()==ord("Q"):
      self.osm.setCenter((54.805109, 9.524913))
      self.Refresh()
      print "set"
    evt.Skip(True)


  # ###########################################################
  # Ruft einmal pro Update-Intervall die Prüfung wegen Refresh
  # auf, um damit ggf. mittlerweile verfügbare Tiles anzuzeigen.
  def onTimer(self, evt):
    if self.refresh_needed:
      self.Refresh()


  # ###########################################################
  # Stellt das Kontext-Menue dar.
  def onContextMenu(self, event):
    self.PopupMenu(self.menue)
    self.Refresh()


  # ###########################################################
  # Legt das Kontext-Menue an.
  def createMenu(self):
    self.menue=wx.Menu()
    self.menue.Append(500, 'Wardriving-Modus', "", True)

    self.Bind(wx.EVT_MENU, self.wardrivingMode, id=500)
    self.menue.Check(500, self.wardriving_mode)


  # ###########################################################
  # Schaltet die Anzeige der Wardriving-AP-Koordinaten um bzw.
  # an oder aus.
  def wardrivingMode(self, evt):
    self.wardriving_mode=not self.wardriving_mode
    if self.wardriving_mode:
      print "wardrivingMode"
      aps=self.db.getAllAPs()
      self.point_list=list()
      for ap in aps:
        #      0    1     2     3       4       5       6             7           8
        # ap=(lat, lon, bssid, name, channel, crypto, calc_points, first_seen, last_seen)
        if ap[3] not in ["Vodafone Hotspot", "Vodafone Homespot", "Telekom_FON"]:
          data=(ap[2], ap[3], int(ap[4]), ap[5], int(ap[6]), ap[7], ap[8], float(ap[0]), float(ap[1]))
          self.point_list.append(((float(ap[0]), float(ap[1])), 3, wx.Brush("RED"), (0, data)))
      self.wardriving_point_handle=self.osm.loadPoints(self.point_list)
    else:
      self.osm.unloadPoints(self.wardriving_point_handle)
      self.wardriving_point_handle=None
      if self.handle_for_ap_list is not None: # wenn eine alte Sichtungs-Liste entfern werden muss
        self.osm.unloadPoints(self.handle_for_ap_list)  # weg damit
        self.ap_seen_at_list=None
        self.handle_for_ap_list=None

    self.osm.forceRefresh()
    self.Refresh()


  # ######################################################################
  # Scrollt von der Geo-Koordinate "posFromLL" nach "posToLL" mit einer
  # Schrittgröße von "step_size" Pixeln und "delay" Sekunden Wartezeit
  # zwischen den Schritten.
  # Der Abstand der Koodinaten sollte in "self.border" passen, weil
  # zwischendurch keine Tiles nachgeladen werden.
  def scrollMapToPoint(self, posFromLL, posToLL, step_size, delay):
    p1XY=self.osm.getPixelForLatLon(posFromLL)
    p2XY=self.osm.getPixelForLatLon(posToLL)
    wXY=self.get_line(p1XY, p2XY)
    pcXY=wXY[0]
    for idx in range(1, len(wXY)-(step_size-1), step_size):
      d=(wXY[idx][0]-pcXY[0], wXY[idx][1]-pcXY[1])
      self.osm.doMoveMap(d)
      self.Refresh()
      wx.Yield()
      if self.esc:
        break
      time.sleep(delay)
    self.osm.endMoveMap()
    self.osm.setCenter(posToLL)
    self.Refresh()
    wx.Yield()


  # ######################################################################
  # Fährt die geladenen Tracks ab.
  def autoFollow(self):
    if not self.file_loaded:
      return
    print "autoGPXfollow"
    self.autofollow_running=True
    delay=0.1
    step_size=4
    self.timer.Stop()
    self.initial_follow_pointLL=None
    for t in range(len(self.lstsLL)):  # über alle Tracks
      curLL=self.lstsLL[t][0]
      self.osm.setCenter(curLL[0])
      self.Refresh()
      for pLL, tl in self.lstsLL[t]:      # über alle (Punkte, Zeiten) des Tracks
        wx.Yield()
        if self.esc:
          break
        if pLL!=curLL[0]:                 # wenn Bewegung stattfand
          self.pt1LL=(pLL, tl)   # Koordinaten+Zeit des aktuellen Punktes für onPaint()
          if self.superspeed:
            self.osm.setCenter(pLL)
            self.Refresh()
            wx.Yield()
          else:
            self.scrollMapToPoint(curLL[0], pLL, step_size, delay)
          curLL=(pLL, tl)
          self.pt0LL=curLL
          self.statusbar.SetStatusText("%14.12f, %14.12f"%(pLL[0], pLL[1]), 3)
        else:
          time.sleep(delay)
        self.statusbar.SetStatusText(str(datetime.fromtimestamp(tl)), 4)
        while self.osm.getOpenRequests()>20:
          time.sleep(0.2)
          self.Refresh()
          wx.Yield()
        if self.esc:
          break
      self.pt1LL=None
      if self.esc:
        break
      self.Refresh()
    self.timer.Start(self.timer_interval)
    self.autofollow_running=False


  # ###########################################################
  # Get all points of a straight line.
  # Quelle: http://stackoverflow.com/a/25913345/3588613
  def get_line(self, (x1, y1), (x2, y2)):
    points = []
    issteep = abs(y2-y1) > abs(x2-x1)
    if issteep:
        x1, y1 = y1, x1
        x2, y2 = y2, x2
    rev = False
    if x1 > x2:
        x1, x2 = x2, x1
        y1, y2 = y2, y1
        rev = True
    deltax = x2 - x1
    deltay = abs(y2-y1)
    error = int(deltax / 2)
    y = y1
    ystep = None
    if y1 < y2:
        ystep = 1
    else:
        ystep = -1
    for x in range(x1, x2 + 1):
        if issteep:
            points.append((y, x))
        else:
            points.append((x, y))
        error -= deltay
        if error < 0:
            y += ystep
            error += deltax
    # Reverse the list if the coordinates were reversed
    if rev:
        points.reverse()
    return points


  # ######################################################################
  # Quelle: https://pypi.python.org/pypi/haversine
  # AVG_EARTH_RADIUS = 6371  - in km
  def haversine(self, point1, point2, miles=False):
    # unpack latitude/longitude
    lat1, lng1 = point1
    lat2, lng2 = point2

    # convert all latitudes/longitudes from decimal degrees to radians
    lat1, lng1, lat2, lng2 = map(math.radians, (lat1, lng1, lat2, lng2))

    # calculate haversine
    lat = lat2 - lat1
    lng = lng2 - lng1
    d = math.sin(lat * 0.5) ** 2 + math.cos(lat1) * math.cos(lat2) * math.sin(lng * 0.5) ** 2
    h = 2 * 6371 * math.asin(math.sqrt(d))
    if miles:
        return h * 0.621371  # in miles
    else:
        return h  # in kilometers



# ###########################################################
# Der Fenster-Rahmen für das Hauptfenster.
class MapFrame(wx.Frame):
  def __init__(self, parent, centerLL, zoom, border, title, pos=wx.DefaultPosition, size=wx.DefaultSize):
    style=wx.DEFAULT_FRAME_STYLE
    wx.Frame.__init__(self, None, wx.ID_ANY, title+" "+VERSION, pos=pos, size=size, style=style)
    win=MapWindow(self, centerLL, zoom, border, title, size)



# ######################################################################
# main()
if __name__=="__main__":
  pt=(54.805060, 9.524878)
  app=wx.App(False)
  frame=MapFrame(None, pt, 16, (2, 2), "OpenStreetMapTest", size=(1280, 768+25))  # 5x3 Tiles + 25 für Statusbar
  frame.Show()
  app.MainLoop()
