#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ###########################################################
# Eine Klasse zur Anzeige einer OpenStreetMap-Karte samt
# Scroll- und Zoom-Steuerung.
#
# Detlev Ahlgrimm, 2017
#
#
# 08.04.2017  erste Version
# 10.04.2017  kompletter Umbau, der jetzt die Position (0, 0) als Referenz-Punkt
#             verwendet (statt vorher den Mittelpunkt der Anzeige).
# 13.04.2017  nochmal deutlicher Umbau, weil ein statischer Faktor für "lat/lon zu Pixel"
#             nicht wirklich funktioniert - zumindest nicht für lat.
# 14.04.2017  nicht existente Tiles (außerhalb der Karte) erkennen und nicht
#             ständig versuchen, sie zu laden (bzw. Unterscheidung zwischen
#             "laden unmöglich" und "laden fehlgeschlagen").
# 15.04.2017  getVisibleRectangle() und getZoomAndCenterForAllPointsVisible() zugefügt
# 16.04.2017  Umstellung auf Nutzung von wx.MemoryDC()
#             Verarbeitung von Track- und Point-Listen
# 17.04.2017  Verarbeitung mehrerer unabhängiger Point-Listen, die über ein Handle
#             referenziert werden, Tracks werden Ausschnitts-bedingt an DrawLines() übergeben
#             nur solche Punkte an DrawCirclePoint() übergeben, die auf der Bitmap liegen
#

import math
import wx               # python-wxWidgets-2.8.12.1-10.4.1.x86_64

from OSM_TileCache import TileCache


# ######################################################################
# Die Referenz ist immer der Punkt (0, 0) im Fenster. Also links oben.
#
# lat - Y-Achse, bei uns gehen höhere Werte nach Norden
# lon - X-Achse, bei uns gehen höhere Werte nach Westen
# östlich von Greenwich wird lon negativ
# südlich vom Äquator wird lat negativ
#
# Notation für Variablennamen:
#   variableXY  - Koordinaten in Pixel auf dem DC
#   variableTN  - TileNumber
#   variableLL  - Koordinaten in Lat/Lon
class OSMscr2():
  def __init__(self, dc_sizeXY, srv_data, centerLL, zoom, border=(2, 2)):
    self.ts=srv_data["tile_size"]
    self.tc=TileCache(srv_data["cache_dir"], srv_data["tile_url"])
    self.max_zoom=srv_data["max_zoom"]
    self.status=self.tc.getStatusDict()
    self.setSize(dc_sizeXY, zoom, border)
    self.shiftXY=self.shift_tempXY=self.shift_statXY=(0, 0)
    self.setCenter(centerLL)

    self.old_params=None
    self.needs_refresh=True

    self.track_listLL=None        # die Trackliste, wie sie an loadTracks() übergeben wurde
    self.track_listXY=None        # die nach XY gewandelte Trackliste - Neuladung bei jeder Änderung von self.zoom
    self.track_listXY_zoom=-1     # Wert von self.zoom, für den self.track_listXY aktuell gültig ist
    self.track_listXY_minmax=None # Eckunkte der Tracks - zum Ausblenden von Tracks außerhalb des Anzeige-Bereiches
    self.track_pen=wx.Pen("MEDIUM VIOLET RED", width=2) # der Pen, mit dem die Tracks gezeichnet werden

    self.all_point_listsLL=dict() # der Speicherplatz für alle Punkt-Listen. Format: {handle:point_list}
    self.all_point_listXY=dict()  # die nach XY gewandelte Punkt-liste - Neuladung bei jeder Änderung von self.zoom
    self.all_point_listXY_zoom=-1 # Wert von self.zoom, für den self.all_point_listXY aktuell gültig ist
    self.current_pointsXY=list()  # wird in __drawPointsDC() auf die Offset-Losen XY-Koordinaten gemäß self.all_point_listXY gesetzt


  # ######################################################################
  # Richtet die Karte so aus, dass der Längen/Breitengrad "centerLL" in
  # der Mitte des Anzeigebereiches zu liegen kommt.
  # Während eines Verschiebevorganges wird das Zentrum um "self.shiftXY"
  # verschoben ausgerichtet.
  def setCenter(self, centerLL):
    self.__setPoint(centerLL, (self.dc_sizeXY[0]/2, self.dc_sizeXY[1]/2))


  # ######################################################################
  # Richtet die Karte so aus, dass der Längen/Breitengrad "posLL" auf der
  # Pixel-Position "posXY" zu liegen kommt.
  def __setPoint(self, posLL, posXY):
    cpdTN=self.deg2numF(posLL, self.zoom)                                         # Tile-Nummer unter der Ziel-Koordinate (float)
    cpbTN=(int(cpdTN[0]), int(cpdTN[1]))                                          # Tile-Nummer unter der Ziel-Koordinate (int)
    cprXY=((cpdTN[0]-int(cpdTN[0]))*self.ts, (cpdTN[1]-int(cpdTN[1]))*self.ts)    # Pixel-Versatz zu cpdTN
    tcbpXY=(posXY[0]-cprXY[0], posXY[1]-cprXY[1])                                 # Basis-Position von Tile unter der Ziel-Koordinate
    tdist=(int(math.ceil(tcbpXY[0]/self.ts)), int(math.ceil(tcbpXY[1]/self.ts)))  # Anzahl Tiles bis zum Tile auf (0, 0)
    tn00TN=(cpbTN[0]-tdist[0], cpbTN[1]-tdist[1])                                 # Tile-Nummer des Tiles auf (0, 0)
    tn0dXY=(tcbpXY[0]-tdist[0]*self.ts, tcbpXY[1]-tdist[1]*self.ts)               # Basis-Position von Tile auf (0, 0)
    t0d=(abs(tn0dXY[0])/self.ts, abs(tn0dXY[1])/self.ts)                          # Abstand von Basis-Position zu (0, 0) in Tile-Bruchteilen
    self.pos0LL=self.num2deg((tn00TN[0]+t0d[0], tn00TN[1]+t0d[1]), self.zoom)     # Geo-Koordinaten der (0, 0)-Position vom Tile auf (0, 0)
    self.needs_refresh=True


  # ######################################################################
  # Passt alle entsprechenden Variablen einer [neuen] Fenstergröße an.
  def setSize(self, dc_sizeXY, zoom, border=(2, 2)):
    self.dc_sizeXY=dc_sizeXY  # Breite, Höhe des Darstellungs-Bereiches
    self.zoom=zoom            # der Zoom-Level von OSM
    self.border=border        # pro Rand zusätzlich zu ladende Anzahl von Tiles (horizontal, vertikal)

    # Anzahl der sichtbaren Tiles (x, y) je nach Anzeigegröße
    self.tile_cnt_dc=(float(dc_sizeXY[0])/self.ts, float(dc_sizeXY[1])/self.ts)
    self.tile_cnt_dc_max=(int(math.ceil(self.tile_cnt_dc[0])), int(math.ceil(self.tile_cnt_dc[1])))


  # ######################################################################
  # Ändert die Zoom-Stufe und sorgt dafür, dass die Koordinate wieder an
  # der Position des Mauszeigers landet, die vor dem Zoom dort war.
  def setZoom(self, zoom, posXY):
    posLL=self.getLatLonForPixel(posXY) # Position bei altem Zoom-Level merken
    self.zoom=zoom                      # Zoom-Level ändern
    self.__setPoint(posLL, posXY)       # gemerkte Position wieder einstellen
    self.tc.flushQueue()  # alte, ggf. noch anstehende Tile-Requests können jetzt weg


  # ######################################################################
  # Bewegt das an Position (0, 0) angezeigte Pixel (samt der restlichen
  # Karte) um "distXY" Pixel.
  def doMoveMap(self, distXY):
    self.shift_tempXY=distXY
    self.shiftXY=self.shift_tempXY+self.shift_statXY


  # ######################################################################
  # Stellt das derzeit an Position (0, 0) angezeigte Pixel als neue
  # Basis ein.
  def endMoveMap(self):
    self.pos0LL=self.getLatLonForPixel(self.shiftXY)
    self.shift_statXY=self.shiftXY=(0, 0)
    self.needs_refresh=True


  # ######################################################################
  # Liefert Längen/Breitengrad unter dem Pixel "posXY".
  # Wird die Karte gerade per doMoveMap() verschoben, muss der aktuelle
  # Versatz gemäß getOffset() auf "posXY" addiert werden.
  def getLatLonForPixel(self, posXY):
    tile0TN=self.deg2num(self.pos0LL, self.zoom)                # TileNummer vom Tile auf (0, 0)
    tile0iLL=self.num2deg(tile0TN, self.zoom)                   # lat/lon des Eckpunktes vom Tile auf (0, 0)
    tile0d0XY=self.distanceLatLonToPixel(self.pos0LL, tile0iLL) # Abstand Eckpunkt zu (0, 0)
    distXY=(posXY[0]-tile0d0XY[0], posXY[1]-tile0d0XY[1])       # Abstand Eckunkt vom Tile auf (0, 0) zu posXY
    tdist=(float(distXY[0])/self.ts, float(distXY[1])/self.ts)  # Abstand von posXY zu (0, 0) in Tiles
    tdp=(tile0TN[0]+tdist[0], tile0TN[1]+tdist[1])              # TileNummer mit Nachkommastellen unter "posXY"
    tpLL=self.num2deg(tdp, self.zoom)                           # lat/lon unter "posXY"
    return(tpLL)


  # ######################################################################
  # Liefert für zwei Längen/Breitengrade deren Abstand in Pixel.
  def distanceLatLonToPixel(self, pos1LL, pos2LL):
    t1TN=self.deg2numF(pos1LL, self.zoom)
    t2TN=self.deg2numF(pos2LL, self.zoom)
    tdist=(t2TN[0]-t1TN[0], t2TN[1]-t1TN[1])
    distXY=(int(round(tdist[0]*self.ts)), int(round(tdist[1]*self.ts)))
    return(distXY)


  # ######################################################################
  # Liefert zu einem Längen/Breitengrad "posLL" die Pixel-Koordinaten.
  # Die Rückgabe ist auch während eines Verschiebevorganges gültig.
  def getPixelForLatLon(self, posLL):
    dx, dy=self.distanceLatLonToPixel(self.pos0LL, posLL)
    return(int(dx)-self.shiftXY[0], int(dy)-self.shiftXY[1])


  # ######################################################################
  # Liefert die Eckpunkte des aktuell sichtbaren Bereiches in lat/lon.
  # Die Rückgabe ist auch während eines Verschiebevorganges gültig.
  def getVisibleRectangle(self):
    p1=self.getLatLonForPixel(self.shiftXY)
    p2=self.getLatLonForPixel((self.shiftXY[0]+self.dc_sizeXY[0], self.shiftXY[1]+self.dc_sizeXY[1]))
    return(p1, p2)


  # ######################################################################
  # Liefert die Größe der Bitmap entsprechend Fenstergröße und Rand.
  def __getBitmapSize(self):
    return((self.tile_cnt_dc_max[0]+2*self.border[0])*self.ts, 
           (self.tile_cnt_dc_max[1]+2*self.border[1])*self.ts)


  # ######################################################################
  # Erstellt einen neuen MemoryDC.
  def __newBitmap(self):
    w, h=self.__getBitmapSize()
    self.bmpbuf=wx.EmptyBitmapRGBA(w, h, 204, 204, 204, 1)
    self.mdc=wx.MemoryDC()
    self.mdc.SelectObject(self.bmpbuf)


  # ######################################################################
  # Stellt die Map dar.
  # Liefert True, wenn mindestens ein Tile ein Leer-Tile ist.
  def drawMap(self):
    # nur wenn sich Karten-Parameter geändert haben oder noch Tiles fehlen
    if self.old_params!=(self.dc_sizeXY, self.zoom, self.border) or self.needs_refresh:
      self.__newBitmap()
      self.needs_refresh=self.__drawMapDC(self.mdc)
      self.__drawTracksDC(self.mdc)
      self.__drawPointsDC(self.mdc)
      self.old_params=(self.dc_sizeXY, self.zoom, self.border)
    return(self.needs_refresh)


  # ######################################################################
  # Erzwingt einen Neuaufbau des Bildes beim nächsten drawMap().
  def forceRefresh(self):
    self.needs_refresh=True


  # ######################################################################
  # Liefert den aktuellen Karten-Ausschnitt als Bitmap und den
  # Ausgabe-Versatz als Tupel.
  def getBitmap(self):
    return(self.bmpbuf, (-(self.shiftXY[0]+self.border_fixXY[0]), -(self.shiftXY[1]+self.border_fixXY[1])))


  # ######################################################################
  # Zeichnet die Karte in den "dc" - also ins Bitmap.
  def __drawMapDC(self, dc):
    imgs, self.border_fixXY=self.__getImages()
    y=0
    needs_refresh=False  # erstmal optimistisch sein
    for yi in imgs:
      x=0
      for tn, (status, xi) in yi:
        if status!=self.status["GOOD"]:
          xi=wx.EmptyBitmapRGBA(256, 256, red=255, alpha=1)
          if status!=self.status["INVALID"]:
            needs_refresh=True
        dc.DrawBitmap(xi, x, y)
        #dc.DrawLine(x, y, x+self.ts, y)
        #dc.DrawLine(x, y, x, y+self.ts)
        x+=self.ts
      y+=self.ts
    return(needs_refresh)


  # ######################################################################
  # Zeichnet Tracks in den "dc", sofern mindestens Teile von ihnen aktuell
  # sichtbar sind.
  def __drawTracksDC(self, dc):
    if self.track_listLL is not None:           # wenn überhaupt Tracks geladen sind
      self.__updateTrack_listXY()               # "self.track_listXY" [ggf. neu] setzen
      dc.SetPen(self.track_pen)
      for ti in range(len(self.track_listXY)):  # Index über alle Tracks
        trackXY=self.track_listXY[ti]           # Namens-Abkürzung
        delta=self.getPixelForLatLon(self.track_listLL[ti][0])  # Basis-Geo-Koordinate holen
        ox, oy=delta[0]-trackXY[0][0], delta[1]-trackXY[0][1]   # Offset berechnen
        (minx, miny, maxx, maxy)=self.track_listXY_minmax[ti]   # Eckunkte davon holen
        if ox+maxx<0 or ox+minx>self.dc_sizeXY[0] or oy+maxy<0 or oy+miny>self.dc_sizeXY[1]:
          pass  # Track liegt komplett außerhalb des Anzeige-Bereiches
        else:
          dc.DrawLines(trackXY, ox+self.border_fixXY[0], oy+self.border_fixXY[1]) # Track mit Offset anzeigen


  # ######################################################################
  # Erstellt bei Änderung des Zoom-Levels ggf. eine neue
  # "self.track_listXY" aus "self.track_listLL".
  # Parallel wird "self.track_listXY_minmax" geschrieben.
  def __updateTrack_listXY(self):
    if self.track_listLL is not None:       # wenn überhaupt eine Trackliste geladen ist
      if self.zoom!=self.track_listXY_zoom: # und sich der Zoom-Level seit dem letzten Aufruf geändert hat
        self.track_listXY=list()            # alten Inhalt löschen
        self.track_listXY_minmax=list()
        for pLL in self.track_listLL:       # alle lat/lon-Tracks umwandeln in relative Pixel-Listen
          lstXY, minmax=self.__convGPX2Pixel(pLL)
          self.track_listXY.append(lstXY)           # Liste von Punkten zufügen
          self.track_listXY_minmax.append(minmax)   # pro Punkte-Liste ein Tupel mit dessen Eckpunkten ablegen
        self.track_listXY_zoom=self.zoom    # und Zoom-Level merken


  # ######################################################################
  # Zeichnet Points in den "dc".
  def __drawPointsDC(self, dc):
    self.__updatePoint_listXY()
    if len(self.all_point_listXY)>0:
      self.current_pointsXY=list()
      sw, sh=self.__getBitmapSize()
      dc.SetPen(wx.Pen("BLACK"))
      for handle in sorted(self.all_point_listXY):  # nach handle aufsteigend über "self.all_point_listXY" iterieren
        delta=self.getPixelForLatLon(self.all_point_listsLL[handle][0][0])  # Basis-Geo-Koordinate holen
        pt0XY=self.all_point_listXY[handle][0][0]     # erster Punkt dieser Punkt-Liste in Geo-Koordinaten
        ox, oy=delta[0]-pt0XY[0], delta[1]-pt0XY[1]   # Offset berechnen
        for ptXY, radius, brush, data in self.all_point_listXY[handle]:
          x, y=(ptXY[0]+ox, ptXY[1]+oy)
          dc.SetBrush(brush)
          if 0<=x+self.border_fixXY[0]<sw and 0<=y+self.border_fixXY[1]<sh:
            dc.DrawCirclePoint((x+self.border_fixXY[0], y+self.border_fixXY[1]), radius)
            self.current_pointsXY.append(((x, y), radius, brush, data))


  # ######################################################################
  # Erstellt bei Änderung des Zoom-Levels ggf. eine neue
  # "self.all_point_listXY" aus "self.all_point_listsLL".
  def __updatePoint_listXY(self):
    if len(self.all_point_listsLL)>0:           # wenn überhaupt eine Punktliste geladen ist
      if self.zoom!=self.all_point_listXY_zoom: # und sich der Zoom-Level seit dem letzten Aufruf geändert hat
        self.all_point_listXY=dict()            # alten Inhalt löschen
        for handle, vLL in self.all_point_listsLL.items(): # über das gesamte Dictionary
          self.all_point_listXY.update({handle:self.__convGPX2PixelForPoints(vLL)})   # alles geändert umkopieren
        self.all_point_listXY_zoom=self.zoom    # und Zoom-Level merken
      else:   # der Zoom-Level hat sich nicht geändert, aber vielleicht wurden weitere Listen geladen
        for handle, vLL in self.all_point_listsLL.items():
          if handle not in self.all_point_listXY:
            self.all_point_listXY.update({handle:self.__convGPX2PixelForPoints(vLL)}) # alles geändert umkopieren


  # ######################################################################
  # Liefert die Bilder als Liste von Listen zusammen mit ihrer TileNummer.
  # Also als:
  #     [ [ ((xTN, yTN), img), ((xTN, yTN), img), ... ],    # erste Zeile
  #       [ ((xTN, yTN), img), ((xTN, yTN), img), ... ],    # zweite Zeile
  #       ... ]
  # Als zweiter Wert wird ein Tupel mit dem Pixelversatz geliefert, damit
  # "self.pos0LL" genau in der linken oberen Ecke erscheint.
  def __getImages(self):
    tile0TN=self.deg2num(self.pos0LL, self.zoom)  # das an (0, 0) darzustellen Image bestimmen
    yl=list()
    tile0iLL=self.num2deg((tile0TN[0]-self.border[0], tile0TN[1]-self.border[1]), self.zoom) # lat/lon vom unsichtbaren Tile00
    border_fixXY=self.distanceLatLonToPixel(tile0iLL, self.pos0LL)
    for y in range(-self.border[1], self.tile_cnt_dc_max[1]+self.border[1]):
      xl=list()
      for x in range(-self.border[0], self.tile_cnt_dc_max[0]+self.border[0]):
        xl.append(((tile0TN[0]+x, tile0TN[1]+y), self.tc.getTile(tile0TN[0]+x, tile0TN[1]+y, self.zoom)))
      yl.append(xl)
    return(yl, border_fixXY)


  # ######################################################################
  # Liefert die Anzahl der noch unbeantworteten Tile-Requests.
  def getOpenRequests(self):
    return(self.tc.getQueueLength())


  # ######################################################################
  # Liefert ein Tupel aus dem höchsten Zoom-Level, bei dem alle
  # Koordinaten aus "point_listLL" auf den Anzeigebereich passen und den
  # Geo-Koordinaten des Mittelpunktes.
  def getZoomAndCenterForAllPointsVisible(self, point_listLL):
    neLL=swLL=point_listLL[0]
    for pLL in point_listLL:  # Eckpunkte suchen
      if pLL[0]>neLL[0]:  neLL=(pLL[0], neLL[1])  # lat(n)
      if pLL[0]<swLL[0]:  swLL=(pLL[0], swLL[1])  # lat(s)
      if pLL[1]<neLL[1]:  neLL=(neLL[0], pLL[1])  # lon(e)
      if pLL[1]>swLL[1]:  swLL=(swLL[0], pLL[1])  # lon(w)
    for zoom in range(self.max_zoom, -1, -1):
      pneTN=self.deg2numF(neLL, zoom) # Tile(nord osten) = links oben
      pswTN=self.deg2numF(swLL, zoom) # Tile(süd westen) = rechts unten
      dx=abs(pswTN[0]-pneTN[0])       # Tile-Delta x
      dy=abs(pswTN[1]-pneTN[1])       # Tile-Delta y
      if dx<self.tile_cnt_dc[0] and dy<self.tile_cnt_dc[1]:
        return(zoom, (neLL[0]-(neLL[0]-swLL[0])/2, swLL[1]-(swLL[1]-neLL[1])/2))


  # ######################################################################
  # Lon./lat. to tile numbers
  # Quelle: http://wiki.openstreetmap.org/wiki/Slippy_map_tilenames#Python
  def deg2num(self, latlon, zoom):
    x, y=self.deg2numF(latlon, zoom)
    return(int(x), int(y))
  def deg2numF(self, (lat_deg, lon_deg), zoom):
    try:
      lat_rad = math.radians(lat_deg)
      n = 2.0 ** zoom
      xtile = (lon_deg + 180.0) / 360.0 * n
      ytile = (1.0 - math.log(math.tan(lat_rad) + (1 / math.cos(lat_rad))) / math.pi) / 2.0 * n
    except:
      print "error deg2num()"
      return(0, 0)
    return(xtile, ytile)


  # ######################################################################
  # Tile numbers to lon./lat.
  # Quelle: http://wiki.openstreetmap.org/wiki/Slippy_map_tilenames#Python
  def num2deg(self, (xtile, ytile), zoom):
    n = 2.0 ** zoom
    lon_deg = xtile / n * 360.0 - 180.0
    lat_rad = math.atan(math.sinh(math.pi * (1 - 2 * ytile / n)))
    lat_deg = math.degrees(lat_rad)
    return(lat_deg, lon_deg)


  # ######################################################################
  # Lädt eine Liste mit Listen von Track-Punkten und liefert die Werte
  # von (Zoom-Level, Track-Mittelpunkt), bei denen alle Tracks ins Bild
  # passen.
  # Erwartetes Eingangs-Format:
  #   [[(lat, lon), (lat, lon), ...], [(lat, lon), (lat, lon), ...], ...]
  def loadTracks(self, track_listLL):
    self.track_listLL=track_listLL
    t0LL=list()
    for track in track_listLL:   # alle Tracks in einer Liste ablegen, um Zoom-Level und Mittelpunkt zu bestimmen
      for pLL in track:
        t0LL.append(pLL)
    zoom, centerLL=self.getZoomAndCenterForAllPointsVisible(t0LL)
    return(zoom, centerLL)


  # ######################################################################
  # Entfernt die Tracks wieder.
  def unloadTracks(self):
    self.track_listLL=None


  # ######################################################################
  # Setzt die Farbe, in der Tracks gezeichnet werden.
  def setTrackPen(self, pen):
    self.track_pen=pen


  # ######################################################################
  # Lädt eine Liste von Punkten samt Zusatzdaten.
  # Erwartetes Eingangs-Format:
  #   [((lat, lon), radius,  wx.Brush(), data), ((lat, lon), radius,  wx.Brush(), data), ...]
  # Zurückgegeben wird ein Handle, mit dem diese Punktliste wieder
  # gelöscht werden kann.
  def loadPoints(self, point_listLL):
    if len(self.all_point_listsLL)>0:
      handle=max(self.all_point_listsLL)+1
    else:
      handle=1
    self.all_point_listsLL.update({handle:point_listLL})
    return(handle)


  # ######################################################################
  # Entfernt die Punkt-Liste zum Kenner "handle" wieder.
  def unloadPoints(self, handle):
    if handle in self.all_point_listsLL:
      del self.all_point_listsLL[handle]
    if handle in self.all_point_listXY:
      del self.all_point_listXY[handle]
    self.current_pointsXY=list()


  # ######################################################################
  # Liefert zur Position "posXY" den Inhalt von "data" der darunter
  # liegenden Points samt ihrer aktuellen XY-Position als Liste von Tupeln
  # oder [], wenn kein Punkt darunter liegt.
  def getDataForPoint(self, posXY):
    rl=list()
    for ptXY, radius, brush, data in self.current_pointsXY:
      if ptXY[0]-radius<posXY[0]<ptXY[0]+radius and ptXY[1]-radius<posXY[1]<ptXY[1]+radius:
        # jaja - das ist jetzt kein Radius, sondern ein Rechteck
        rl.append((data, ptXY))
    return(rl)


  # ######################################################################
  # Liefert den GPX-Track "lstLL" nach XY umgewandelt. Weiterhin wird ein
  # Tupel mit dessen Eckpunkten (in Pixeln) geliefert, um damit Tracks
  # ausblenden zu können, die aktuell komplett nicht sichtbar sind.
  def __convGPX2Pixel(self, lstLL):
    if len(lstLL)==0: return([], (0, 0, 0, 0))
    lstXY=list()
    minx, miny=self.getPixelForLatLon(lstLL[0]) # nur für Init der min/max-Variablen
    maxx, maxy=(minx, miny)
    for pLL in lstLL:
      x, y=self.getPixelForLatLon(pLL)
      minx, miny, maxx, maxy=(min(minx, x), min(miny, y), max(maxx, x), max(maxy, y))
      lstXY.append((x, y))
    return(lstXY, (minx, miny, maxx, maxy))


  # ######################################################################
  # Liefert die Point-Liste "lstLL" nach XY umwandelt.
  # Eingangsformat: [((lat, lon), radius,  wx.Brush(), data), ((lat, lon), radius,  wx.Brush(), data), ...]
  # Ausgangsformat: [((x, y), radius,  wx.Brush(), data), ((x, y), radius,  wx.Brush(), data), ...]
  def __convGPX2PixelForPoints(self, lstLL):
    if len(lstLL)==0: return([])
    lstXY=list()
    for pt in lstLL:
      lstXY.append((self.getPixelForLatLon(pt[0]), pt[1], pt[2], pt[3]))
    return(lstXY)
