#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ###########################################################
# gcv.py  v1.1
#
#   Ein Script zum visualisieren einer GCode-Datei.
#
#
# Detlev Ahlgrimm, 2019
#
# 12/13.03.2019 v1.0  erste Version
# 14-17.03.2019 v1.1  GCode-Dateien per D&D laden, Skalierung des Modells bei Änderung der Fenstergröße,
#                     Zoom- und Shift-Funktion zugefügt, Fenster-Daten speichern

import wx
import sys
import os
import time

VERSION="1.1"
CFG_FILE=".gcv.rc"

HEIZBETT=[ ("Anycubic", 210, 210), ("K8200", 200, 170), ("test", 300, 300) ]


# ###########################################################
# Liefert die Parameter (X, Y, Z, E) aus einer G0/G1-Zeile.
def parseLine(ln):
  X=Y=Z=E=None
  for e in ln:
    if e.startswith(";"):
      break
    elif e.startswith("X"):
      X=float(e[1:])
    elif e.startswith("Y"):
      Y=float(e[1:])
    elif e.startswith("Z"):
      Z=float(e[1:])
    elif e.startswith("E"):
      E=float(e[1:])
  return X, Y, Z, E

# ###########################################################
# Lädt die Datei "inputfile" und liefert sie als Liste aus
# Layern, die ihrerseits wieder Listen aus Tupeln
# (X, Y, Z, E) sind.
# Moves mit vorheriger Retraktion werden durch "E=-1"
# gekennzeichnet.
def getFile(inputfile):
  with open(inputfile, "r") as fh:
    filedata=fh.readlines()
  lastE=0
  retract=False
  layers=[]
  cur_layer=[]
  for ln in filedata:
    lns=ln.split()
    if len(lns)>0 and lns[0]=="G28":                                    # home all axis
      if len(cur_layer)>1:  layers.append(cur_layer)
      cur_layer=[]
    if len(lns)>0 and (lns[0]=="G0" or lns[0]=="G1"):
      X, Y, Z, E=parseLine(lns)
      if E is not None:
        if E<lastE:
          retract=True
        lastE=E
      if Z is not None:                                                 # Z-Move
        if len(cur_layer)>1:  layers.append(cur_layer)
        if retract:
          if E is not None:
            retract=False
            cur_layer=[(X, Y, Z, E)]
          else:
            cur_layer=[(X, Y, Z, -1)]
        else:
          cur_layer=[(X, Y, Z, E)]
      elif X is not None and Y is not None:                             # X/Y-Move
        if retract:
          if E is not None:
            retract=False
            cur_layer.append((X, Y, Z, E))
          else:
            cur_layer.append((X, Y, Z, -1))
        else:
          cur_layer.append((X, Y, Z, E))
  if len(cur_layer)>1:  layers.append(cur_layer)
  return layers



# ###########################################################
# Eine kleine Klasse, um neue Dateien via Drag&Drop öffnen
# zu können.
class FileDrop(wx.FileDropTarget):
  def __init__(self, window):
    wx.FileDropTarget.__init__(self)
    self.window=window

  def OnDropFiles(self, x, y, filenames):
    if len(filenames)!=1:
      return False
    self.window.fileDropped(filenames)
    return True



# ###########################################################
# Das Fenster.
class GCVPanel(wx.Panel):
  def __init__(self, parent):
    super(GCVPanel, self).__init__(parent)

    self.parent=parent
    self.hsizer=wx.BoxSizer(wx.HORIZONTAL)
    self.vsizer=wx.BoxSizer(wx.VERTICAL)
    
    self.Bind(wx.EVT_PAINT, self.on_paint)
    self.Bind(wx.EVT_SIZE, self.on_size)
    #self.Bind(wx.EVT_TIMER, self.on_timer)
    #self.timer=wx.Timer(self)
    self.Bind(wx.EVT_MOUSEWHEEL, self.on_mouse_wheel)
    self.Bind(wx.EVT_CONTEXT_MENU, self.on_context_menu)
    self.Bind(wx.EVT_LEFT_DOWN, self.on_left_down)
    self.Bind(wx.EVT_LEFT_UP, self.on_left_up)
    self.Bind(wx.EVT_MOTION, self.on_motion)

    self.scrollbarV=wx.ScrollBar(self, size=(-1, -1), style=wx.SB_VERTICAL)
    self.scrollbarV.Bind(wx.EVT_SCROLL, self.on_vscroll)

    self.scrollbarH=wx.ScrollBar(self, size=(-1, -1), style=wx.SB_HORIZONTAL)
    self.scrollbarH.Bind(wx.EVT_SCROLL, self.on_hscroll)

    self.hsizer.AddStretchSpacer()
    self.hsizer.Add(self.scrollbarV, 0, flag=wx.EXPAND|wx.ALL)
    self.vsizer.Add(self.hsizer, 1, flag=wx.EXPAND| wx.ALL)
    self.vsizer.Add(self.scrollbarH, 0, flag=wx.EXPAND| wx.ALL)
    self.SetSizer(self.vsizer)

    filename_dt=FileDrop(self)                                          # Drag&Drop von GCode-Dateien aktivieren
    self.SetDropTarget(filename_dt)

    self.dc=None
    self.layerData=[]                                                   # die Daten aus getFile()
    self.inputfile=""                                                   # der aktuelle Dateiname
    self.raster_anzeigen=True                                           # ein Flag zur Anzeige des Rasters
    wx.CallLater(50, self.WarteAufDC)                                   # "on_paint" muss gelaufen sein

    self.cur_layer_number=0       # die Nummer des aktuellen Layers (wird in on_vscroll() verändert)
    self.cur_pos_in_layer=0       # die Position innerhalb des aktuellen Layers (wird in on_hscroll() verändert)
    self.m=[]                     # Linienzüge
    self.p=[]                     # die Farben der Linienzüge
    self.baseR=(10, 10)           # statischer/minimaler Abstand des Heizbetts vom Fensterrand [in Schirm-Koordinaten]
    self.base=(0, 0)              # Abstand des Heizbetts vom Fensterrand [in Schirm-Koordinaten], wird in on_size() gesetzt
    self.bed=HEIZBETT[0][1:]      # die Größe des Heizbetts [in mm]
    self.maxy=0                   # wird in on_size() auf die Fenster-Höhe minus Scrollbar gesetzt
    self.fx, self.fy=0, 0         # wird in on_size() auf die Skalierungsfaktoren gesetzt [für Bett-Koordinaten]
    self.shift=(0, 0)             # Verschiebung des Bett-Nullpunktes bei Zoom [in Bett-Koordinaten]
    self.zoom=2                   # der (doppelte) Zoomfaktor [für Bett-Koordinaten]

    self.left_down_pos=None       # wird in on_left_down() gesetzt und in on_left_up() zurückgesetzt
    self.anzeige_extr=True
    self.anzeige_trav=True
    self.heizbett_idx=0

    self.statusbar=self.parent.CreateStatusBar(6)
    self.statusbar.SetStatusWidths([100, 130, 70, 120, 120, -1])
    self.sbi={"layer":0, "move":1, "zoom":2, "pos":3, "shift":4}  # symbolische Namen für die Statusbar-Bereiche

    self.updateStatusbar()

    self.colors={ "Hintergrund"       : (0, "#CCCCCC", 1),    # Reihenfolge, Farbcode, Linienstärke
                  "Raster"            : (1, "#FFFFFF", 1),
                  "Extrusion"         : (2, "#000000", 1),
                  "Travel"            : (3, "#EFFF00", 1),
                  "Retraction Travel" : (4, "#0055F9", 1)
                }
    self.loadColorsFromConfig()
    self.setColors()


  # ###########################################################
  # Stellt die Farben gemäß "self.colors" ein.
  def setColors(self):
    self.bg=self.colors["Hintergrund"][1]
    self.pen_grid=wx.Pen(self.colors["Raster"][1])
    self.pen_extr=wx.Pen(self.colors["Extrusion"][1], self.colors["Extrusion"][2])
    self.pen_trav=wx.Pen(self.colors["Travel"][1], self.colors["Travel"][2])
    self.pen_retr=wx.Pen(self.colors["Retraction Travel"][1], self.colors["Retraction Travel"][2])

  # ###########################################################
  # Lädt die Farben aus dem Konfig-File.
  def loadColorsFromConfig(self):
    fc=wx.FileConfig(localFilename=CFG_FILE)
    for k, v in self.colors.items():
      c=fc.Read(k, self.colors[k][1])
      self.colors[k]=(self.colors[k][0], c, self.colors[k][2])

  # ###########################################################
  # Ruft sich so lange selbst wieder auf, bis "self.dc" in
  # "on_paint" gesetzt wurde, um dann die Datei zu laden.
  def WarteAufDC(self):
    if self.dc==None:
      wx.CallLater(50, self.WarteAufDC)
    elif len(sys.argv)>1:                 # wenn das Script mit Parametern aufgerufen wurde...
      self.inputfile=sys.argv[1]          # ...dann soll das Pfad+Name einer GCode-Datei sein
    self.loadFile(self.inputfile)
    self.updateStatusbar()

  # ###########################################################
  # Aktualisiert die Elemente der Statusbar.
  def updateStatusbar(self):
    self.statusbar.SetStatusText("layer: %d / %d"%(self.cur_layer_number, len(self.layerData)), self.sbi["layer"])
    self.statusbar.SetStatusText("move: %d / %d"%(self.cur_pos_in_layer, len(self.m)), self.sbi["move"])
    self.statusbar.SetStatusText("zoom: %d"%(self.zoom/2,), self.sbi["zoom"])
    if self.left_down_pos is not None:
      self.statusbar.SetStatusText("pos: %.1f, %.1f"%self.left_down_pos, self.sbi["pos"])
    else:
      self.statusbar.SetStatusText("pos: ?, ?", self.sbi["pos"])
    self.statusbar.SetStatusText("shift: %.1f, %.1f"%self.shift, self.sbi["shift"])

  # ###########################################################
  # Lädt die GCode-Datei gemäß "inputfile", stellt Layer-0 ein,
  # passt die Scrollbars an die Datei an und erzwingt den
  # Aufruf von on_paint().
  def loadFile(self, inputfile=""):
    if inputfile=="":
      self.layerData=[[]]
    else:
      self.layerData=getFile(inputfile)
    self.cur_layer_number=0
    self.recalcDrawing()
    self.cur_pos_in_layer=len(self.m)
    self.scrollbarH.SetScrollbar(len(self.m), 1, len(self.m), 1)
    self.scrollbarV.SetScrollbar(len(self.layerData), 1, len(self.layerData), 1)
    self.parent.SetTitle("GCode-View v"+VERSION+" ("+os.path.basename(inputfile)+")")
    self.Refresh(False)

  # ###########################################################
  # Wird aus FileDrop aufgerufen, wenn ein oder mehrere
  # DateiObjekte auf dem Fenster "fallengelassen" wurden.
  def fileDropped(self, filenames):
    self.inputfile=filenames[0]
    self.loadFile(self.inputfile)
    self.updateStatusbar()

  # ###########################################################
  # Ruft einmal pro Update-Intervall die Anzeige-Update-Funktion
  # auf.
  #def on_timer(self, event):
  #  self.cur_pos_in_layer+=1
  #  self.Refresh(False)

  # ###########################################################
  # Fenster-Grössen-Änderung verarbeiten.
  def on_size(self, event=None):
    maxx, maxy=self.GetSize()                                           # Größe des Fenters
    maxx-=self.scrollbarV.GetSize()[0]+2*self.baseR[0]                  # Größe des genutzten Fenter-Bereichs
    maxy-=self.scrollbarH.GetSize()[1]+2*self.baseR[1]
    svb=self.bed[0]/float(self.bed[1])                                  # Seitenverhältnis des Heizbetts (als Faktor für Y)
    xv=maxx/float(self.bed[0])                                          # Pixel-Breite eines Millimeters
    yv=maxy/float(self.bed[1])                                          # Pixel-Höhe eines Millimeters
    if xv>yv:                                                           # wenn der X-Rand verbreitert werden muss
      self.base=(self.baseR[0]+(maxx-self.bed[0]*yv)/2, self.baseR[1])  # X korrigieren
      maxx=maxy*svb                                                     # Heizbett anpassen (quadratisch machen)
    else:                                                               # wenn der Y-Rand verbreitert werden muss
      self.base=(self.baseR[0], self.baseR[1]+(maxy-self.bed[1]*xv)/2)  # Y korrigieren
      maxy=maxx/svb                                                     # Heizbett anpassen (quadratisch machen)
    self.maxy=maxy                                                      # maxy wird außerhalb gebraucht
    self.fx=float(maxx)/self.bed[0]                                     # Skalierung des Bettes auf den Fenter-Bereich
    self.fy=float(maxy)/self.bed[1]                                     # fx und fy sollten jetzt identisch sein...weil quadratisch
    if self.dc is not None:
      self.recalcDrawing()
      self.Refresh(False)
    if event is not None:
      event.Skip()

  # ###########################################################
  # An der horizontalen Scrollbar wurde der Wert verändert.
  def on_hscroll(self, event):
    self.cur_pos_in_layer=self.scrollbarH.GetThumbPosition()
    self.updateStatusbar()
    self.Refresh(False)

  # ###########################################################
  # An der vertikalen Scrollbar wurde der Wert verändert.
  def on_vscroll(self, event=None):
    self.cur_layer_number=len(self.layerData)-1-self.scrollbarV.GetThumbPosition()
    self.recalcDrawing()
    self.cur_pos_in_layer=len(self.m)
    self.scrollbarH.SetScrollbar(len(self.m), 1, len(self.m), 15)
    self.updateStatusbar()
    self.Refresh(False)

  # ###########################################################
  # Aktualisiert das Fenster.
  def on_paint(self, event):
    self.dc=wx.AutoBufferedPaintDC(self)
    self.dc.SetBackground(wx.Brush(self.bg))
    self.dc.Clear()

    if self.raster_anzeigen:
      self.dc.SetPen(self.pen_grid)
      for x in range(0, self.bed[0]+1, 10):                             # die vertikalen Raster-Linien
        u, o=self.bed2screen(x, 0), self.bed2screen(x, self.bed[1])
        self.dc.DrawLine(u[0], u[1], o[0], o[1])
      for y in range(0, self.bed[1]+1, 10):                             # die horizontalen Raster-Linien
        u, o=self.bed2screen(0, y), self.bed2screen(self.bed[0], y)
        self.dc.DrawLine(u[0], u[1], o[0], o[1])
    
    if self.cur_pos_in_layer<len(self.m):                               # wenn nicht der komplette Layer gezeichnet werden soll
      self.dc.DrawLineList(self.m[:self.cur_pos_in_layer], self.p[:self.cur_pos_in_layer])
      if self.cur_pos_in_layer>0:
        self.dc.DrawCircle(self.m[self.cur_pos_in_layer-1][2], self.m[self.cur_pos_in_layer-1][3], 4)
    else:                                                               # kompletten Layer zeichnen
      self.dc.DrawLineList(self.m, self.p)

  # ###########################################################
  # Liefert die Daten des Layers "layer_number" als move- und
  # pen-Liste bzw. Input für DrawLineList().
  def getLayerAsLineListAndPenList(self, layers, layer_number):
    layer=layers[layer_number]
    move=[]
    pen=[]

    if layer_number==0:                                                 # Startposition = Home = 0, 0
      lastX, lastY=self.bed2screen(0, 0)
    else:                                                               # Startposition = letzte Position aus dem vorigen Layer
      lastX, lastY, _, _=layers[layer_number-1][-1]
      lastX, lastY=self.bed2screen(lastX, lastY)

    for p in layer:
      X, Y, Z, E=p
      X, Y=self.bed2screen(X, Y)
      #move.append((lastX, lastY, X, Y))
      if E is None or E<0:                                              # Travel-Move
        if E is None:                                                   # Travel-Move ohne Retraktion
          if self.anzeige_trav:
            pen.append(self.pen_trav)
            move.append((lastX, lastY, X, Y))
        else:                                                           # Travel-Move mit Retraktion
          if self.anzeige_trav:
            pen.append(self.pen_retr)
            move.append((lastX, lastY, X, Y))
      else:                                                             # Extrusion-Move
        if self.anzeige_extr:
          pen.append(self.pen_extr)
          move.append((lastX, lastY, X, Y))
      lastX, lastY=X, Y
    return move, pen

  # ###########################################################
  # Berechnet die Linienzüge neu.
  def recalcDrawing(self):
    self.m, self.p=self.getLayerAsLineListAndPenList(self.layerData, self.cur_layer_number)

  # ###########################################################
  # Liefert zu Koordinaten im Fenster die Koordinaten auf dem
  # Heizbett.
  def screen2bed(self, x, y):
    x-=self.base[0]                                                     # Basisabstand vom Rand rausrechnen
    y=self.maxy-(y-self.base[1])                                        # bei Y zusätzlich umkehren
    X=(x/(self.fx*self.zoom/2))-self.shift[0]                           # skalieren und ggf. verschieben
    Y=(y/(self.fy*self.zoom/2))-self.shift[1]
    return X, Y

  # ###########################################################
  # Liefert zu Koordinaten auf dem Heizbett die Koordinaten
  # im Fenster.
  def bed2screen(self, x, y):
    X=round((x+self.shift[0])*(self.fx*self.zoom/2))
    Y=round((y+self.shift[1])*(self.fy*self.zoom/2))
    return X+self.base[0], (self.maxy-Y)+self.base[1]

  # ###########################################################
  # Scrollrad wurde betätigt.
  def on_mouse_wheel(self, event):
    x, y=event.GetPosition()            # Fenster-Koordinaten unter dem Mauszeiger
    oldpos=self.screen2bed(x, y)        # Bett-Koordinaten unter dem Mauszeiger
    if event.GetWheelRotation()>0:
      self.zoom+=2
      self.zoom=min(self.zoom, 40)
      newpos=self.screen2bed(x, y)
      self.shift=int(self.shift[0]+(newpos[0]-oldpos[0])), int(self.shift[1]+(newpos[1]-oldpos[1]))
    else:
      self.zoom-=2
      self.zoom=max(self.zoom, 2)
      newpos=self.screen2bed(x, y)
      self.shift=(self.shift[0]+(newpos[0]-oldpos[0]), self.shift[1]+(newpos[1]-oldpos[1]))
    self.updateStatusbar()
    self.recalcDrawing()
    self.Refresh(False)

  # ###########################################################
  # Stellt das Kontext-Menue dar.
  def on_context_menu(self, event):
    self.menue=wx.Menu()
    self.menue.Append(200, 'Grundstellung')
    self.Bind(wx.EVT_MENU, self.M_Grundstellung, id=200)

    self.submenue_anzeige=wx.Menu()
    self.menue.AppendSubMenu(self.submenue_anzeige, 'anzeigen...')
    self.submenue_anzeige.Append(300, 'Raster anzeigen', "", True)
    self.Bind(wx.EVT_MENU, self.M_Anzeige, id=300)
    self.submenue_anzeige.Check(300, self.raster_anzeigen)
    self.submenue_anzeige.Append(310, 'Extrusion-Moves anzeigen', "", True)
    self.Bind(wx.EVT_MENU, self.M_Anzeige, id=310)
    self.submenue_anzeige.Check(310, self.anzeige_extr)
    self.submenue_anzeige.Append(320, 'Travel-Moves anzeigen', "", True)
    self.Bind(wx.EVT_MENU, self.M_Anzeige, id=320)
    self.submenue_anzeige.Check(320, self.anzeige_trav)

    self.menue.AppendSeparator()
    self.submenue_heizbett=wx.Menu()
    self.menue.AppendSubMenu(self.submenue_heizbett, 'Heizbett')
    for i in range(len(HEIZBETT)):
      self.submenue_heizbett.Append(400+i, HEIZBETT[i][0], "", True)
      self.Bind(wx.EVT_MENU, self.M_Heizbett, id=400+i)
      self.submenue_heizbett.Check(400+i, self.heizbett_idx==i)

    self.menue.Append(600, 'Farben ändern')
    self.Bind(wx.EVT_MENU, self.M_Farben, id=600)

    self.menue.Append(900, 'Fenster-Daten speichern')
    self.Bind(wx.EVT_MENU, self.M_KonfigSpeichern, id=900)

    self.PopupMenu(self.menue)
    self.Refresh()

  # ###########################################################
  # Richtet das Heizbett gleichmäßig im Fenster aus.
  def M_Grundstellung(self, event):
    self.zoom=2
    self.shift=(0, 0)
    self.recalcDrawing()
    self.updateStatusbar()

  # ###########################################################
  # Submenü "anzeigen..." steuern
  def M_Anzeige(self, event):
    mid=event.GetId()
    if mid==300:
      self.raster_anzeigen=not self.raster_anzeigen
    elif mid==310:
      self.anzeige_extr=not self.anzeige_extr
      self.recalcDrawing()
      self.cur_pos_in_layer=len(self.m)
      self.scrollbarH.SetScrollbar(len(self.m), 1, len(self.m), 1)
      self.updateStatusbar()
    elif mid==320:
      self.anzeige_trav=not self.anzeige_trav
      self.recalcDrawing()
      self.cur_pos_in_layer=len(self.m)
      self.scrollbarH.SetScrollbar(len(self.m), 1, len(self.m), 1)
      self.updateStatusbar()

  # ###########################################################
  # Submenü "Heizbett" steuern
  def M_Heizbett(self, event):
    self.heizbett_idx=event.GetId()-400
    self.bed=HEIZBETT[self.heizbett_idx][1:]
    self.on_size()
    self.updateStatusbar()

  # ###########################################################
  # Öffnet den Farb-Dialog
  def M_Farben(self, event):
    dlg=FarbenDialog(self, "Farben", self.colors)
    ok_cancel=dlg.ShowModal()
    if ok_cancel==wx.ID_OK:
      self.colors=dlg.GetValues()
      self.setColors()
      self.recalcDrawing()

  # ###########################################################
  # Speichert Fenster-Größe und Position und ein paar weitere
  # Einstellungen.
  def M_KonfigSpeichern(self, event):
    fc=wx.FileConfig(localFilename=CFG_FILE)
    sp=self.parent.GetScreenPosition()
    ss=self.parent.GetSizeTuple()
    fc.WriteInt("pos_x",    sp[0])
    fc.WriteInt("pos_y",    sp[1])
    fc.WriteInt("size_x" ,  ss[0])
    fc.WriteInt("size_y" ,  ss[1])
    for k, v in self.colors.items():
      fc.Write(k, v[1])
    fc.Flush()

  # ###########################################################
  # Die linke Maustaste wurde gedrückt.
  def on_left_down(self, event):
    x, y=event.GetPosition()
    self.left_down_pos=self.screen2bed(x, y)
    self.updateStatusbar()

  # ###########################################################
  # Die linke Maustaste wurde losgelassen.
  def on_left_up(self, event):
    x, y=event.GetPosition()
    left_up_pos=self.screen2bed(x, y)
    self.shift=self.shift[0]+(left_up_pos[0]-self.left_down_pos[0]), self.shift[1]+(left_up_pos[1]-self.left_down_pos[1])
    self.left_down_pos=None   # vermerken, dass die linke Maustaste losgelassen wurde
    self.recalcDrawing()
    self.Refresh()

  # ###########################################################
  # Bei Mausbewegung über dem Fenster.
  def on_motion(self, event):
    if self.left_down_pos is not None:    # nur, wenn gerade die linke Maustaste gedrückt ist
      x, y=event.GetPosition()
      left_up_pos=self.screen2bed(x, y)
      self.shift=self.shift[0]+(left_up_pos[0]-self.left_down_pos[0]), self.shift[1]+(left_up_pos[1]-self.left_down_pos[1])
      self.recalcDrawing()
      self.updateStatusbar()
      self.Refresh()



# ###########################################################
# Das Dialogfenster zur Farb-Änderung
class FarbenDialog(wx.Dialog):
  def __init__(self, parent, caption, col_dict):
    wx.Dialog.__init__(self, parent, wx.ID_ANY, caption, size=(600, 800), style=wx.CAPTION|wx.RESIZE_BORDER|wx.CLOSE_BOX)
    self.col_dict=col_dict

    st=[]       # StaticText
    self.cpc=[] # ColourPickerCtrl
    sc=[]       # Colour
    idx=0
    for k, v in sorted(col_dict.items(), key=lambda kv:kv[1][0]):
      st.append(wx.StaticText(self, label=k+":"))
      self.cpc.append(wx.ColourPickerCtrl(self, wx.ID_ANY))
      self.cpc[idx].SetColour(v[1])
      idx+=1
    ok=wx.Button(self,      wx.ID_OK,     "&Ok")
    cancel=wx.Button(self,  wx.ID_CANCEL, "&Cancel")
    ok.SetDefault()
    ok.SetFocus()
      
    sizer=wx.GridBagSizer()

    flag=wx.ALIGN_CENTER_VERTICAL|wx.ALL
    border=5
    for i in range(len(st)):
      sizer.Add(st[i],        pos=(i//2, (i%2)*2+0), flag=flag, border=border)
      sizer.Add(self.cpc[i],  pos=(i//2, (i%2)*2+1), flag=flag, border=border)

    sizer.Add(wx.StaticLine(self),  pos=(len(st)//2+1, 0), span=(1, 4),  flag=wx.ALL|wx.GROW, border=5)
    sizer.Add(ok,                   pos=(len(st)//2+2, 0),             flag=flag, border=border)
    sizer.Add(cancel,               pos=(len(st)//2+2, 1),             flag=flag, border=border)

    self.SetSizerAndFit(sizer)
    self.Center()

  # ###########################################################
  # Liefert die eingestellten Farben als Dictionary zurück.
  def GetValues(self):
    col_dict={}
    idx=0
    for k, v in sorted(self.col_dict.items(), key=lambda kv:kv[1][0]):
      col=self.cpc[idx].GetColour().GetAsString(wx.C2S_HTML_SYNTAX)
      col_dict[k]=(self.col_dict[k][0], col, self.col_dict[k][2])
      idx+=1
    return(col_dict)





# ###########################################################
# Der Fenster-Rahmen fuer das Hauptfenster.
class GCVFrame(wx.Frame):
  def __init__(self, parent, pos=wx.DefaultPosition, size=wx.DefaultSize):
    style=wx.DEFAULT_FRAME_STYLE
    wx.Frame.__init__(self, None, wx.ID_ANY, "GCode-View v"+VERSION, pos=pos, size=size, style=style)
    self.panel=GCVPanel(self)


# ###########################################################
# Der Starter.
if __name__=='__main__':
  fc=wx.FileConfig(localFilename=CFG_FILE)
  spx=fc.ReadInt("pos_x", -1)
  spy=fc.ReadInt("pos_y", -1)
  ssx=fc.ReadInt("size_x", 800)
  ssy=fc.ReadInt("size_y", 600)
  sp=(spx, spy) # (-1, -1) entspricht wx.DefaultPosition
  ss=(ssx, ssy) # (-1, -1) entspricht wx.DefaultSize

  app=wx.App(False)
  frame=GCVFrame(None, pos=sp, size=ss)
  frame.Show(True)
  app.MainLoop()
