#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os, sys
from optparse import OptionParser
import time

# ------------------------------------------------------------------------------------------------
# Dieses Script liefert eine ziemlich optimale Datei-Zusammenstellung der Dateien unter "src_pth"
# bezüglich des Füllgrades des gemounteten Datenträgers "dest_pth".
# Die Dateien aus "src_pth", die sich bereits auf "dest_pth" befinden, werden nicht als erneut
# zu kopieren berücksichtigt. Dadurch können zusammengehörige Dateien vorab auf "dest_pth"
# kopiert und der Datenträger danach mit diesem Script randvoll gefüllt werden.
#
# D.A. 11.12.2013
#
# 23.01.2014  Restkapazität wird nun anhand von sz.f_bavail statt sz.f_bfree bestimmt.
#             Außerdem den Parameter -r hinzugefügt.
#             Damit klappt es dann auch auf Containern, die nicht mit vFAT formatiert sind.
#             Zusätzlich noch den "Trick" mit der virtuellen 0-Byte-Datei eingebaut, durch den nun
#             nur noch ein Aufruf von findeZusammenstellung() notwendig ist. Nebenbei wird dadurch
#             nun auch ein sinnvolles Ergebnis geliefert, wenn die Restkapatität kleiner als die
#             zweitgrößte Datei ist.
# ------------------------------------------------------------------------------------------------


# die Dateinen unter den folgenden Pfaden sollen gesichert werden
src_pth_g=["/2TB/BackupWartend", "/1TB/ISOs_neu"]

# unterhalb von diesem Pfad ist der True- bzw. Real-Crypt-Container gemounted
dest_pth_g="/tc"

dest_defaults_kapazitaet  =24931450880   # Kapazität auf leeren TC-Container für BD setzen
dest_defaults_blockgroesse=8192          # analog für die Blocksize des TC-Containers


# ###########################################################
# Initialisiert den OptionParser.
def aufsetzenOptionParser():
  usage="fillBD.py [optionen]\n\n" \
        "Liefert eine ziemlich optimale Datei-Zusammenstellung der Dateien unter\n"
  for i in src_pth_g:
    usage+="   "+i+"\n"
  usage+="bezueglich des Fuellgrades des gemounteten Datentraegers unter\n"
  usage+="   "+dest_pth_g+"\n\n"
  usage+="Dateien, die sich bereits auf dem Zieldatentraeger befinden, werden\n" \
         "ausgeblendet. Dadurch koennen zusammengehoerige Dateien vorab auf den\n" \
         "Zieldatentraeger kopiert und der Datentraeger danach mit diesem Script\n" \
         "gefuellt werden."
  parser=OptionParser(usage)
  parser.add_option("-d", "--dry", dest="dry", action="store_true", 
                  help="so tun, als waere ein leerer TC-Container gemountet")
  parser.add_option("-r", "--reserve", dest="reserve", action="store_true", 
                  help="fuer Journaling-Dateisysteme (ext3/ext4) pro Datei vier zusaetzliche Bloecke reservieren")
  parser.add_option("-v", "--verbose", dest="verbose", action="store_const", const=1,
                  help="Status-Ausgaben einschalten (jede jeweils bessere Kombination)")
  parser.add_option("-n", "--noisy", dest="verbose", action="store_const", const=2,
                  help="Status-Ausgaben einschalten (jede getestete Kombination)")
  return(parser.parse_args())


# ###########################################################
# Liefert "num" in der passendsten Groessenangabe.
def prettySize(num):
  for x in ['','KB','MB','GB', 'TB']:
    if num<1024.0:
      return("{0:3.0f} {1:s}".format(num, x))
    num/=1024.0


# ###########################################################
# Liefert "wert" als Dualzahl in der Länge "laenge-1".
# Das unterste Bit wird abgeschnitten.
# "0" wird durch "-" und "1" durch "X" ersetzt.
def int2dual(wert, laenge):
  d="-"*laenge + bin(wert)[2:].replace("0", "-").replace("1", "X")
  e=d[len(d)-laenge:]
  return(e[:len(e)-1])


# ###########################################################
# Prüft alle Objekte unter "unterhalb_von_verzeichnis", ob
# sie ein Mountpoint sind.
# Existiert dort genau ein Mountpoint, wird dessen
# Restkapazität, Blockgröße und Name als Tupel geliefert.
# Ansonsten wird (0, 0, "") geliefert.
def holeMountpoint(unterhalb_von_verzeichnis):
  f=os.listdir(unterhalb_von_verzeichnis)
  mp_name=""    # Name des Mountpoits (ohne Pfad)
  cnt=0         # Zähler für die Anzahl der Mountpoints
  for i in f:                                       # über alle Datei- und Verzeichnis-Namen
    pthf=os.path.join(unterhalb_von_verzeichnis, i) # um Pfad ergänzen
    if os.path.isdir(pthf)==True:                   # wenn es ein Verzeichnis ist...
      if os.path.ismount(pthf)==True:               # und gemountet ist...
        cnt+=1                                      # dann zählen
        mp_name=i                                   # und merken

  if cnt==1:                                        # wenn genau ein Mountpoint gefunden wurde
    sz=os.statvfs(os.path.join(unterhalb_von_verzeichnis, mp_name)) # dessen Daten auslesen
    return((sz.f_bsize*sz.f_bavail, sz.f_bsize, mp_name))  # und liefern als (Restkapazität, Blockgröße, Name)
  return((0, 0, ""))  # Kennung für "kein oder mehr als ein Mountpoint"


# ###########################################################
# Liefert eine Liste mit allen Dateien unter "src_pth_g", die
# nicht bereits auch unter dem Mountpoint zu finden sind
# sowie die Summe der Größen dieser Dateien als Tupel.
# Die Liste enthält Elemente im Format: (Pfad, Dateiname, Größe)
#
# Lesend genutzte globale Variablen: src_pth_g, dest_pth_g
def holeZuKopierendeDateien(mountpoint, blockgroesse_ziel, reserve):
  dateienSchonAufZiel=[]    # Liste aller Dateien, die bereits vorab auf das Ziel kopiert wurden
  if mountpoint!="":
    f=os.listdir(os.path.join(dest_pth_g, mountpoint))
    for i in f:
      dateienSchonAufZiel.append(i)

  dateienAufQuelle=[]                   # Liste aller Dateien mit Pfad aus "src_pth_g"
  for i in src_pth_g:
    f=os.listdir(i)
    for j in f:
      if j not in dateienSchonAufZiel:  # aber ohne die vorab aufs Ziel kopierten Dateien
        dateienAufQuelle.append((i, j))

  gesamt_dateigroesse=0
  dateienZuKopieren=[]                  # Liste aller Dateien mit Pfad und Dateigröße
  for p, n in dateienAufQuelle:
    fs=os.path.getsize(os.path.join(p, n))
    fsb=fs+(blockgroesse_ziel-(fs%blockgroesse_ziel)) # Dateigröße auf Blockgröße ausrichten
    if reserve==True:
      fsb+=4*blockgroesse_ziel                        # für ext[2,3,4] noch pro Datei vier Blocks draufschlagen
    gesamt_dateigroesse+=fsb
    dateienZuKopieren.append((p, n, fsb))
  return((dateienZuKopieren, gesamt_dateigroesse))


# ###########################################################
# Liefert eine Liste mit den Summen der Dateigrößen, die ab
# dem jeweiligen Index in "dateiListe_sortiert" noch kommen.
#
# Lesend genutzte globale Variablen: dateiListe_sortiert, anzahl_dateien
def bildeRestSummenListe():
  lst=[]
  summe=0
  for i in range(anzahl_dateien-1, -1, -1): # von klein nach groß
    summe+=dateiListe_sortiert[i][2]        # Zwischensummen bilden
    lst.append(summe)                       # und ablegen
  lst.reverse() # Liste umdrehen, damit auf Index 0 die Gesamtsumme landet
  return(lst)


# ###########################################################
# Sucht rekursiv eine optimale Zusammenstellung der Dateien
# in der globalen Liste "dateiListe_sortiert".
#
# Parameter:
#   index             - der aktuell getestete Index in "dateiListe_sortiert"
#   bisherige_groesse - bisherige Summe der Dateigrößen der gewählten Dateien
#   beste_restgroesse - bisherige kleinste gefundene Restkapazität
#   zielgroesse       - statisch: die Kapazität des Zieldatenträgers
#   bitmap            - Zahl als gedachte Bitmap mit einem Bit pro Datei[größe]
#   verbose           - auf None, 1, 2
#
# Lesend genutzte globale Variablen: dateiListe_sortiert, anzahl_dateien, 
#                                    dateiListe_RestSummen, start_zeit
def findeZusammenstellung(index, bisherige_groesse, beste_restgroesse, zielgroesse, bitmap, blockgr, verbose):
  neue_groesse=bisherige_groesse+dateiListe_sortiert[index][2]  # aktuelle Dateigröße zufügen
  bitmap|=(2**index)                                            # und in der Bitmap vermerken
  ende=False

  if neue_groesse<zielgroesse:                                  # wenn es noch gepasst hat
    verbleibende_groesse=zielgroesse-neue_groesse               # neue Restkapazität berechnen

    if verbose==2:                                              # auf Wunsch jede getestete Bitmap anzeigen
      print int2dual(bitmap, anzahl_dateien), "{0:15} {1:15}".format(verbleibende_groesse, beste_restgroesse)

    if verbleibende_groesse>0 and verbleibende_groesse<beste_restgroesse: # wenn neue beste Zusammenstellung gefunden
      beste_restgroesse=verbleibende_groesse                              # merken
      if verbose==1:                                                      # und auf Wunsch...
        print int2dual(bitmap, anzahl_dateien), prettySize(beste_restgroesse) # anzeigen
      if beste_restgroesse<=(blockgr):              # bei maximal "blockgr" Restkapazität...
        return((bitmap, beste_restgroesse, True))   # ist das gut genug -> Rekursion beenden

    # Optimierung: wenn Restkapazität kleiner als die kleinste Datei...
    if verbleibende_groesse<dateiListe_sortiert[anzahl_dateien-1][2]:
      # dann macht es hier keinen Sinn mehr, weiter zu testen
      return((bitmap, beste_restgroesse, ende))

    if (time.clock()-start_zeit)>10:  # Abbruch nach 10 Sekunden
      return((bitmap, beste_restgroesse, ende))

    if index<anzahl_dateien: # wenn nicht bereits die letzte Datei erreicht wurde
      bitmap2=bitmap  # Bitmap sichern
      for i in range(1, anzahl_dateien-index): # rekursiver Abstieg über alle möglichen kleineren Dateigrößen
        # Die aktuelle Größe in "neue_groesse" zusammen mit der maximal möglichen Restgröße in
        # "dateiListe_RestSummen" ergibt die maximal noch erreichbare Gesamtgröße.
        # Die Größe des Datenträgers in "zielgroesse" abzüglich der maximal erreichbaren Gesamtgröße ergibt
        # die bestenfalls noch erreichbare Restgröße.
        # Ist die größer als die bereits gefundene "beste_restgroesse", braucht dafür kein rekursiver
        # Abstieg mehr erfolgen.
        if (zielgroesse-(neue_groesse+dateiListe_RestSummen[index+i]))>beste_restgroesse:
          #print int2dual(bitmap, anzahl_dateien), index+i, zielgroesse, neue_groesse, \
          #      dateiListe_RestSummen[index+i], (neue_groesse+dateiListe_RestSummen[index+i]), beste_restgroesse
          break
        (bm, br, ende)=findeZusammenstellung(index+i, neue_groesse, beste_restgroesse, zielgroesse, bitmap, blockgr, verbose)
        if br<beste_restgroesse:  # wenn neue beste Zusammenstellung gefunden
          beste_restgroesse=br    # Restkapazität merken
          bitmap2=bm              # und Bitmap nach der Schleife an Aufrufer übergeben
        if ende==True:            # wenn Zusammenstellung mit maximal "blockgr" Restkapazität gefunden wurde
          break                   # Schleife abbrechen und Rekursion beenden
      bitmap=bitmap2  # möglicherweise gefundene bessere Bitmap einstellen oder sonst alte Bitmap wiederherstellen
  return((bitmap, beste_restgroesse, ende))





if __name__ == "__main__":
  (parameter, args)=aufsetzenOptionParser()

  if parameter.dry==True:
    restkap=dest_defaults_kapazitaet   # Kapazität auf leeren TC-Container für BD setzen
    blockgr=dest_defaults_blockgroesse # analog für die Blocksize des TC-Containers
    mountpoint=""
  else:
    restkap, blockgr, mountpoint=holeMountpoint(dest_pth_g)
    if restkap==0 and blockgr==0 and mountpoint=="":  # wenn Fehler-Kennung...
      print "Kein, oder mehr als ein MOUNT unter", dest_pth_g, "gefunden!"
      sys.exit()
  dateiListe, groesse=holeZuKopierendeDateien(mountpoint, blockgr, parameter.reserve)

  print "Kapazität auf dem Ziel:", prettySize(restkap)
  if groesse<restkap:
    print "Benötigte Kapazität   :", groesse/(1024*1024), "MB"
    print "Passt doch alles komplett.....!?"
    sys.exit()

  anzahl_dateien=len(dateiListe)
  print "Dateien:", anzahl_dateien, "  Gesamtgröße:", prettySize(groesse)#/(1024*1024), "MB"

  # Die größten Dateien kommen an den Anfang der Liste
  dateiListe_sortiert=sorted(dateiListe, key=lambda x: x[2], reverse=True)
  # Vor die größte Datei wird eine garantiert passende Datei (Länge=0) gelegt,
  # damit von da aus der rekursive Abstieg erfolgen kann und bereits die erste
  # "echte" Datei ein Kandidat ist, um ausgelassen bzw. übersprungen zu werden.
  dateiListe_sortiert.insert(0, ("/dev", "null", 0))
  anzahl_dateien+=1

  # In dieser Liste wird auf jedem Index gemäß "dateiListe_sortiert" der
  # Platzbedarf aller jeweils kleineren Dateien abgelegt. Damit kann in
  # findeZusammenstellung() erkannt werden, dass ein rekursiver Abstieg
  # nicht mehr durchlaufen werden muss.
  # Auf dem ersten Index liegt also die Gesamtgröße aller Dateien.
  # Auf dem zweiten Index fehlt nur die Größe der ersten Datei.
  # Auf dem letzten Index liegt die Größe der kleinsten Datei.
  dateiListe_RestSummen=bildeRestSummenListe()

  start_zeit=time.clock()
  bitmap, beste_restgroesse, f=findeZusammenstellung(0, 0, restkap, restkap, 0, blockgr, parameter.verbose)

  if beste_restgroesse==restkap:
    print "Kein Platz auf dem Ziel!"
    sys.exit()

  summe=0
  print
  if mountpoint!="":
    print 'cd "' + os.path.join(dest_pth_g, mountpoint) + '"'
  for j in range(1, len(dateiListe_sortiert)):  # startet bei 1, um die leere Datei zu überspringen
    if bitmap&(2**j)>0:
      print 'cp "' + os.path.join(dateiListe_sortiert[j][0], dateiListe_sortiert[j][1]) + '" .'
      summe+=dateiListe_sortiert[j][2]

  fmt="{0:15}"
  print
  print "Summe Dateigrößen:", fmt.format(summe),         "=", prettySize(summe)
  print "Datenträger      :", fmt.format(restkap),       "=", prettySize(restkap)
  print "Restkapazität    :", fmt.format(restkap-summe), "=", prettySize(restkap-summe)

