#!/usr/bin/env python
# -*- coding: utf-8 -*-

import wx
import globalStuff as glb

# ----------------------------------------------------------------------
# 
class Table():
    def __init__(self, name, cols, pos, imported_from=None, comment=""):
        self.name=name                                                  # der Name der Tabelle
        self.cols=cols                                                  # die Spaltennamen der Tabelle im Format (name, typ, attribute)
        self.pos=glb.pointOnRaster(pos)                                 # die Position der Tabelle
        self.imported_from=imported_from                                # True, wenn die Tabelle per IMPORT_TABLE geladen wurde
        self.comment=comment.strip().replace('\\n', '\n')               # nur gesetzt, wenn die Tabelle eigentlich ein Kommentar ist
        self.header_color="#00ffff"                                     # die Hintergrundfarbe des Tabellen-Headers
        self.size=(0, 0)                                                # die Abmessungen der Tabelle
        self.size_set=False                                             # wird in self.calcSize() auf True gesetzt (um Doppel-Berechnung zu blocken)
        self.row_height=None                                            # wird in self.calcSize() auf die Zeilen-Höhe gesetzt
        self.row_width=None                                             # wird in self.calcSize() auf die benötigte Zeilen-Breite gesetzt
        self.row_char_len=None                                          # wird zur Ausrichtung in write() benötigt
        self.col_y={}                                                   # wird in self.calcSize() auf die y-Position je Spaltenname gesetzt
        for n, t, a in cols:                                            # [eigentlich überflüssige] Initialisierung des Dictionaries...
            self.col_y.update({n: 0})                                   # ...für die relative Y-Position jeder Spalte

    # ------------------------------------------------------------------
    # Liefert True, wenn diese Tabelle eine NOTE ist.
    def isNote(self):
        return self.comment!=""

    # ------------------------------------------------------------------
    # Richtet die Position gemäß new_raster neu aus.
    def changeRaster(self, new_raster):
        x=int(self.pos[0]/float(glb.raster))*new_raster
        y=int(self.pos[1]/float(glb.raster))*new_raster
        self.pos=glb.pointOnNewRaster((x, y), new_raster)

    # ------------------------------------------------------------------
    # Stellt die Darstellungs-Position der Tabelle auf new_xy ein.
    def moveAbs(self, new_xy):
        self.pos=glb.pointOnRaster(new_xy)

    # ------------------------------------------------------------------
    # Ändert die Darstellungs-Position der Tabelle um delta_xy.
    def moveRel(self, delta_xy):
        dx, dy=delta_xy
        self.pos=(self.pos[0]+dx, self.pos[1]+dy)

    # ------------------------------------------------------------------
    # Liefert True, wenn xy über der Tabelle liegt.
    def isUnderXY(self, xy):
        return self.pos[0]<=xy[0]<=self.pos[0]+self.size[0] and self.pos[1]<=xy[1]<=self.pos[1]+self.size[1]

    # ------------------------------------------------------------------
    # Liefert das gemeinsame Rechteck von norm_rect und der Tabelle
    # oder None, wenn es keine Überschneidung gibt.
    # Quelle: https://stackoverflow.com/a/25068722/3588613
    def hasIntersectionWithRectangle(self, norm_rect):
        x1=max(min(self.pos[0], self.pos[0]+self.size[0]), min(norm_rect[0][0], norm_rect[1][0]))
        y1=max(min(self.pos[1], self.pos[1]+self.size[1]), min(norm_rect[0][1], norm_rect[1][1]))
        x2=min(max(self.pos[0], self.pos[0]+self.size[0]), max(norm_rect[0][0], norm_rect[1][0]))
        y2=min(max(self.pos[1], self.pos[1]+self.size[1]), max(norm_rect[0][1], norm_rect[1][1]))
        if x1<x2 and y1<y2:
            return (x1, y1), (x2, y2)
        return None

    # ------------------------------------------------------------------
    # Erzwingt eine Neuberechnung der Tabellen-Abmessungen.
    # Stichwort: ZoomIn/ZoomOut
    def recalcSize(self):
        self.size_set=False
        self.row_height=None
        self.row_width=None
        self.row_char_len=None
        self.col_y={}

    # ------------------------------------------------------------------
    # Berechnet anhand der darzustellenden Texte die Abmessungen der
    # Tabelle / des Kommentars.
    def calcSize(self, dc):
        if self.size_set:
            return
        self.size_set=True
        if self.isNote():
            dc.SetFont(wx.Font(glb.font_size, wx.FONTFAMILY_TELETYPE, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
            sx, sy=dc.GetTextExtent("-")
            lns=self.comment.split("\n")
            self.row_height=sy
            for ln in lns:
                row_width, row_height=dc.GetTextExtent(ln)
                if self.row_width is None:
                    self.row_width=row_width
                else:
                    self.row_width=max(self.row_width, row_width)
                self.row_height+=row_height
            self.size=(self.row_width+sx, self.row_height)
        else:
            dc.SetFont(wx.Font(glb.font_size, wx.FONTFAMILY_TELETYPE, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
            hdr_row_width, self.row_height=dc.GetTextExtent(" "+self.name+" ")
            dc.SetFont(wx.Font(glb.font_size, wx.FONTFAMILY_TELETYPE, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
            self.row_width=[0, 0, 0]
            self.row_char_len=[0, 0, 0]
            for c in self.cols:
                for i in (0, 1, 2):
                    self.row_width[i]=max(self.row_width[i], dc.GetTextExtent(c[i]+" ")[0])
                    self.row_char_len[i]=max(self.row_char_len[i], len(c[i]))
            s=glb.pointOnRaster((max(sum(self.row_width), hdr_row_width), 10))
            self.size=(s[0], 1+len(self.cols)*glb.raster+3*glb.raster)
            for i, c in enumerate(self.cols):
                self.col_y[c[0]]=int((i+3)*glb.raster)

    # ------------------------------------------------------------------
    # Stellt die Tabelle (oder den Kommentar) auf dc dar.
    # Bei Übergabe von isSelected=True wird die Umrandung dicker und in
    # einer anderen Farbe dargestellt.
    def drawTable(self, dc, isSelected=False):
        if isSelected:
            dc.SetPen(wx.Pen("#ff0000", 3))
        if self.isNote():
            dc.SetBrush(wx.Brush("#89FF06", wx.SOLID))
            dc.DrawRectangle(self.pos[0], self.pos[1], self.size[0], self.size[1])
            dc.SetFont(wx.Font(glb.font_size, wx.FONTFAMILY_TELETYPE, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
            sx, sy=dc.GetTextExtent("-")
            dc.DrawText(self.comment, self.pos[0]+sx/2, self.pos[1]+sy/2)
        else:
            dc.SetBrush(wx.Brush(self.header_color, wx.SOLID))
            dc.DrawRectangle(self.pos[0], self.pos[1], self.size[0], 1+2*glb.raster)    # der Tabellenname ist immer 2*RASTER groß
            dc.SetBrush(wx.Brush("WHITE", wx.SOLID))
            dc.DrawRectangle(self.pos[0], self.pos[1]+2*glb.raster-1, self.size[0], self.size[1]-2*glb.raster)
            dc.SetFont(wx.Font(glb.font_size, wx.FONTFAMILY_TELETYPE, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
            sx=dc.GetTextExtent("-")[0]/2
            for i, c in enumerate(self.cols):
                y=self.pos[1]+(3+i)*glb.raster-self.row_height/2
                dc.DrawText(c[0], self.pos[0]+sx,                                     y)
                dc.DrawText(c[1], self.pos[0]+sx+self.row_width[0],                   y)
                dc.DrawText(c[2], self.pos[0]+sx+self.row_width[0]+self.row_width[1], y)
            dc.SetFont(wx.Font(glb.font_size, wx.FONTFAMILY_TELETYPE, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
            dc.DrawText(self.name, self.pos[0]+sx, self.pos[1]+glb.raster-self.row_height/2)
            dc.SetFont(wx.Font(glb.font_size, wx.FONTFAMILY_TELETYPE, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
        if isSelected:
            dc.SetPen(wx.Pen("#000000", 1))

    # ------------------------------------------------------------------
    # Liefert die TableConnector-Koordinate für die Spalte col_name an
    # der Seite border.
    def getReferencePoint(self, border, col_name):
        if col_name in self.col_y:
            if border=="L":
                return self.pos[0], self.pos[1]+self.col_y[col_name]
            elif border=="R":
                return self.pos[0]+self.size[0], self.pos[1]+self.col_y[col_name]

    # ------------------------------------------------------------------
    # Liefert ein Tupel aus zwei Strings. Der erste String enthält die
    # (statische) Tabellen-Deklarationen, der zweite String enthält die
    # (variablen) Positionsdaten.
    def write(self):
        strg_d="\n"
        if self.imported_from:
            strg_d+="IMPORT_TABLE %s\n"%(self.imported_from,)
            if self.header_color!="#00ffff":
                strg_d+="TABCOLOR %s %s\n"%(self.name, self.header_color)
        else:
            strg_d+="TABLE %s\n"%(self.name,)
            for col in self.cols:
                if col[2]=="":
                    strg_d+="    %s  %s\n"%(col[0].ljust(self.row_char_len[0]), col[1].ljust(self.row_char_len[1]))
                else:
                    strg_d+="    %s  %s  %s\n"%(col[0].ljust(self.row_char_len[0]), col[1].ljust(self.row_char_len[1]), col[2])
            if self.header_color!="#00ffff":
                strg_d+="TABCOLOR %s %s\n"%(self.name, self.header_color)
        strg_p="TABPOS %s (%s,%s)\n"%(self.name, self.pos[0], self.pos[1])
        return strg_d, strg_p
