#!/usr/bin/env python
# -*- coding: utf-8 -*-

import globalStuff as glb

# ----------------------------------------------------------------------
# 
class Relation():
    def __init__(self, tables, name, rel_type, start_col, end_col, border_start="R", border_end="R", is_constraint=True, point_list_between=None):
        self.tables=tables                              # die Liste der Table-Objekte
        self.name=name                                  # der Name der Relation
        self.is_constraint=is_constraint                # auf True, wenn die Relation tatsächlich ein Constraint ist
        self.rel_type=rel_type.split("-")               # der Typ der Relation ("1-1", "1-n", ...)
        self.start_col=start_col                        # Tabelle+"."+Spalte, von der die Relation startet ("domains.id")
        self.border_start=border_start                  # Start der Relation [L]inks oder [R]echts an der Tabelle
        self.end_col=end_col                            # Tabelle+"."+Spalte, von der die Relation endet ("domain_to_ip.domain_id")
        self.border_end=border_end                      # Ende der Relation [L]inks oder [R]echts an der Tabelle
        if point_list_between is None:                  # ggf. die zwischen Start und Ende liegenden Punkte laden
            self.line=[]
        else:
            self.line=point_list_between                # erwartetes Format: [(10,10), (10,50), (50,50), (100,50)]
        self.start_tab=self.__getTableFor(start_col)    # das Table-Objekt, bei dem diese Relation startet
        self.end_tab=self.__getTableFor(end_col)        # das Table-Objekt, bei dem diese Relation endet

    # ------------------------------------------------------------------
    # Liefert das Table-Objekt zur tab_col (wie etwa "ips.ip_num").
    def __getTableFor(self, tab_col):
        tab, col=tab_col.split(".")
        for t in self.tables:
            if t.name==tab:
                return t
        print("Table not found:", tab_col)
        return None

    # ------------------------------------------------------------------
    # Liefert den bei Instanziierung übergebenen Typ der Relation.
    def isConstraint(self):
        return self.is_constraint

    # ------------------------------------------------------------------
    # Richtet alle Eckpunkte gemäß new_raster neu aus.
    def changeRaster(self, new_raster):
        for i, l in enumerate(self.line):
            x=int(l[0]/float(glb.raster))*new_raster
            y=int(l[1]/float(glb.raster))*new_raster
            self.line[i]=glb.pointOnNewRaster((x, y), new_raster)

    # ------------------------------------------------------------------
    # Stellt die gegenüber-liegende Seite des jew. Borders ein.
    def swapStartBorder(self):
        if self.border_start=="L":  self.border_start="R"
        else:                       self.border_start="L"
    def swapEndBorder(self):
        if self.border_end=="L":    self.border_end="R"
        else:                       self.border_end="L"

    # ------------------------------------------------------------------
    # Liefert die komplette Linie zur Darstellung der Relation.
    def getFullList(self):
        return [self.start_tab.getReferencePoint(self.border_start, self.start_col.split(".")[1])] + \
               self.line + \
               [self.end_tab.getReferencePoint(self.border_end, self.end_col.split(".")[1])]

    # ------------------------------------------------------------------
    # Liefert None, wenn point nicht in der Nähe dieser Relation liegt.
    # Wenn point auf einem Eckpunkt liegt, wird ("CM", Index) geliefert.
    # Wenn point auf einem Teilstück zwischen Eckpunkten liegt, wird
    # ("LN", Index) geliefert.
    def isPointOnThisRelation(self, point):
        i1=self.isPointOnLine(point)          # zuerst der aufwendige Test...
        if i1 is None:                        # ...weil der für Eckpunkte und Teilabschitte funktioniert
            return None
        i2=self.isPointOnCornerMark(point)    # erst wenn ein Treffer für einen Teilabschnitt gefunden wurde,...
        if i2 is not None:                    # ...wird auch der Eckpunkt-Test ausgeführt...
            return "CM", i2+1                 # ...und ggf. bevorzugt zurückgeliefert.
        return "LN", i1

    # ------------------------------------------------------------------
    # Liefert den Index i in getFullList(), hinter dem ein neuer
    # Eckpunkt eingefügt werden muss, wenn point gemäß r nah genug auf
    # der Linie zwischen i und i+1 lag. Lag point zu weit von allen
    # Linien bzw. Teilabschnitten entfernt, wird None geliefert.
    def isPointOnLine(self, point, r=4):
        fl=self.getFullList()
        for i in range(len(fl)-1):
            if self.lineSquareIntersect(fl[i], fl[i+1], point, r):
                return i
        return None

    # ------------------------------------------------------------------
    # Liefert den Index des Eckpunktes in self.line, der gemäß r
    # nah genug an point liegt.
    # Wird kein passender Eckpunkt gefunden, wird None geliefert.
    def isPointOnCornerMark(self, point, r=4):
        for i, p in enumerate(self.line):
            if point[0]-r<=p[0]<=point[0]+r and point[1]-r<=p[1]<=point[1]+r:
                return i
        return None

    # ------------------------------------------------------------------
    # Liefert ein Set von Indices (in self.line) der Eckpunkte, die
    # im normalisierten Rechteck norm_rect liegen.
    def getCornerMarksUnderRectangle(self, norm_rect):
        s=set()
        for i, p in enumerate(self.line):
            if norm_rect[0][0]<p[0]<norm_rect[1][0] and norm_rect[0][1]<p[1]<norm_rect[1][1]:
                s.add(i)
        return s

    # ------------------------------------------------------------------
    # Liefert True, wenn sich point über der Koordinate eines
    # TableConnectors befindet.
    def isPointOnTableConnector(self, point, r=4):
        fl=self.getFullList()
        for i in (0, -1):
            try:
                if point[0]-r<=fl[i][0]<=point[0]+r and point[1]-r<=fl[i][1]<=point[1]+r:
                    return (i, fl[i])
            except:
                print(fl)
                raise
        return None

    # ------------------------------------------------------------------
    # Fügt einen neuen Eckpunkt an der Koordinate point hinter dem
    # index_before ein.
    def insertCornerMarkAfterIndex(self, point, index_before):
        self.line.insert(index_before, point)

    # ------------------------------------------------------------------
    # Löscht den Eckpunkt auf index.
    def removeCornerMarkOnIndex(self, index):
        self.line.pop(index-1)

    # ------------------------------------------------------------------
    # Ändert die Koordinate des index'ten Eckpunktes auf new_xy.
    def moveAbsCornerMarkOnIndex(self, new_xy, index):
        p=glb.pointOnRaster(new_xy)
        self.line[index]=p

    # ------------------------------------------------------------------
    # Ändert die Koordinate des index'ten Eckpunktes um delta_xy.
    def moveRelCornerMarkOnIndex(self, delta_xy, index):
        self.line[index]=(self.line[index][0]+delta_xy[0], self.line[index][1]+delta_xy[1])

    # ------------------------------------------------------------------
    # Liefert True, wenn die Linie (p0, p1) den Punkt pp mit dem
    # "Radius" bzw. der Unschärfe r schneidet.
    def lineSquareIntersect(self, p0, p1, pp, r=4):
        p0_x, p0_y=p0
        p1_x, p1_y=p1
        pp_x, pp_y=pp
        sl=[  ((pp_x-r, pp_y-r), (pp_x+r, pp_y+r)),   # \
              ((pp_x-r, pp_y+r), (pp_x+r, pp_y-r)) ]  # /
        for ((p2x, p2y), (p3x, p3y)) in sl:
            if self.lineIntersection((p0_x, p0_y), (p1_x, p1_y), (p2x, p2y), (p3x, p3y)):
                return(True)
        return(False)

    # ------------------------------------------------------------------
    # Liefert True, wenn sich die Linien (A, B) und (C, D) schneiden.
    def lineIntersection(self, A, B, C, D):
        return(self.lineIntersectionPoint(A, B, C, D) is not None)

    # ------------------------------------------------------------------
    # Liefert den Schnittpunkt der Linien (A, B) mit (C, D) oder None,
    # wenn sich die Linien nicht schneiden.
    # https://stackoverflow.com/a/1968345/3588613 (in den Kommentaren)
    # ...mit einem zusätzlichen float(), damit er auch bei Übergabe von
    # Integers noch korrekte Ergebnisse liefert.
    def lineIntersectionPoint(self, A, B, C, D): 
        Bx_Ax=B[0]-A[0]
        By_Ay=B[1]-A[1] 
        Dx_Cx=D[0]-C[0] 
        Dy_Cy=D[1]-C[1] 
        determinant=float(-Dx_Cx*By_Ay + Bx_Ax*Dy_Cy) 
        if abs(determinant)<1e-20: 
            return None
        s=(-By_Ay*(A[0]-C[0]) + Bx_Ax*(A[1]-C[1]))/determinant 
        t=( Dx_Cx*(A[1]-C[1]) - Dy_Cy*(A[0]-C[0]))/determinant 
        if s>=0 and s<=1 and t>=0 and t<=1: 
            return (A[0]+(t*Bx_Ax), A[1]+(t*By_Ay)) 
        return None

    # ------------------------------------------------------------------
    # Liefert ein Tupel aus zwei Strings. Der erste String enthält die
    # (statische) Relations-Deklarationen, der zweite String enthält die
    # (variablen) Positionsdaten.
    def write(self):
        strg_d="RELATION %s %s-%s %s %s\n"%(self.name, self.rel_type[0], self.rel_type[1], self.start_col, self.end_col)
        strg_p="RELPOS %s %s%s "%(self.name, self.border_start, self.border_end)
        strg2=""
        for pos in self.line:
            strg2+=" (%s,%s)"%(pos[0], pos[1])
        strg_p+=strg2[1:]+"\n"
        return strg_d, strg_p
