#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os
import sys

from Table    import Table
from Relation import Relation
import globalStuff as glb

# ----------------------------------------------------------------------
# Operationen auf einer *.ERMzilla-Datei.
#
# gültige Keywords innerhalb der Datei:
#   #                                                                   - Kommentar
#   NOTE         note_name ...beliebiger Text...                        - Definition eines im ERM anzuzeigenden Kommentars
#   TABLE        table_name                                             - Start und Benennung einer Tabelle
#   "  "         column_name colum_type column_attributes               - Definition einer Tabellen-Spalte
#   TABEND                                                              - kennzeichnet das Ende einer inline-Tabelle (und ist mittlerweile eigentlich überflüssig)
#   TABCOLOR     table_name color                                       - optional (Hintergrund-Farbe des Headers einstellen)
#   TABPOS       table_name (x,y)                                       - optional (Positionsangaben für eine Tabelle oder einen Kommentar)
#   RELATION     rel_name rel_dir tab1.col tab2.col                     - Definition einer informativen Relation
#   CONSTRAINT   rel_name rel_dir tab1.col tab2.col                     - Definition einer echten Relation (die FOREIGN_KEY-tauglich ist)
#   RELPOS       rel_name border1+border2 {(x,y)}                       - optional (Positionsangaben für eine Relation)
#   IMPORT_TABLE file_name                                              - Tabellen-Definition aus Datei lesen (mit Pfad relativ zur geladenen Datei)
#   ZOOMLVL      raster                                                 - der zuletzt eingestellte Wert für raster
class FileOps():
    def __init__(self, tables, relations):
        self.tables=tables
        self.relations=relations
        self.filename=None
        self.dirname=None
        self.relation_name_generator_last=0
        self.raster=None
        self.log=[]
        self.valid_keywords={"NOTE", "TABLE", "TABEND", "TABCOLOR", "TABPOS", "RELATION", "CONSTRAINT", "RELPOS", "IMPORT_TABLE", "ZOOMLVL"}

    # ------------------------------------------------------------------
    # Datei laden.
    def load(self, filename):
        if not os.path.isfile(filename):
            self.log.append("Error! File not found")
            return
        self.filename=filename
        self.dirname=os.path.dirname(filename)
        rct=self.__loadTables(self.filename)
        if rct: rcr=self.__loadRelations(self.filename)                 # ohne Tabellen brauchts auch keine Relationen
        else:   rcr=False
        if self.raster is not None:
            glb.raster=self.raster
        return rct and rcr                                              # True, wenn weder Tabellen noch Relationen kritische Fehler hatten

    # ------------------------------------------------------------------
    # Liest die Tabellen-Definitionen aus der Datei filename.
    # Beim Keyword IMPORT_TABLE ruft sich diese Funtion selbst auf.
    # In diesem Fall ist imported_filename zu übergeben, damit keine
    # Endlos-Rekurstion entstehen kann.
    def __loadTables(self, filename, imported_filename=None):
        with open(filename, "rt") as fl:
            ln_cnt=0
            table_name=None
            cols=[]
            for ln in fl:
                ln_cnt+=1
                if ln.strip()=="" or ln.startswith("#"):                # Leerzeilen und Kommentare überlesen
                    continue
                lns=ln.split()
                if ln.startswith("ZOOMLVL "):
                    if table_name:
                        t_tmp=Table(table_name, cols, (0,0), imported_filename)
                        self.tables.append(t_tmp)
                        table_name=None
                        cols=[]
                    if len(lns)!=2:
                        self.log.append("Warning! Illegal ZOOMLVL in line %d"%(ln_cnt,))
                        continue
                    try:
                        self.raster=int(lns[1])
                    except:
                        self.raster=None
                        self.log.append("Warning! Illegal ZOOMLVL in line %d"%(ln_cnt,))
                        continue
                elif ln.startswith("NOTE "):
                    if table_name:
                        t_tmp=Table(table_name, cols, (0,0), imported_filename)
                        self.tables.append(t_tmp)
                        table_name=None
                        cols=[]
                    if len(lns)<3:
                        self.log.append("Warning! Strange NOTE in line %d"%(ln_cnt,))
                        continue
                    _, note_name, comment=ln.split(" ", 2)
                    if self.__getTable(note_name):
                        self.log.append("Error! duplicate table/note declaration in line %d"%(ln_cnt,))
                        return False
                    t_tmp=Table(note_name, [], (0,0), comment=comment)
                    self.tables.append(t_tmp)
                elif ln.startswith("TABLE "):
                    if table_name:
                        t_tmp=Table(table_name, cols, (0,0), imported_filename)
                        self.tables.append(t_tmp)
                        table_name=None
                        cols=[]
                    if len(lns)==2:
                        table_name=lns[1]
                    elif len(lns)>2:
                        table_name=lns[1]
                        self.log.append("Warning! Too many elements in line %d"%(ln_cnt,))
                    else:
                        self.log.append("Error! table_name missing in line %d"%(ln_cnt,))
                        return False
                    if self.__getTable(table_name):
                        self.log.append("Error! duplicate table declaration in line %d"%(ln_cnt,))
                        return False
                elif ln.startswith("  "):
                    if not table_name:
                        self.log.append("Warning! Column outside table in line %d"%(ln_cnt,))
                        continue
                    if len(lns)==3:
                        cols.append(tuple(lns))
                    elif len(lns)==2:
                        cols.append((lns[0], lns[1], ""))
                    elif len(lns)>3:
                        cols.append((lns[0], lns[1], lns[2]))
                        self.log.append("Warning! Too many elements in line %d"%(ln_cnt,))
                    else:
                        self.log.append("Error! Attribute[s] missing in line %d"%(ln_cnt,))
                        return False
                elif ln.strip()=="TABEND":
                    if not table_name:
                        self.log.append("Warning! TABEND outside table in line %d"%(ln_cnt,))
                        continue
                    t_tmp=Table(table_name, cols, (0,0), imported_filename)
                    self.tables.append(t_tmp)
                    table_name=None
                    cols=[]
                elif ln.startswith("TABCOLOR "):
                    if table_name:
                        t_tmp=Table(table_name, cols, (0,0))
                        self.tables.append(t_tmp)
                        table_name=None
                        cols=[]
                    if len(lns)<3:
                        self.log.append("Error! Attribute[s] missing in line %d"%(ln_cnt,))
                        return False
                    elif len(lns)>3:
                        self.log.append("Warning! Too many elements in line %d"%(ln_cnt,))
                    tn=self.__getTable(lns[1])
                    if not tn:
                        self.log.append("Error! Unknown table_name in line %d"%(ln_cnt,))
                        return False
                    if glb.testColor(lns[2]):
                        tn.header_color=lns[2]
                    else:
                        self.log.append("Warning! Strange color detected in line %d"%(ln_cnt,))
                elif ln.startswith("TABPOS ") and not imported_filename:
                    if table_name:
                        t_tmp=Table(table_name, cols, (0,0))
                        self.tables.append(t_tmp)
                        table_name=None
                        cols=[]
                    if len(lns)<3:
                        self.log.append("Error! Attribute[s] missing in line %d"%(ln_cnt,))
                        return False
                    elif len(lns)>3:
                        self.log.append("Warning! Too many elements in line %d"%(ln_cnt,))
                    tn=self.__getTable(lns[1])
                    if not tn:
                        self.log.append("Warning! Unknown table_name in line %d"%(ln_cnt,))
                        continue
                    pos=self.__parseTuple(lns[2])
                    if pos:
                        tn.pos=pos
                    else:
                        self.log.append("Error! Invalid coordinate in line %d"%(ln_cnt,))
                        return False
                elif ln.startswith("IMPORT_TABLE ") and not imported_filename:
                    if table_name:
                        t_tmp=Table(table_name, cols, (0,0))
                        self.tables.append(t_tmp)
                        table_name=None
                        cols=[]
                    if len(lns)>=2:
                        if len(lns)>2:
                            self.log.append("Warning! Too many elements in line %d"%(ln_cnt,))
                        fn=lns[1]
                        if not os.path.isfile(fn):
                            fn=os.path.join(self.dirname, lns[1])
                        if os.path.isfile(fn):
                            lb=len(self.log)
                            rc=self.__loadTables(fn, imported_filename=lns[1])
                            if len(self.log)!=lb:
                                self.log.append("Info! last %d messages apply to IMPORT_TABLE in line %d"%(len(self.log)-lb, ln_cnt))
                            if not rc:
                                return False
                        else:
                            self.log.append("Error! File not found in line %d / %s"%(ln_cnt, lns[1]))
                            return False
                    else:
                        self.log.append("Error! Filename missing in line %d"%(ln_cnt,))
                        return False
                else:
                    if lns[0] not in self.valid_keywords:
                        self.log.append("Warning! Unknown keyword in line %d"%(ln_cnt,))
        if table_name:
            t_tmp=Table(table_name, cols, (0,0), imported_filename)
            self.tables.append(t_tmp)
        return True

    # ------------------------------------------------------------------
    # Liest die Relations-Definitionen aus der Datei filename.
    def __loadRelations(self, filename):
        with open(filename, "rt") as fl:
            ln_cnt=0
            relation_names=set()
            for ln in fl:
                ln_cnt+=1
                if ln.strip()=="" or ln.startswith("#"):                # Leerzeilen und Kommentare überlesen
                    continue
                lns=ln.split()
                if ln.startswith("RELATION ") or ln.startswith("CONSTRAINT "):      #   RELATION     rel_name rel_dir tab1.col tab2.col
                    if len(lns)<5:
                        self.log.append("Error! Attribute[s] missing in line %d"%(ln_cnt,))
                        return False
                    elif len(lns)>5:
                        self.log.append("Warning! Too many elements in line %d"%(ln_cnt,))
                        _, rel_name, rel_dir, tab1_col, tab2_col=lns[:5]
                    else:
                        _, rel_name, rel_dir, tab1_col, tab2_col=lns
                    if rel_name=="*":
                        rel_name=self.__relation_name_generator()
                    if len(rel_name)<3:
                        self.log.append("Warning! rel_name too short in line %d"%(ln_cnt,))
                    if not rel_dir.find("-")>0:
                        self.log.append("Error! invalid rel_dir in line %d"%(ln_cnt,))
                        return False
                    if not self.__isValidTabCol(tab1_col):
                        self.log.append("Error! invalid tab1.col in line %d"%(ln_cnt,))
                        return False
                    if not self.__isValidTabCol(tab2_col):
                        self.log.append("Error! invalid tab2.col in line %d"%(ln_cnt,))
                        return False
                    if rel_name in relation_names:
                        self.log.append("Error! duplicate rel_name in line %d"%(ln_cnt,))
                        return False
                    relation_names.add(rel_name)
                    if ln.startswith("CONSTRAINT "):
                        r_tmp=Relation(self.tables, rel_name, rel_dir, tab1_col, tab2_col, is_constraint=True)
                    else:
                        r_tmp=Relation(self.tables, rel_name, rel_dir, tab1_col, tab2_col, is_constraint=False)
                    self.relations.append(r_tmp)
                elif ln.startswith("RELPOS "):                          #   RELPOS       rel_name border1+border2 {(x,y)}
                    if len(lns)<3:
                        self.log.append("Error! Attribute[s] missing in line %d"%(ln_cnt,))
                        continue
                    if lns[1] not in relation_names:
                        self.log.append("Error! Unknown rel_name in line %d"%(ln_cnt,))
                        continue
                    if lns[2] not in ("LL", "LR", "RL", "RR"):
                        self.log.append("Error! invalid border in line %d"%(ln_cnt,))
                        continue
                    borders=lns[2]
                    pl=[]
                    for p in lns[3:]:
                        pt=self.__parseTuple(p)
                        if not pt:
                            self.log.append("Error! Invalid coordinate in line %d"%(ln_cnt,))
                            continue
                        pl.append(pt)
                    r=self.__getRelation(lns[1])
                    if r:
                        if borders:
                            r.border_start=borders[0]
                            r.border_end=borders[1]
                        r.line=pl
                    else:
                        # kann eigentlich nicht vorkommen....
                        self.log.append("Error! rel_name not found in line %d"%(ln_cnt,))
                        return False
        return True

    # ------------------------------------------------------------------
    # Liefert das Tabellen-Objekt mit dem Namen table_name oder None.
    def __getTable(self, table_name):
        for t in self.tables:
            if t.name==table_name:
                return t
        return None

    # ------------------------------------------------------------------
    # Liefert das Relations-Objekt mit den Namen relation_name oder
    # None.
    def __getRelation(self, relation_name):
        for r in self.relations:
            if r.name==relation_name:
                return r
        return None

    # ------------------------------------------------------------------
    # Liefert True, wenn tab_col bereits bekannt ist.
    def __isValidTabCol(self, tab_col):
        for t in self.tables:
            for c, _, _ in t.cols:
                if "%s.%s"%(t.name, c)==tab_col:
                    return True
        return False

    # ------------------------------------------------------------------
    # Zerlegt einen String in strg im Format "(111,222)" und liefert es
    # als Tupel (111, 222) zurück. Im Fehlerfall wird None geliefert.
    def __parseTuple(self, strg):
        if strg.startswith("(") and strg.endswith(")"):
            ss=strg.strip("(").strip(")").split(",")
            if len(ss)==2:
                try:
                    a=int(ss[0])
                    b=int(ss[1])
                    return (a, b)
                except ValueError:
                    return None
            else:
                return None
        else:
            return None

    # ------------------------------------------------------------------
    # Liefert einen (neuen) eindeutigen Relationsnamen.
    def __relation_name_generator(self):
        self.relation_name_generator_last+=1
        return "rel_%06d"%(self.relation_name_generator_last)
