#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ERMzilla
#
# Ein Programm zum Zeichnen von Entity Relationship Modellen.
#
# Detlev Ahlgrimm  14.11.2020 - 20.11.2020
#
# D.A.  21.11.2020  v1.02   Tests auf neueren wxPython-Versionen,
#                           Export vom ERMs als png zugefügt
# D.A.  22.11.2020  v1.03   Umzug der Border-Definition von RELATION nach RELPOS,
#                           Keyword TABCOL zugefügt
# D.A.  26.11.2020  v1.04   Tabellenabmessungen korrigiert bei langen Tabellennamen
# D.A.  28.11.2020  v1.05   leichte Umbauten, um die Tastatur zu aktivieren,
#                           sichtbarer Ausschnitt wird jetzt mit gedrückt gehaltener Strg-Taste bewegt,
#                           Markierung eines Ausschnitts führt zur Selektion der drunterliegenden Objekte,
#                           die Verschiebe-Aktion bezieht sich auf alle derzeit markierten Objekte
# D.A.  29.11.2020  v1.06   Aufgeräumt und Kommentare zugefügt
# D.A.  30.11.2020  v1.06a  bereits beim Aufziehen der Ausschnitts-Markierung die Objekte als markiert darstellen
# D.A.  06.12.2020  v1.07   "save" schreibt nur die dynamischen Daten neu,
#                           Auto-Positionierung von Tabellen auf pos=(0, 0)
# D.A.  13.12.2020  v1.08   UndoHistory zugefügt
# D.A.  14.12.2020  v1.09   Keyword CONSTRAINT zugefügt
# D.A.  14.12.2020  v1.10   Keyword NOTE zugefügt
# D.A.  16.12.2020  v1.11   Fenster-Größe/Position speicherbar, Align-ERM, ZoomIn/ZoomOut,
#                           Seitenwechsel eines [ggf. auswählbaren] TableConnectors
# D.A.  17.12.2020  v1.12   [wieder] nur echte Konstanten in Großbuchstaben, Footer im Snapshot,
#                           neues Keyword ZOOMLVL
# D.A.  23.12.2020  v1.13   diverse kleine Korrekturen, Auslagerung von SelectedObjects,
#                           DPI-Wert-Unabhängigkeit mittels dynamischer Font-Größen


# getestet mit folgenden Versionen von wxPython:
#   wx.version() = "2.8.12.1 (gtk2-unicode)"                (unter "Python 2.7.12")
#   wx.version() = '3.0.2.0 gtk3 (classic)'                 (unter "Python 2.7.17")
#   wx.version() = '4.1.0 gtk3 (phoenix) wxWidgets 3.1.4'   (unter "Python 3.8.5") ... dazu in allen Dateien den Shebang auf python3 geändert!


import os
import sys
import time
import datetime
import random
import wx

from Table       import Table
from Relation    import Relation
from FileOps     import FileOps
from UndoHistory import UndoHistory
from SelObj      import SelectedObjects
import globalStuff as glb

VERSION="v1.13"

tables=[]           # hier landen die Table()-Objekte
relations=[]        # hier landen die Relation()-Objekte
filename=""         # Dateiname der aktuell geladenen Datei

# ----------------------------------------------------------------------
# Eine kleine Klasse, um neue Dateien via Drag&Drop öffnen zu können.
# In der aufrufenden Klasse muss die Funktion fileDropped() existieren.
class FileDrop(wx.FileDropTarget):
    def __init__(self, window):
        wx.FileDropTarget.__init__(self)
        self.window=window
        self.Ctrl_pressed=False

    def OnDropFiles(self, x, y, filenames):
        self.window.fileDropped(filenames)
        return True                                                     # will "Phoenix" so haben - sonst gibts Warnings

    def OnDragOver(self, x, y, d):
        if d==wx.DragCopy:
            if wx.GetKeyState(wx.WXK_CONTROL):
                self.Ctrl_pressed=True
                return(wx.DragLink)
            self.Ctrl_pressed=False
            return(wx.DragCopy)
        return(d)


# ----------------------------------------------------------------------
#
class ERMzillaWindow(wx.Window):
    def __init__(self, parent):
        wx.Window.__init__(self, parent)
        self.parent=parent
        self.raster_to_font_size={}
        self.measureFontSize()
        self.InitUI()

    # ------------------------------------------------------------------
    # Fenter initialisieren
    def InitUI(self):
        self.ctrlIsDown=False                                           # True, wenn die Strg-Taste gedrückt ist
        self.shiftIsDown=False                                          # True, wenn die Shift-Taste gedrückt ist
        self.leftDownPos=None                                           # die Koordinate, an der die linke Maus-Taste geklickt wurde (nur während leftDown!=None)
        self.errors_in_file=None                                        # True, wenn die geladene Datei Fehler enthielt (zur Sperre von "save")
        self.lastPosForShift=None                                       # zur Move-Delta-Bestimmung während Ausschnitts-Verschiebungen
        self.selectionRect=None                                         # Auswahl-Rechteck (nur während leftDown!=None)
        self.ignore_lost_focus=False                                    # zur Abschaltung der Änderungsanzeige bei wx.PopupMenu()
        self.obj_counts=(0, 0, 0)                                       # nimmt die Zähler in der Statuszeile auf
        self.selObj=SelectedObjects(tables, relations)                  # Datenstruktur zur Aufnahme der markierten Objekte
        self.undo=UndoHistory(tables, relations)                        # Datenstruktur zur Verwaltung der Historie

        self.SetBackgroundStyle(wx.BG_STYLE_CUSTOM)
        self.bg_color="#FFFFFF"

        self.statusbar=self.parent.CreateStatusBar()
        self.statusbar.SetFieldsCount(6)
        self.statusbar.SetStatusStyles([wx.SB_NORMAL, wx.SB_NORMAL, wx.SB_RAISED, wx.SB_RAISED, wx.SB_RAISED, wx.SB_RAISED])
        self.statusbar.SetStatusWidths([0, 100, 80, 80, -1, 150])

        self.__createMenu()
        
        self.filename_dt=FileDrop(self)                                 # Drag&Drop von ermz-Dateien aktivieren
        self.SetDropTarget(self.filename_dt)

        self.Bind(wx.EVT_CONTEXT_MENU,  self.onContextMenu)
        self.Bind(wx.EVT_PAINT,         self.onPaint)
        self.Bind(wx.EVT_MOTION,        self.onMotion)
        self.Bind(wx.EVT_LEFT_UP,       self.onLeftUp)
        self.Bind(wx.EVT_LEFT_DOWN,     self.onLeftDown)
        self.Bind(wx.EVT_LEFT_DCLICK,   self.onLeftDclick)
        self.Bind(wx.EVT_MOUSEWHEEL,    self.onMouseWheel)
        self.Bind(wx.EVT_KEY_DOWN,      self.onKeyDown)
        self.Bind(wx.EVT_KEY_UP,        self.onKeyUp)
        self.Bind(wx.EVT_SET_FOCUS,     self.onGotFocus)
        self.Bind(wx.EVT_KILL_FOCUS,    self.onLostFocus)

        self.Centre()
        wx.CallLater(10, self.SetFocus)
        wx.CallLater(50, self.loadFile)

    # ------------------------------------------------------------------
    # Stellt ein Raster auf dem dc dar.
    def showRaster(self, dc):
        dc.SetPen(wx.Pen("#eeeeee", width=1, style=wx.SOLID))
        sx, sy=dc.GetSize()
        for x in range(int(sx/glb.raster)+1):
            dc.DrawLine(x*glb.raster, 0, x*glb.raster, sy)
        for y in range(int(sy/glb.raster)+1):
            dc.DrawLine(0, y*glb.raster, sx, y*glb.raster)
        dc.SetPen(wx.BLACK_PEN)

    # ------------------------------------------------------------------
    # Lädt die Datei gemäß der globalen Variable filename.
    def loadFile(self):
        if filename=="" or not os.path.isfile(filename):
            self.setFrameTitle()
            return
        glb.font_size=glb.FONT_SIZE_BASE
        glb.raster=glb.RASTER_BASE
        file_ops=FileOps(tables, relations)
        self.errors_in_file=not file_ops.load(filename)                 # ändert ggf. glb.raster
        glb.font_size=self.raster_to_font_size[glb.raster]
#        print("glb.font_size", glb.font_size)
        self.spreadTables()
        if len(file_ops.log)>0:
            lns=""
            for ln in file_ops.log:
                lns+=ln+"\n"
            wx.MessageBox(lns, "Warnings / Errors")
            file_ops.log=[]
        tc=nc=0
        for t in tables:
            if t.isNote():  nc+=1
            else:           tc+=1
        self.obj_counts=(tc, len(relations), nc)
        self.updateStatusbar()
        self.setFrameTitle()
        self.Refresh()

    # ------------------------------------------------------------------
    # Stellt den Dateinamen im Fenster-Header ein.
    def setFrameTitle(self):
        if len(tables)==0 and len(relations)==0:
            self.parent.SetTitle("ERMzilla %s  (empty or bad file)"%(VERSION,))
        else:
            self.parent.SetTitle("ERMzilla %s  (%s)"%(VERSION, filename))

    # ------------------------------------------------------------------
    # Versucht für alle Tabellen mit pos=(0, 0) einen freien Platz zu
    # finden und stellt diesen ein.
    def spreadTables(self):
        if "phoenix" in wx.version():
            sx, sy=self.parent.GetSize()
            bmp=wx.Bitmap(1, 1)                                         # "Phoenix" mag EmptyBitmap() nicht mehr
        else:
            sx, sy=self.parent.GetSizeTuple()
            bmp=wx.EmptyBitmap(1, 1)                                    # wxPython 3.x braucht ein Bitmap für GetTextExtent()
        sx=max(100, sx)
        sy=max(100, sy)
        memDC=wx.MemoryDC()
        memDC.SelectObject(bmp)
        set_tabs=[]
        for t in tables:
            t.calcSize(memDC)
            if t.pos!=(0, 0):
                set_tabs.append(t)
        for t in tables:
            if t in set_tabs:
                continue
            if t.pos==(0, 0):
                t.pos=(random.randint(0, sx-50), random.randint(0, sy-50))
            for _ in range(20):
                intersection_found=False
                for t2 in set_tabs:
                    if t2.hasIntersectionWithRectangle((t.pos, (t.pos[0]+t.size[0], t.pos[1]+t.size[1]))):
                        intersection_found=True
                        break
                if not intersection_found:
                    set_tabs.append(t)
                    break
                t.pos=(random.randint(0, sx-50), random.randint(0, sy-50))

    # ------------------------------------------------------------------
    # Löscht die internen Datenstrukturen für Tabellen und Relationen.
    def unloadFile(self):
        global relations, tables
        for r in relations:
            del r
        for t in tables:
            del t
        relations=[]
        tables=[]
        del self.selObj
        del self.undo
        self.selObj=SelectedObjects(tables, relations)
        self.undo=UndoHistory(tables, relations)

    # ------------------------------------------------------------------
    # Wird aus FileDrop aufgerufen, wenn ein oder mehrere DateiObjekte
    # auf dem Fenster "fallengelassen" wurden.
    def fileDropped(self, filenames):
        global filename
        if len(filenames)!=1:
            wx.MessageBox("dropping multiple files is not allowed", "Error")
            return
        self.unloadFile()
        filename=filenames[0]
        self.loadFile()
        self.bg_color="#FFFFFF"

    # ------------------------------------------------------------------
    # Darstellung der Tabellen und Relationen.
    def onPaint(self, event):
        self.dc=wx.AutoBufferedPaintDC(self)
        self.__drawDC(self.dc)
        self.updateStatusbar()

    # ------------------------------------------------------------------
    # Darstellung der Tabellen und Relationen auf dc.
    # Wird auch von saveSnapshot() mit einem MemoryDC aufgerufen.
    def __drawDC(self, dc):
        dc.SetBackground(wx.Brush(self.bg_color))
        dc.Clear()
        dc.SetTextForeground("#000000")
        if self.raster_mode:
            self.showRaster(dc)                                         # ggf. Raster darstellen

        for t in tables:                                                # Tabellen darstellen
            t.calcSize(dc)
            t.drawTable(dc, t in self.selObj.obj)
        dc.SetBrush(wx.Brush("WHITE", wx.SOLID))
        for r in relations:                                             # Relationen darstellen
            fl=r.getFullList()
            if r.isConstraint():
                dc.SetPen(wx.Pen("#000000", 1))
            else:
                dc.SetPen(wx.Pen("#F7BE3C", 1))
            dc.DrawLines(fl)
            sx, sy=dc.GetTextExtent(r.rel_type[0])
            if r.border_start=="L":
                dc.DrawText(r.rel_type[0], fl[0][0]-5-sx, fl[0][1]-sy)
            else:
                dc.DrawText(r.rel_type[0], fl[0][0]+5, fl[0][1]-sy)
            sx, sy=dc.GetTextExtent(r.rel_type[1])
            if r.border_end=="L":
                dc.DrawText(r.rel_type[1], fl[-1][0]-5-sx, fl[-1][1]-sy)
            else:
                dc.DrawText(r.rel_type[1], fl[-1][0]+5, fl[-1][1]-sy)

            if self.rel_points_mode:                                    # ggf. Eckpunkte darstellen
                for p in fl:
                    dc.DrawCircle(p[0], p[1], 2)
            if r in self.selObj.obj:                                    # markierte Eckpunkte werden immer dargestellt
                data=self.selObj.obj[r]
                dc.SetPen(wx.Pen("#ff0000", 3))
                for i, p in enumerate(fl[1:]):
                    if i in data:
                        dc.DrawCircle(p[0], p[1], 2)
                dc.SetPen(wx.Pen("#000000", 1))
        if self.selectionRect:                                          # ggf. das Markierungs-Rechteck darstellen
            (x1, y1), (x2, y2)=self.selectionRect
            pen=dc.GetPen()
            dc.SetPen(wx.Pen("#0000ff"))
            dc.DrawLines([(x1,y1), (x2,y1), (x2,y2), (x1,y2), (x1,y1)])
            dc.SetPen(pen)

    # ------------------------------------------------------------------
    # Liefert True, wenn pos auf einem markierten Objekt/Element liegt.
    def __isClickPosOnMarkedObject(self, pos):
        for r in relations:
            if r in self.selObj.obj:                                    # markierte Relation gefunden
                i=r.isPointOnCornerMark(pos)
                if i in self.selObj.obj[r]:
                    return True                                         # pos liegt über einem markierten Eckpunkt dieser Relation
        for t in tables:
            if t in self.selObj.obj:
                if t.isUnderXY(pos):
                    return True                                         # pos liegt über einer markierten Tabelle
        return False

    # ------------------------------------------------------------------
    # Linke Maustaste wurde geklickt.
    # Bei gedrückter Strg-Taste wird das gesamte ERM verschoben.
    # Bei 
    def onLeftDown(self, event):
        #print("onLeftDown")
        self.undo.prepare()
        self.leftDownPos=event.GetPosition()
        if self.ctrlIsDown:
            self.lastPosForShift=self.leftDownPos                       # Verschiebung des Bildschim-Inhaltes vorbereiten
            return

        if self.selObj.hasElements() and not self.shiftIsDown:          # es gibt markierte Elemente, die Shift-Taste ist nicht gedrückt...
            #print("self.shiftIsDown", self.shiftIsDown)
            if self.__isClickPosOnMarkedObject(self.leftDownPos):       # ...und der Klick war auf einem markierten Element
                self.selObj.initMoveable(self.leftDownPos)              # also sollen jetzt Objekte verschoben werden.
                return

        if self.__testForClickOnTableConnectorAndSwapBorder():          # ggf. Seitenwechsel einer Relation ausführen
            return

        # testen, ob ein Objekt angeklickt/markiert wurde
        sel_done=False
        for r in relations:                                             # wurde ein Eckpunkt angeklickt?
            cm=r.isPointOnCornerMark(self.leftDownPos)
            if cm is not None:                                          # Eckpunkt wurde angeklickt
                sel_done=True
                if self.shiftIsDown:                                    # Shift-Taste ist gedrückt
                    if self.selObj.find(r, cm):                         # wenn der Eckpunkt bereits markiert war...
                        self.selObj.remove(r, cm)                       # ...Markierung weg
                        break
                    else:
                        self.selObj.add(r, cm)                          # ...sonst Markierung setzen
                        break
                else:
                    self.selObj.set(r, cm)                              # Eckpunkt wurde ohne Shift geklickt -> Einzel-Markierung setzen
                    break
        if not sel_done:                                                # wenn bei den Relationen nix gefunden wurde
            for t in tables:                                            # wurde eine Tabelle angeklickt?
                if t.isUnderXY(self.leftDownPos):                       # Tabelle wurde angeklickt
                    sel_done=True
                    if self.shiftIsDown:                                # Shift-Taste ist gedrückt
                        if self.selObj.find(t):                         # wenn die Tabelle bereits markiert war...
                            self.selObj.remove(t)                       # ...Markierung weg
                            break
                        else:
                            self.selObj.add(t)                          # ...sonst Markierung setzen
                            break
                    else:
                        self.selObj.set(t)                              # Tabelle wurde ohne Shift geklickt -> Einzel-Markierung setzen
                        break
        #if not self.shiftIsDown:
        self.selObj.initMoveable(self.leftDownPos)

        if not sel_done and not self.shiftIsDown:                       # weder Eckpunkt noch Tabelle wurde (ohne Shift) angeklickt
            self.selObj.clearAll()                                      # also gibts jetzt keine markierten Objekte mehr

    # ------------------------------------------------------------------
    # Prüft, ob die zuletzt angeklicke Maus-Position über einem
    # TableConnector liegt und liefert False, wenn nicht.
    # Wenn -anderenfalls- nur eine Relation andockt, wird dessen Seite
    # gewechselt. Bei mehr als einer angedockten Relation wird ein
    # PopUp-Menü zur Auswahl einer Relation geöffnent und nach Auswahl
    # nur genau dessen Seite gewechselt.
    def __testForClickOnTableConnectorAndSwapBorder(self):
        self.rel_border_swap={}
        cnt=0
        for r in relations:
            tc=r.isPointOnTableConnector(self.leftDownPos)
            if tc:
                self.rel_border_swap.update({cnt:(r, tc[0])})
                cnt+=1
        if len(self.rel_border_swap)==0:
            return False
        if len(self.rel_border_swap)==1:
            if self.rel_border_swap[0][1]==0:   self.rel_border_swap[0][0].swapStartBorder()
            else:                               self.rel_border_swap[0][0].swapEndBorder()
            self.undo.trackModifications()
        elif len(self.rel_border_swap)>1:
            self.ignore_lost_focus=True
            self.__createRelationSelectionMenu()
            self.PopupMenu(self.rs_menue)
            self.ignore_lost_focus=False
        self.Refresh()
        self.undo.prepare()
        self.leftDownPos=None
        return True

    # ------------------------------------------------------------------
    # Erstellt ein Kontextmenü mit den Relationsnamen gemäß
    # self.rel_border_swap und bindet alle an die Funktion __rel_sel().
    def __createRelationSelectionMenu(self):
        self.rs_menue=wx.Menu()
        for i, rel_name in self.rel_border_swap.items():
            self.rs_menue.Append(10+i, rel_name[0].name)
            self.Bind(wx.EVT_MENU, self.__rel_sel, id=10+i)

    # ------------------------------------------------------------------
    # Wird bei Auswahl eines zum Seitenwechsel vorgesehenen
    # Relationsnamens aufgerufen und wechselt dessen Seite.
    def __rel_sel(self, event):
        i=event.GetId()-10
        if self.rel_border_swap[i][1]==0:   self.rel_border_swap[i][0].swapStartBorder()
        else:                               self.rel_border_swap[i][0].swapEndBorder()
        self.undo.trackModifications()

    # ------------------------------------------------------------------
    # Linke Maustaste wurde losgelassen.
    def onLeftUp(self, event):
        self.leftDownPos=None
        if self.selectionRect:                                          # wenn zuvor ein Bereich ausgewählt wurde...
            self.__markElementsUnderRectAsSelected()                    # ...die Objekte als "markiert" setzen
        self.selectionRect=None
        for t in tables:                                                # Tabellen am Raster ausrichten
            t.pos=glb.pointOnRaster(t.pos)
        for r in relations:                                             # und die Relationen ebenso
            for i, p in enumerate(r.line):
                r.line[i]=glb.pointOnRaster(p)
        #print("onLeftUp")
        self.undo.trackModifications()
        self.Refresh()

    # ------------------------------------------------------------------
    # Stellt alle Objekte unterhalb von self.selectionRect als
    # "markiert" ein.
    def __markElementsUnderRectAsSelected(self):
        self.selObj.clearAll()
        nr=glb.normalizeRectangle(self.selectionRect)
        for r in relations:
            ip=r.getCornerMarksUnderRectangle(nr)
            if ip:
                self.selObj.add_many(r, ip)
        for t in tables:
            if t.hasIntersectionWithRectangle(nr):
                self.selObj.add(t)

    # ------------------------------------------------------------------
    # Der MausCursor wird über den DC bewegt.
    def onMotion(self, event):
        cur_pos=event.GetPosition()
        self.statusbar.SetStatusText("pos: %s"%(cur_pos,), 5)
        if self.leftDownPos:
            if self.ctrlIsDown:                                         # bei gedrückter Strg-Taste soll das gesamte ERM bewegt werden
                scroll_distXY=cur_pos-self.lastPosForShift
                self.lastPosForShift=cur_pos
                self.__moveERM(scroll_distXY)
            elif self.selObj.isReadyToBeMoved():                        # wenn markierte Objekte existieren -> diese bewegen
                x, y=cur_pos
                for t in tables:
                    if t in self.selObj.obj:
                        x1=x-(self.selObj.drag_start_mouse_pos[0]-self.selObj.base_pos[t][0])
                        y1=y-(self.selObj.drag_start_mouse_pos[1]-self.selObj.base_pos[t][1])
                        t.moveAbs((x1, y1))
                for r in relations:
                    if r in self.selObj.obj:
                        for i in self.selObj.obj[r]:
                            x1=x-(self.selObj.drag_start_mouse_pos[0]-self.selObj.base_pos[r][i][0])
                            y1=y-(self.selObj.drag_start_mouse_pos[1]-self.selObj.base_pos[r][i][1])
                            r.moveAbsCornerMarkOnIndex((x1, y1), i)
            else:                                                       # ...ansonsten soll ein Bereich markiert werden
                self.selectionRect=(self.leftDownPos, cur_pos)
                self.__markElementsUnderRectAsSelected()
            self.Refresh()

    # ------------------------------------------------------------------
    # Verschiebt das gesamte ERM um delta Pixel.
    def __moveERM(self, delta):
        for t in tables:
            t.moveRel(delta)
        for r in relations:
            for i, p in enumerate(r.line):
                r.line[i]=(p[0]+delta[0], p[1]+delta[1])

    # ------------------------------------------------------------------
    # Linke Maustaste wurde doppel-geklickt.
    # Doppelklick auf Relations-Linie fügt einen neuen Eckpunkt ein.
    # Doppelklick auf Relations-Eckpunkt löscht den Eckpunkt.
    # Doppelklick führt zur Sequenz: onLeftDown + onLeftUp + onLeftDclick + onLeftUp
    # Daher braucht es hier immer einen undo.prepare().
    def onLeftDclick(self, event):
        #print("onLeftDclick")
        if self.ctrlIsDown or self.shiftIsDown:
            self.undo.prepare()
            return
        x, y=event.GetPosition()
        for r in relations:
            a=r.isPointOnThisRelation((x, y))
            if a:
                self.undo.prepare()
                if a[0]=="LN":
                    r.insertCornerMarkAfterIndex((x, y), a[1])
                    self.Refresh()
                    break
                elif a[0]=="CM":
                    r.removeCornerMarkOnIndex(a[1])
                    self.selObj.clearAll()
                    self.Refresh()
                    break

    # ------------------------------------------------------------------
    # ZoomIn / ZoomOut mit dem Scrollrad.
    def onMouseWheel(self, event):
        pa=event.GetPosition()
        dx=dy=0
        if event.GetWheelRotation()<0:      # Wheel runter
            if glb.raster>glb.RASTER_MIN:
                self.changeRaster(glb.raster-1)
                dx=int(pa[0]/float(glb.raster))
                dy=int(pa[1]/float(glb.raster))
                glb.raster-=1
                glb.font_size=self.raster_to_font_size[glb.raster]
        else:
            if glb.raster<glb.RASTER_MAX:
                self.changeRaster(glb.raster+1)
                dx=-int(pa[0]/float(glb.raster))
                dy=-int(pa[1]/float(glb.raster))
                glb.raster+=1
                glb.font_size=self.raster_to_font_size[glb.raster]
#        print("glb.raster", glb.raster, glb.font_size)
        dx, dy=glb.pointOnRaster((dx, dy))
        for t in tables:
            t.recalcSize()
        self.__moveERM((dx, dy))
        self.undo.shiftBy(dx, dy)
        self.updateStatusbar()
        self.Refresh()

    # ------------------------------------------------------------------
    # Richtet alle Positionen gemäß new_raster neu aus.
    def changeRaster(self, new_raster):
        for t in tables:
            t.changeRaster(new_raster)
        for r in relations:
            r.changeRaster(new_raster)
        self.undo.changeRaster(new_raster)

    # ------------------------------------------------------------------
    # Eine Taste wurde betätigt.
    def onKeyDown(self, event):
        #print("onKeyDown", event.GetKeyCode())
        if event.GetKeyCode()==wx.WXK_CONTROL:
            self.ctrlIsDown=True
        if event.GetKeyCode()==wx.WXK_SHIFT:
            self.shiftIsDown=True
        if event.GetKeyCode()==ord("Z") and self.ctrlIsDown:
            self.undo.lastModification()
            self.selObj.clearAll()
            self.Refresh()

    # ------------------------------------------------------------------
    # Eine Taste wurde losgelassen.
    def onKeyUp(self, event):
        #print("onKeyUp", event.GetKeyCode())
        if event.GetKeyCode()==wx.WXK_CONTROL:
            self.ctrlIsDown=False
        if event.GetKeyCode()==wx.WXK_SHIFT:
            self.shiftIsDown=False

    # ------------------------------------------------------------------
    # Stellt das PopUp-Menü dar.
    def onContextMenu(self, event):
        self.ignore_lost_focus=True
        if os.path.isfile(filename) and not self.errors_in_file:
            self.menue.Enable(100, True)
        else:
            self.menue.Enable(100, False)
        self.PopupMenu(self.menue)
        self.Refresh()
        self.ignore_lost_focus=False

    # ------------------------------------------------------------------
    # Erstellt das PopUp-Menü.
    def __createMenu(self):
        self.raster_mode=True
        self.rel_points_mode=True
        self.menue=wx.Menu()
        self.menue.Append(80, 'export snapshot')
        self.menue.Append(90, 'reload')
        self.menue.Append(100, 'save')
        self.menue.AppendSeparator()
        self.menue.Append(200, 'undo')
        self.menue.Append(300, 'align')
        self.menue.AppendSeparator()
        self.menue.Append(500, 'raster on/off', "", True)
        self.menue.Append(510, 'relation points on/off', "", True)
        self.menue.AppendSeparator()
        self.menue.Append(600, 'save settings')
        self.menue.AppendSeparator()
        self.menue.Append(900, 'about')
        self.Bind(wx.EVT_MENU, self.men_export_snapshot, id=80)
        self.Bind(wx.EVT_MENU, self.men_reload, id=90)
        self.Bind(wx.EVT_MENU, self.men_save, id=100)
        self.Bind(wx.EVT_MENU, self.men_undo, id=200)
        self.Bind(wx.EVT_MENU, self.men_align, id=300)
        self.Bind(wx.EVT_MENU, self.men_raster, id=500)
        self.Bind(wx.EVT_MENU, self.men_rel_points, id=510)
        if not os.path.isfile(filename):
            self.menue.Enable(100, False)
        self.Bind(wx.EVT_MENU, self.men_saveConfig, id=600)
        self.Bind(wx.EVT_MENU, self.men_aboutWin, id=900)

        fc=wx.FileConfig(localFilename=glb.CFGFILE)
        self.raster_mode=fc.ReadInt("raster_mode", True)
        self.rel_points_mode=fc.ReadInt("rel_points_mode", True)
        del fc
        self.menue.Check(500, self.raster_mode)
        self.menue.Check(510, self.rel_points_mode)

    # ------------------------------------------------------------------
    # Menü - export snapshot
    # Exportiert das ERM als PNG-Datei. Die Bild-Datei landet im selben
    # Verzeichnis wie die ERM-Datei - die Datei-Endung wird durch
    # png ersetzt.
    def men_export_snapshot(self, event):
        self.selObj.clearAll()
        orm=self.raster_mode
        ocm=self.rel_points_mode
        self.raster_mode=False
        self.rel_points_mode=False
        se=os.path.splitext(os.path.abspath(filename))
        fn=se[0]+".png"
        self.saveSnapshot(fn)
        self.raster_mode=orm
        self.rel_points_mode=ocm

    # ------------------------------------------------------------------
    # Menü - reload
    def men_reload(self, event):
        self.unloadFile()
        self.loadFile()
        self.undo.reset()
        self.Refresh()

    # ------------------------------------------------------------------
    # Menü - save
    def men_save(self, event):
        strg_d=""
        strg_p=""
        for t in tables:
            sd, sp=t.write()
            strg_d+=sd
            strg_p+=sp
            for r in relations:
                if r.start_col.split(".")[0]==t.name:
                    sd, sp=r.write()
                    strg_d+=sd
                    strg_p+=sp
        flos=""
        with open(filename, "rt") as fl:                                # Datei bis zum ersten TABPOS oder RELPOS laden 
            for ln in fl:
                if ln.strip()=="" or ln.startswith("#"):
                    flos+=ln
                    continue
                if ln.startswith("TABPOS ") or \
                    ln.startswith("RELPOS ") or \
                    ln.startswith("ZOOMLVL "):
                    break
                flos+=ln
        with open(filename, "wt") as fl:                                # Datei mit neuen TABPOS-/RELPOS-Daten schreiben
            fl.write(flos)
            fl.write("ZOOMLVL %d\n"%(glb.raster,))
            fl.write(strg_p)
        self.undo.reset()

    # ------------------------------------------------------------------
    # Menü - wie Strg-Z = letzte Verschiebe-Operation rückgängig machen.
    def men_undo(self, event):
        self.undo.lastModification()
        self.selObj.clearAll()
        self.Refresh()

    # ------------------------------------------------------------------
    # Menü - richtet das ERM so aus, dass die jeweils nördlichst und
    # westlichst gelegenen Objekte auf der Postion (RASTER, RASTER)
    # landen.
    def men_align(self, event):
        x1, y1, x2, y2=self.getBoundariesERM()
        dx=-x1+glb.raster
        dy=-y1+glb.raster
        self.undo.prepare()
        self.__moveERM((dx, dy))
        #self.undo.shiftBy(dx, dy)
        self.undo.trackModifications()

    # ------------------------------------------------------------------
    # Menü - Raster an/aus
    def men_raster(self, event):
        self.raster_mode=not self.raster_mode

    # ------------------------------------------------------------------
    # Menü - Darstellung der Eckpunkte an/aus
    def men_rel_points(self, event):
        self.rel_points_mode=not self.rel_points_mode

    # ------------------------------------------------------------------
    # Liefert die äußeren Abmessungen des ERMs
    def getBoundariesERM(self):
        min_x=min_y=max_x=max_y=None
        for t in tables:
            if min_x is None:   min_x=t.pos[0]
            else:               min_x=min(min_x, t.pos[0])
            if min_y is None:   min_y=t.pos[1]
            else:               min_y=min(min_y, t.pos[1])
            if max_x is None:   max_x=t.pos[0]+t.size[0]
            else:               max_x=max(max_x, t.pos[0]+t.size[0])
            if max_y is None:   max_y=t.pos[1]+t.size[1]
            else:               max_y=max(max_y, t.pos[1]+t.size[1])
        # der Test auf None kann hier entfallen, weil es keine Relation
        # ohne Table gibt
        for r in relations:
            fl=r.getFullList()
            for p in fl:
                min_x=min(min_x, p[0])
                min_y=min(min_y, p[1])
                max_x=max(max_x, p[0])
                max_y=max(max_y, p[1])
        return (min_x, min_y, max_x, max_y)

    # ------------------------------------------------------------------
    # Speichert das ERM als <filename>.png.
    def saveSnapshot(self, filename_snapshot):
        memDC=wx.MemoryDC()
        fnt=wx.Font(glb.FONT_SIZE_BASE, wx.FONTFAMILY_TELETYPE, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL)
        memDC.SetFont(fnt)
        txt1="%s - %s"%(os.path.basename(filename), datetime.datetime.now().strftime("%d.%m.%Y %H:%M"))
        txt2="ERMzilla %s"%(VERSION,)
        if "phoenix" in wx.version():
            bmp=wx.Bitmap(1, 1)                                         # "Phoenix" mag EmptyBitmap() nicht mehr
        else:
            bmp=wx.EmptyBitmap(1, 1)                                    # wxPython 3.x braucht ein Bitmap für GetTextExtent()
        memDC.SelectObject(bmp)
        wt, ht=memDC.GetTextExtent(txt2)

        x1, y1, x2, y2=self.getBoundariesERM()                          # Abmessungen des ERMs bestimmen
        w=x2-x1+2*glb.raster                                            # ganz links und rechts je ein leeres Raster
        h=y2-y1+2*glb.raster+ht+2                                       # ganz oben und unten je ein leeres Raster + Footer + Trennlinie

        if "phoenix" in wx.version():
            bmp=wx.Bitmap(w, h)                                         # "Phoenix" mag EmptyBitmap() nicht mehr
        else:
            bmp=wx.EmptyBitmap(w, h)
        
        memDC.SelectObject(bmp)

        dx=-x1+glb.raster
        dy=-y1+glb.raster

        if dx!=0 or dy!=0:                                              # wenn das ERM nicht schon auf (raster, raster) liegt...
            self.__moveERM((dx, dy))                                    # ...ERM ausrichten
        self.__drawDC(memDC)                                            # ERM darstellen
        if dx!=0 or dy!=0:                                              # wenn das ERM zuvor ausgerichtet wurde...
            self.__moveERM((-dx, -dy))                                  # ...zu vorheriger Position zurückkehren

        memDC.DrawLine(0, h-ht-2, w, h-ht-2)                            # Footer-Trennzeile
        memDC.SetFont(fnt)
        memDC.DrawText(txt1, 10, h-ht)
        memDC.DrawText(txt2, w-wt-10, h-ht)

        img=bmp.ConvertToImage()
        img.SaveFile(filename_snapshot, wx.BITMAP_TYPE_PNG)             # Bild-Datei speichern

    # ------------------------------------------------------------------
    # ERMzilla hat den Focus erhalten
    def onGotFocus(self, event):
        #print("onGotFocus")
        self.bg_color="#FFFFFF"

    # ------------------------------------------------------------------
    # ERMzilla hat den Focus verloren
    def onLostFocus(self, event):
        #print("onLostFocus")
        if self.ignore_lost_focus:
            return
        if self.undo.length()>0:
            self.bg_color="#FFCCCC"
            self.Refresh()

    # ------------------------------------------------------------------
    # Speichert Fenster-Größe und Position sowie ein paar Einstellungen.
    def men_saveConfig(self, event):
        fc=wx.FileConfig(localFilename=glb.CFGFILE)
        sp=self.parent.GetScreenPosition()
        if "phoenix" in wx.version():
            ss=self.parent.GetSize()
        else:
            ss=self.parent.GetSizeTuple()
        fc.WriteInt("pos_x",    sp[0])
        fc.WriteInt("pos_y",    sp[1])
        fc.WriteInt("size_x" ,  ss[0])
        fc.WriteInt("size_y" ,  ss[1])
        fc.WriteInt("raster_mode", self.menue.IsChecked(500))
        fc.WriteInt("rel_points_mode", self.menue.IsChecked(510))
        fc.Flush()

    # ------------------------------------------------------------------
    # About-Fenster öffnen.
    def men_aboutWin(self, event):
        info=wx.AboutDialogInfo()
        info.SetName("ERMzilla")
        info.SetVersion(VERSION)
        info.SetCopyright("D.A.  (12.2020)")
        info.SetDescription("Ein Tool zur Anzeige und Bearbeitung von ERMs")
        info.SetLicence("Dieses Programm ist freie Software gemaess GNU General Public License")
        info.AddDeveloper("Detlev Ahlgrimm")
        wx.AboutBox(info)

    # ------------------------------------------------------------------
    # Den Inhalt der Statusbar aktualisieren.
    def updateStatusbar(self):
        self.statusbar.SetStatusText("", 1)                             # Workaround für eine stabile Anzeige des Inhaltes
        self.statusbar.SetStatusText("t:%d, r:%d, n:%d"%self.obj_counts, 1)
        self.statusbar.SetStatusText("", 2)
        self.statusbar.SetStatusText("hist: %d"%(self.undo.length(),), 2)
        self.statusbar.SetStatusText("", 3)
        self.statusbar.SetStatusText("zoom: %d"%(glb.raster-glb.RASTER_BASE,), 3)

    # ------------------------------------------------------------------
    # Misst die Font-Größen und füllt das Dictionary
    # self.raster_to_font_size mit der Ralation "raster zu font_size".
    def measureFontSize(self):
        memDC=wx.MemoryDC()
        if "phoenix" in wx.version():
            bmp=wx.Bitmap(1, 1)                                         # "Phoenix" mag EmptyBitmap() nicht mehr
        else:
            bmp=wx.EmptyBitmap(1, 1)                                    # wxPython 3.x braucht ein Bitmap für GetTextExtent()
        memDC.SelectObject(bmp)
        font_size_to_raster={}
        for fs in range(0, glb.RASTER_MAX+10):
            fnt=wx.Font(fs, wx.FONTFAMILY_TELETYPE, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL)
            memDC.SetFont(fnt)
            wt, ht=memDC.GetTextExtent("-")
            font_size_to_raster.update({fs:max(wt, ht)})
        for r in range(glb.RASTER_MIN, glb.RASTER_MAX+1):
            for fs, sz in sorted(font_size_to_raster.items()):
                if sz<=r:
                    self.raster_to_font_size.update({r:fs})

# ----------------------------------------------------------------------
# 
class ERMzillaFrame(wx.Frame):
    def __init__(self, parent, pos=wx.DefaultPosition, size=wx.DefaultSize):
        style=wx.DEFAULT_FRAME_STYLE
        wx.Frame.__init__(self, None, wx.ID_ANY, pos=pos, size=size, style=style)
        self.panel=ERMzillaWindow(self)

# ----------------------------------------------------------------------
# 
if __name__=='__main__':
    if len(sys.argv)>1:
        ff=os.path.isfile(sys.argv[1])
        if ff:
            filename=sys.argv[1]

    app=wx.App(False)

    fc=wx.FileConfig(localFilename=glb.CFGFILE)
    spx=fc.ReadInt("pos_x", 100)
    spy=fc.ReadInt("pos_y", 100)
    ssx=fc.ReadInt("size_x", 1350)
    ssy=fc.ReadInt("size_y", 750)
    sp=(spx, spy) # (-1, -1) entspricht wx.DefaultPosition
    ss=(ssx, ssy) # (-1, -1) entspricht wx.DefaultSize
    del fc

    frame=ERMzillaFrame(None, pos=sp, size=ss)
    frame.Show(True)
    app.MainLoop()
