#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ###########################################################
# Zeigt ein mit "tun0moni.py" erzeugtes Tages-Logfile als
# Grafik an.
# Innerhalb der Grafik kann "gezoomt" werden.
# Auch kann die Grafik als PNG gespeichert werden.
# Ein Tages-Logfile wird per Drag&Drop geladen.
#
# D.A. 06.2014
# ###########################################################
# 09.06.2014  v1.0  erste Version
# 13.06.2014  v1.1  Zoom-Out in setPresentationAreaVariables() zugefügt
# 14.06.2014  v1.2  changeFile() über Cursortasten zugefügt
#                   OptionParser durch argparse ersetzt
# 15.06.2014  v1.21 kleine Änderung an setupArgParser() gemäß
#                   http://stackoverflow.com/a/24221850/3588613
# 16.06.2014  v1.22 in loadFile() nicht existente Datei abfangen
# 18.06.2014  v1.3  Prüfung auf valides LogFile eingebaut
#                   onMouseWheel() eingebaut
#                   intersectionPoint() und cutYToMax() eingebaut und damit eine saubere
#                   Verarbeitung von Kurven, die über die Skala hinaus gehen, erreicht
# 20.06.2014  v1.31 Anzeige der gesamt übertragenen Bytes eingebaut
#                   Korrektur in drawCurve() bzgl. Ende==23:59
# 22.06.2014  v1.4  Verarbeitung von in Archiven liegenden LogFiles eingebaut bzw. reingemurkst
# 23.06.2014  v1.41 Parameter -z und -c zugefügt
#                   getMaxBwForShownCurves() eingebaut mit der self.max_bw nun die aktuell
#                   angezeigten Kurven berücksichtigt
#

import wx
import time
import os
import sys
import argparse
import tarfile

CFGFILE=os.path.join(os.path.expanduser('~'), ".tun0graf.conf")
VERSION="1.41"

TIME_MIN=0
TIME_MAX=1439
BW_MAX=10737418240
BW_MIN=100

filename_g=""   # wird ggf. vom ArgParser gesetzt
snapname_g=""   # wird ggf. vom ArgParser gesetzt
curves_g=""     # wird ggf. vom ArgParser gesetzt

# ###########################################################
# Liefert "num" in der passendsten Größenangabe
def prettySize(num):
  for x in ['','KB','MB','GB', 'TB']:
    if num<1024.0:
      if x in ['', 'KB']:
        # Byte und Kilobyte ohne Nachkommastelle
        return("{0:3.0f} {1:s}".format(num, x))
      else:
        return("{0:3.1f} {1:s}".format(num, x))
    num/=1024.0


# ###########################################################
# Ein Tages-Logfile laden und als Dictionary mit einem Wert
# pro Minute zurückgeben. Wird "logFromTar" übergeben, wird
# darin ein bereits geladenes LogFile erwartet.
class DayLog():
  def __init__(self, filename, logFromTar=None):
    self.filename=filename  # Dateiname
    self.isValid=False      # auf True, wenn Datei korrekt geladen werden konnte
    self.errorReason=""     # String mit einem Fehlertext
    self.size=0             # aktuelle Dateigröße von self.filename
    self.maxr=0             # höchster Wert in Kurve (empfangen)
    self.maxs=0             # höchster Wert in Kurve (gesendet)
    self.sumr=0             # Summe der empfangenen Bytes der Kurve (/60)
    self.sums=0             # Summe der gesendeten Bytes der Kurve (/60)
    self.sumCache={}        # ein Cache für die bereits berechneten Summen
    self.data_dict={}       # die Kurve
    self.logFromTar=logFromTar  # Log kommt von außerhalb und muss nicht geladen werden
    self.__initDict()
    self.__loadDict()

  # ###########################################################
  # Liefert das Tages-Logfile mit einem Wert pro Minute als
  # Dictionary.
  def getDict(self):
    return(self.data_dict)

  # ###########################################################
  # Liefert die Maximalwerte als Tupel (received, sent).
  def getMaxTuple(self):
    return((self.maxr, self.maxs))

  # ###########################################################
  # Liefert die insgesamt übertragenen Bytes als Tupel
  # (received, sent).
  def getSumTuple(self):
    return((self.sumr*60, self.sums*60))

  # ###########################################################
  # Liefert True, wenn sich die Datei "self.filename" seit dem
  # Init oder letzten Aufruf von refresh() bzw. __loadFile()
  # geändert hat.
  # Erfolgt der Zugriff auf das Log-File via LAN auf ein
  # NFS-Share, sollte beim mount ein "-o lookupcache=none"
  # mitgegeben werden - andernfalls kann es schon mal 45 Sek.
  # dauern, bis die Änderung an der Datei erkannt wird.
  def hasChanged(self):
    return(self.size!=self.__getSize())

  # ###########################################################
  # Liefert True, wenn ein valides LogFile geladen ist - sonst
  # False.
  def isValidLogFile(self):
    return(self.isValid)

  # ###########################################################
  # Liefert den Grund als String, warum die Datei nicht valide
  # ist.
  def reasonForError(self):
    return(self.errorReason)

  # ###########################################################
  # Lädt ggf. neue Zeilen in "self.filename" nach
  # "self.data_dict".
  def refresh(self):
    self.__loadDict()

  # ###########################################################
  # Liefert die Maximalwerte als Tupel (received, sent) für den
  # Zeitbereich von "start_idx" bis "end_idx".
  def getMaxTupleForRange(self, start_idx, end_idx):
    start_idx=max(0, start_idx)
    end_idx=min(60*24-1, end_idx)
    maxs=maxr=0
    for i in xrange(start_idx, end_idx+1):
      maxr=max(maxr, self.data_dict[i][0])
      maxs=max(maxs, self.data_dict[i][1])
    return((maxr, maxs))

  # ###########################################################
  # Liefert die insgesamt übertragenen Bytes als Tupel
  # (received, sent) für den Zeitbereich von "start_idx" bis
  # "end_idx". Einmal berechnete Werte für einen Zeitbereich
  # werden gecached (aber von __loadDict() ggf. wieder
  # gelöscht), um sie nicht bei jedem Bildschirmaufbau neu
  # summieren zu müssen.
  def getSumTupleForRange(self, start_idx=0, end_idx=1439):
    start_idx=max(0, start_idx)
    end_idx=min(60*24-1, end_idx)
    # Zeitbereich im Cache suchen
    sumr, sums=self.sumCache.get((start_idx, end_idx), (-1, -1))
    if sumr!=-1 and sums!=-1:     # wenn im Cache gefunden
      return((sumr*60, sums*60))  # daraus zurückliefern
    sums=sumr=0
    for i in xrange(start_idx, end_idx+1):
      sumr+=self.data_dict[i][0]
      sums+=self.data_dict[i][1]
    # Summen zum Zeitbereich im Cache ablegen
    self.sumCache.update({(start_idx, end_idx) : (sumr, sums)})
    return((sumr*60, sums*60))

  # ###########################################################
  # Initalisiert "self.data_dict" mit einem Wert pro Minute
  # für einen Tag.
  def __initDict(self):
    for idx in xrange(60*24+1): # eine Minute drüber (ist immer 0)
      self.data_dict.update({idx:(0, 0)})

  # ###########################################################
  # Liefert die Größe von "self.filename" oder None, wenn
  # "self.filename" nicht gefunden wurde.
  def __getSize(self):
    try:
      size=os.path.getsize(self.filename)
    except:
      size=None
    return(size)

  # ###########################################################
  # Lädt die Datei in "self.filename" in Rohform ab Position
  # "self.size" und setzt "self.size" auf das [derzeitige] neue
  # Ende der Datei.
  #
  # Es wird read() statt readlines() verwendet, um die zu
  # lesende Anzahl von Bytes bestimmen zu können. Ansonsten
  # wäre es denkbar, dass sich die Datei genau in dem Moment
  # zwischen readlines() und getsize() ändert.
  def __loadFile(self):
    lnl=""
    try:
      fobj=open(self.filename, "r")
    except:
      self.errorReason="Datei kann nicht geöffnet werden!"
      return([])
    if fobj!=None:
      fobj.seek(self.size)                # letzte Endposition anfahren
      size=os.path.getsize(self.filename) # neue Endposition holen
      if size>102400:                     # wenn >100KB...
        self.errorReason="Dateigröße übersteigt 100KB."
        return([])                        # ...ist das kein Tages-Log
      # Neues lesen und nach Liste aus Zeilen wandeln
      lnl=fobj.read(size-self.size).split("\n")
      fobj.close()
      self.size=size  # neu gemerktes Ende für nächsten Lauf speichern
    return(lnl)

  # ###########################################################
  # Lädt die Datei in "self.filename" in "self.data_dict" oder
  # verlängert "self.data_dict", wenn "self.filename" gewachsen
  # ist.
  def __loadDict(self):
    if self.logFromTar!=None:
      lnl=self.logFromTar.split("\n")
    else:
      lnl=self.__loadFile()

    if lnl==[]:
      self.isValid=False
      return

    for ln in lnl:      # über alle Zeilen
      if ln!="":        # Leerzeilen ignorieren
        try:
          tm, rr, sr=ln.split()   # Zeile zerlegen
          h, m=tm.split(":")      # Zeit zerlegen
          rri=int(rr)             # Typ wandeln
          sri=int(sr)
        except:
          self.isValid=False
          self.errorReason="Das Format der Daten ist illegal."
          return
        self.maxr=max(self.maxr, rri) # Maxima merken bzw. korrigieren
        self.maxs=max(self.maxs, sri)
        self.sumr+=rri                # Summen bilden bzw. korrigieren
        self.sums+=sri
        # der SummenCache wird gelöscht, weil er evt. Summen enthält, die
        # jetzt (nach refresh()) nicht mehr stimmen.
        self.sumCache={}
        self.data_dict[int(m)+int(h)*60]=(rri, sri) # ins Dictionary aufnehmen
    self.isValid=True




# ###########################################################
# Eine kleine Klasse, um neue Dateien via Drag&Drop öffnen
# zu können.
class FileDrop(wx.FileDropTarget):
  def __init__(self, window):
    wx.FileDropTarget.__init__(self)
    self.window=window

  def OnDropFiles(self, x, y, filenames):
    if len(filenames)>1:
      wx.MessageBox('Es kann nur eine Datei geladen werden!', 'Fehler', wx.OK | wx.ICON_ERROR)
      return
    self.window.fileDropped(filenames[0])




# ###########################################################
# Das Fenster.
class TunBwWindow(wx.Window):
  def __init__(self, parent):
    wx.Window.__init__(self, parent)
    self.parent=parent

    self.color_background="#CCCCCC"
    self.color_receive=   "#00B0FF"
    self.color_send=      "#FF3800"
    self.color_overlay=   "#E991AA"
    self.color_grid=      "#FFFFFF"
    self.color_text=      "#000000"
    self.color_marker=    "#000000"
    self.color_selection= "#FFFFFF" # invertiert

    self.SetBackgroundStyle(wx.BG_STYLE_CUSTOM)

    self.filename=("", "")  # Format: (dateiname, "") | (archivname, dateiname)
    self.tarIsLoaded=False
    self.createMenu()

    self.leftMouseIsDown=False
    self.ctrlIsDown=False
    self.logdata=None

    self.max_bw_dummy=BW_MAX   # irgend ein Wert, solange keine Datei geladen ist
    self.max_bw=self.max_bw_dummy
    self.filename_dt=FileDrop(self)
    self.SetDropTarget(self.filename_dt)

    self.fileModTimer=wx.Timer(self)
    self.Bind(wx.EVT_TIMER, self.onFileModTimer, self.fileModTimer)

    self.winMoveTimer=wx.Timer(self)
    self.Bind(wx.EVT_TIMER, self.onWinMoveTimer, self.winMoveTimer)

    self.zoom_history=[]
    self.time_start=TIME_MIN
    self.time_end=TIME_MAX

    if filename_g!="":  # via cmdline übergeben
      if tarfile.is_tarfile(filename_g)==True:  # wenn es ein Archiv ist
        lst, dummy=self.getFilenameListFromArchive(filename_g)
        if len(lst)>0:                          # wenn mindestens eine Datei enthalten ist
          self.loadFile((filename_g, lst[0]))   # erste Datei laden
      else:                                     # kein Archiv
        self.loadFile((filename_g, ""))         # Datei laden

    self.screen_width, self.screen_height=self.parent.GetSize()
    self.setPresentationAreaVariables()

    self.parent.Bind(wx.EVT_MOVE,   self.onMove)
    self.Bind(wx.EVT_CONTEXT_MENU,  self.onContextMenu)
    self.Bind(wx.EVT_PAINT,         self.onPaint)
    self.Bind(wx.EVT_SIZE,          self.onSize)
    self.Bind(wx.EVT_LEFT_UP,       self.onLeftUp)
    self.Bind(wx.EVT_LEFT_DOWN,     self.onLeftDown)
    self.Bind(wx.EVT_MOTION,        self.onMotion)
    self.Bind(wx.EVT_KEY_DOWN,      self.onKeyDown)
    self.Bind(wx.EVT_KEY_UP,        self.onKeyUp)
    self.Bind(wx.EVT_SET_FOCUS,     self.onSetFocus)
    self.Bind(wx.EVT_MOUSEWHEEL,    self.onMouseWheel)

    self.statusbar=self.parent.CreateStatusBar(4)
    self.parent.SetStatusWidths([-20, -2, -10, -10])
    wx.FutureCall(200, self.checkForBatchSaveSnapshot)

    # ohne SetFocus() kommen EVT_SET_FOCUS, EVT_KEY_UP und EVT_KEY_DOWN nicht
    wx.FutureCall(500, self.SetFocus)

  # ###########################################################
  # Scrollen mit dem MouseWheel verarbeiten.
  def onMouseWheel(self, event):
    w=self.screen_width/10            # 10% der Fensterbreite ist ein Schritt
    w=max(self.pixel_per_minute+1, w) # aber mindestens eine Minute
    h=self.screen_height/10           # bei der Höhe ebenfalls 10%
    if self.ctrlIsDown==True:         # wenn Strg-Taste gedrückt ist
      m=4                             # vertikal scrollen
      if event.GetWheelRotation()<0:  # Wheel runter
        s=wx._core.Point(0, 0)        # Kurve nach unten verschieben
        e=wx._core.Point(0, h)        # also Bandbreite vergrößern
      else:                           # Wheel rauf
        s=wx._core.Point(0, h)        # Kurve nach oben verschieben
        e=wx._core.Point(0, 0)        # also Bandbreite verkleinern
    else:                             # wenn Strg-Taste nicht gedrückt ist
      m=3                             # horizontal scrollen
      if event.GetWheelRotation()<0:  # Wheel runter
        s=wx._core.Point(w, 0)        # Kurve nach links verschieben
        e=wx._core.Point(0, 0)        # also in der Zeit vorwärts
      else:                           # Wheel rauf
        s=wx._core.Point(0, 0)        # Kurve nach rechts verschieben
        e=wx._core.Point(w, 0)        # also in der Zeit rückwärts
    self.setPresentationAreaVariables(m, s, e)
    self.paint()

  # ###########################################################
  # Fokus-Wiederherstellung verarbeiten.
  def onSetFocus(self, event):
    self.blockOnLeftDown=True
    # onLeftDown() kurz abschalten, um den Fall abzufangen, dass
    # direkt mit SetFocus auch eine Bereichsauswahl gestartet
    # wird - das läuft dann nämlich (warum auch immer) nicht
    # korrekt mit XOR...und mit den Farben fürs Grid.
    wx.FutureCall(10, self.enableOnLeftDown)
    self.Refresh()

  # ###########################################################
  # Wird 10ms nach SetFocus aufgerufen und reaktiviert
  # onLeftDown().
  def enableOnLeftDown(self):
    self.blockOnLeftDown=False

  # ###########################################################
  # Wird aus FileDrop aufgerufen, wenn ein DateiObjekt auf dem
  # Fenster "fallengelassen" wurde.
  def fileDropped(self, filename):
    self.loadAndShowFile((filename, ""))

  # ###########################################################
  # Lädt "fn" und stellt dessen Inhalt als Kurve dar.
  def loadAndShowFile(self, filenameTuple):
    self.loadFile(filenameTuple)
    self.setPresentationAreaVariables()
    self.paint()

  # ###########################################################
  # Wenn Aufruf-Parameter -s übergeben wurde, einen Snapshot
  # der Kurve speichern und Programm beenden.
  def checkForBatchSaveSnapshot(self):
    if snapname_g!="":
      self.show_minutemarks=False # MinutenMarker abschalten
      self.statusbar.Hide()       # Statusbar abschalten
      wx.Yield()                  # und auch sofort machen
      self.paint(True)            # bevor neu gezeichnet wird
      self.saveSnapshot(self.dc, snapname_g+".png")
      self.parent.Close()

  # ###########################################################
  # Lädt die Datei im Tupel "fn" nach "self.logdata" und setzt
  # "self.time_start" und "self.time_end" entsprechend.
  def loadFile(self, fn):
    #print "loadFile()=", fn
    if tarfile.is_tarfile(fn[0])==True:     # wenn es ein Archiv ist
      if fn[1]!="":                         # wenn Datei im Archiv schon bestimmt ist
        tarfl=tarfile.open(fn[0])           # Archiv öffnen
        tfobj=tarfl.getmember(fn[1])        # Handle aus Dateiname
        logdata=tarfl.extractfile(tfobj).read() # Datei-Inhalt lesen
        tarfl.close()                       # Archiv schließen
        self.logdata=DayLog(fn[1], logdata) # nach self.logdata laden
        self.tarIsLoaded=True
        self.filename=(fn[0], fn[1])
      else:                                 # Archiv ist bekannt, Datei aber noch nicht
        lst, dummy=self.getFilenameListFromArchive(fn[0])
        an=os.path.basename(fn[0])
        dlg=ListBoxDialog(self, "Auswahl "+an, (210, self.screen_height), lst)
        if dlg.ShowModal()==wx.ID_OK:       # wenn eine Datei gewählt wurde
          idx=dlg.getSelectedIdx()          # deren Index (in lst) holen
          tarfl=tarfile.open(fn[0])         # Archiv öffnen
          tfobj=tarfl.getmember(lst[idx])   # Handle aus Index
          logdata=tarfl.extractfile(tfobj).read() # Datei-Inhalt lesen
          tarfl.close()                     # Archiv schließen
          dlg.Destroy()
          self.logdata=DayLog(fn[0], logdata)  # nach self.logdata laden
          self.tarIsLoaded=True
          self.filename=(fn[0], lst[idx])
        else:                               # Datei-Auswahl abgebrochen
          dlg.Destroy()
          return
    else:
      self.logdata=DayLog(fn[0])            # Non-Archiv-Datei nach self.logdata laden
      self.tarIsLoaded=False
      self.filename=(fn[0], "")

    self.zoom_history=[]
    self.time_start=TIME_MIN
    self.time_end=TIME_MAX

    if self.logdata.isValidLogFile()!=True:
      wx.MessageBox("Die Datei kann nicht verarbeitet werden!" \
                    "\n\nGrund: %s"%(self.logdata.reasonForError()), 
                    "Fehler", wx.OK|wx.ICON_ERROR, self)
      self.logdata=None
      self.max_bw=self.max_bw_dummy
      return

    self.max_bw=self.getMaxBwForShownCurves()
    if self.max_bw==0:
      self.max_bw=self.max_bw_dummy
    self.setFilenameInStatusBar()
    self.parent.SetStatusText(self.getSumTupleText(), 3)
    self.resetFileModMenu()
    self.startStopFollowTimer()

  # ###########################################################
  # Liefert die maximale Übertragungsgeschwindigkeit (ggf. für
  # den gewählten Zeitbereich) abhängig von den aktuell
  # angeschalteten Kurven.
  def getMaxBwForShownCurves(self, start_idx=None, end_idx=None):
    if start_idx!=None and end_idx!=None:
      max_rec, max_snd=self.logdata.getMaxTupleForRange(start_idx, end_idx)
    else:
      max_rec, max_snd=self.logdata.getMaxTuple()

    if self.show_send==True and self.show_receive==True:
      max_bw=max(max_rec, max_snd)
    elif self.show_receive==True: # nur Empfangskurve an
      max_bw=max_rec
    else:                         # keine, Überlappungskurve oder Sendekurve an
      max_bw=max_snd
    return(max_bw)

  # ###########################################################
  # Liefert eine Liste mit den .log-Dateien in dem Archiv
  # "arc_name". Mit Übergabe von "logfileToFind" wird der Index
  # von dem LogfileNamen geliefert.
  def getFilenameListFromArchive(self, arc_name, logfileToFind=""):
    tarfl=tarfile.open(arc_name)      # Archiv öffnen
    lst=tarfl.getnames()              # Inhaltsverzeichnis lesen
    lst.sort()                        # Inhaltsverzeichnis sortieren
    filelst=[]
    idx=0
    curidx=None
    for f in lst:
      tfobj=tarfl.getmember(f)        # Handle aus Dateiname
      if tfobj.isfile()==True:        # wenn es eine Datei ist
        ext=os.path.splitext(f)[1]    # Extension abtrennen
        if ext.lower()==".log":       # wenn .log-Datei
          filelst.append(f)           # dann in Zielliste kopieren
          if f==logfileToFind:
            curidx=idx
          idx+=1
    tarfl.close()
    return((filelst, curidx))

  # ###########################################################
  # Setzt "self.filename" gemäß "self.tarIsLoaded" in der
  # StatusBar.
  def setFilenameInStatusBar(self):
    if self.tarIsLoaded==True:
      self.parent.SetStatusText(self.filename[0]+"("+self.filename[1]+")", 0)
    else:
      self.parent.SetStatusText(self.filename[0], 0)

  # ###########################################################
  # Stellt das Kontext-Menue dar.
  def onContextMenu(self, event):
    self.cancelSelection()
    self.PopupMenu(self.menue)
    self.Refresh()
    # ohne Refresh ist nur noch der Bereich, an dem das
    # Kontextmenü dargestellt wurde, schreibbar

  # ###########################################################
  # Legt das Kontext-Menue an.
  def createMenu(self):
    self.menue=wx.Menu()
    self.menue.Append(99, 'letzten Zoom zurücknehmen')
    self.menue.Append(100, 'Zoom auf Grundstellung')
    self.menue.AppendSeparator()
    self.menue.Append(109, 'Log laden')
    self.menue.AppendSeparator()
    self.menue.AppendCheckItem(101, 'SendeKurve anzeigen')
    self.menue.AppendCheckItem(102, 'EmpfangsKurve anzeigen')
    self.menue.AppendCheckItem(103, 'Überlagerung anzeigen')
    self.menue.AppendCheckItem(104, 'MinutenMarker anzeigen')
    self.menue.AppendSeparator()
    self.menue.AppendCheckItem(107, 'Änderungen verfolgen')
    self.menue.AppendCheckItem(108, 'lokale Y-Anpassung')
    self.menue.AppendSeparator()
    self.menue.Append(105, 'Snapshot speichern')
    self.menue.AppendSeparator()
    self.menue.Append(106, 'aktuelle Einstellungen speichern')
    self.menue.AppendSeparator()
    self.menue.Append(110, 'über das Programm')

    fc=wx.FileConfig(localFilename=CFGFILE)
    self.show_send        =fc.ReadInt("dsp_snd", True)
    self.show_receive     =fc.ReadInt("dsp_rec", True)
    self.show_overlap     =fc.ReadInt("dsp_ovl", True)
    self.show_minutemarks =fc.ReadInt("dsp_mrk", False)
    self.follow_mode      =fc.ReadInt("follow",  False)
    self.autoY            =fc.ReadInt("autoY",   False)
    del fc

    if curves_g!="":  # cmd-Parameter überstimmt cfg-File
      c=curves_g.upper()
      if "S" in c: self.show_send=True
      else:        self.show_send=False
      if "R" in c: self.show_receive=True
      else:        self.show_receive=False
      if "O" in c: self.show_overlap=True
      else:        self.show_overlap=False

    self.menue.Check(101, self.show_send)
    self.menue.Check(102, self.show_receive)
    self.menue.Check(103, self.show_overlap)
    self.menue.Check(104, self.show_minutemarks)
    self.menue.Check(107, self.follow_mode)
    self.menue.Check(108, self.autoY)
    self.resetFileModMenu()
    self.justify_Y=self.menue.IsChecked(108)

    self.Bind(wx.EVT_MENU, self.men_undoZoom,           id=99)
    self.Bind(wx.EVT_MENU, self.men_resetZoom,          id=100)
    self.Bind(wx.EVT_MENU, self.men_aboutWin,           id=110)
    self.Bind(wx.EVT_MENU, self.men_switchSend,         id=101)
    self.Bind(wx.EVT_MENU, self.men_switchReceive,      id=102)
    self.Bind(wx.EVT_MENU, self.men_switchOverlap,      id=103)
    self.Bind(wx.EVT_MENU, self.men_switchMinuteMarker, id=104)
    self.Bind(wx.EVT_MENU, self.men_saveNamedSnapshot,  id=105)
    self.Bind(wx.EVT_MENU, self.men_saveConfig,         id=106)
    self.Bind(wx.EVT_MENU, self.men_followTimer,        id=107)
    self.Bind(wx.EVT_MENU, self.men_justifyY,           id=108)
    self.Bind(wx.EVT_MENU, self.men_loadLogFile,        id=109)

  # ###########################################################
  # Lädt ein LogFile via Auswahl-Dialog.
  def men_loadLogFile(self, event):
    fc=wx.FileConfig(localFilename=CFGFILE)
    fn=fc.Read("lastFile", "")
    del fc
    if fn=="":                        # wenn noch keine Datei im Configfile abgelegt ist...
      basepth=os.path.expanduser('~') # ...Homeverzeichnis einstellen
    else:
      basepth=os.path.dirname(fn)     # sonst den Pfad der zuletzt geladenen Datei
    dlg=wx.FileDialog(self, message="open file", defaultDir=basepth, defaultFile="", \
                      wildcard="log|*.log;*.tar.gz;*.tar.bz2|all|*", \
                      style=wx.FD_OPEN|wx.FD_FILE_MUST_EXIST)
    if dlg.ShowModal()!=wx.ID_OK:
      dlg.Destroy()
      return
    fn=dlg.GetPath()
    dlg.Destroy()
    fc=wx.FileConfig(localFilename=CFGFILE)
    fc.Write("lastFile", fn)
    fc.Flush()
    del fc
    self.loadAndShowFile((fn, ""))

  # ###########################################################
  # Setzt das Menü neu, wenn neue Datei geladen wird. Der
  # Menüpunkt "Änderungen verfolgen" wird freigeschaltet, wenn
  # der Name des Logfiles dem Tagesdatum entspricht - also
  # potentiell noch verlängert wird. Allerdings nicht, wenn
  # die Datei aus einem tar.bz2 geladen wurde.
  def resetFileModMenu(self):
    if self.tarIsLoaded==False:
      if os.path.basename(self.filename[0])==time.strftime("%Y_%m_%d")+".log":
        self.menue.Enable(107, True)
        self.menue.Enable(108, True)
        return
    self.menue.Enable(107, False)
    self.menue.Enable(108, False)

  # ###########################################################
  # SendeKurve an/aus
  def men_switchSend(self, event):
    self.show_send=self.menue.IsChecked(101)
    self.paint()

  # ###########################################################
  # EmpfangsKurve an/aus
  def men_switchReceive(self, event):
    self.show_receive=self.menue.IsChecked(102)
    self.paint()

  # ###########################################################
  # ÜberlagerungsKurve an/aus
  def men_switchOverlap(self, event):
    self.show_overlap=self.menue.IsChecked(103)
    self.paint()

  # ###########################################################
  # WerteMarker an/aus
  def men_switchMinuteMarker(self, event):
    self.show_minutemarks=self.menue.IsChecked(104)
    self.paint()

  # ###########################################################
  # Fragt einen Speicher-Dateinamen ab und speichert dann den
  # Inhalt des DC unter diesem Namen als PNG-File.
  def men_saveNamedSnapshot(self, event):
    dlg=wx.FileDialog(self, message="Screenshot speichern",
                      defaultDir=os.path.join(os.path.expanduser('~')),
                      defaultFile="snapshot.png",
                      wildcard="PNGs|*.png|alle|*", style=wx.FD_SAVE)
    if dlg.ShowModal()!=wx.ID_OK:
      dlg.Destroy()
      return
    filename=dlg.GetPath()
    dlg.Destroy()
    self.paint(True)
    self.saveSnapshot(self.dc, filename)
  
  # ###########################################################
  # Speichert den DC "dcSource" als PNG.
  # Quelle: http://stackoverflow.com/a/2196912/3588613
  def saveSnapshot(self, dcSource, filename):
    # based largely on code posted to wxpython-users by Andrea Gavana 2006-11-08
    size = dcSource.Size

    # Create a Bitmap that will later on hold the screenshot image
    # Note that the Bitmap must have a size big enough to hold the screenshot
    # -1 means using the current default colour depth
    bmp = wx.EmptyBitmap(size.width, size.height)

    # Create a memory DC that will be used for actually taking the screenshot
    memDC = wx.MemoryDC()

    # Tell the memory DC to use our Bitmap
    # all drawing action on the memory DC will go to the Bitmap now
    memDC.SelectObject(bmp)

    # Blit (in this case copy) the actual screen on the memory DC
    # and thus the Bitmap
    memDC.Blit( 0, # Copy to this X coordinate
        0, # Copy to this Y coordinate
        size.width, # Copy this width
        size.height, # Copy this height
        dcSource, # From where do we copy?
        0, # What's the X offset in the original DC?
        0  # What's the Y offset in the original DC?
        )

    # Select the Bitmap out of the memory DC by selecting a new
    # uninitialized Bitmap
    memDC.SelectObject(wx.NullBitmap)

    img = bmp.ConvertToImage()
    img.SaveFile(filename, wx.BITMAP_TYPE_PNG)

  # ###########################################################
  # Speichert Fenster-Größe und Position sowie die Einstellung
  # der anzuzeigenden Kurven.
  def men_saveConfig(self, event):
    fc=wx.FileConfig(localFilename=CFGFILE)
    sp=self.parent.GetScreenPosition()
    ss=self.parent.GetSizeTuple()
    fc.WriteInt("pos_x",    sp[0])
    fc.WriteInt("pos_y",    sp[1])
    fc.WriteInt("size_x" ,  ss[0])
    fc.WriteInt("size_y" ,  ss[1])
    fc.WriteInt("dsp_snd",  self.menue.IsChecked(101))
    fc.WriteInt("dsp_rec",  self.menue.IsChecked(102))
    fc.WriteInt("dsp_ovl",  self.menue.IsChecked(103))
    fc.WriteInt("dsp_mrk",  self.menue.IsChecked(104))
    fc.WriteInt("follow",   self.menue.IsChecked(107))
    fc.WriteInt("autoY",    self.menue.IsChecked(108))
    fc.Flush()

  # ###########################################################
  # About-Fenster öffnen.
  def men_aboutWin(self, event):
    info=wx.AboutDialogInfo()
    info.SetName("tun0graf")
    info.SetVersion(VERSION)
    info.SetCopyright("D.A.  (06.2014)")
    info.SetDescription("Ein Tool zur Anzeige von Bandbreiten-Logs")
    info.SetLicence("Dieses Programm ist freie Software gemaess GNU General Public License")
    info.AddDeveloper("Detlev Ahlgrimm")
    wx.AboutBox(info)

  # ###########################################################
  # Stellt die jeweils vorige Zoom-Stufe wieder her.
  def men_undoZoom(self, event):
    if len(self.zoom_history)>0:
      self.time_start, self.time_end, self.max_bw=self.zoom_history.pop()
      self.setPresentationAreaVariables()
      self.paint()

  # ###########################################################
  # Stellt den angezeigten Bereich wieder auf Anfangswerte ein.
  def men_resetZoom(self, event):
    self.zoom_history=[]
    self.time_start=TIME_MIN
    self.time_end=TIME_MAX
    if self.logdata!=None:
      self.max_bw=self.getMaxBwForShownCurves()
    else:
      self.max_bw=self.max_bw_dummy
    self.setPresentationAreaVariables()
    self.paint()

  # ###########################################################
  # Start einer Bereichs-Auswahl.
  def onLeftDown(self, event):
    if self.blockOnLeftDown==True:  # wenn der Focus gerade erst erhalten wurde
      return                        # ...noch nix selektieren
    self.sel_start_point=event.GetPosition()
    self.sel_actu_point=self.sel_start_point
    self.SetCursor(wx.StockCursor(wx.CURSOR_CROSS))
    self.dc.SetLogicalFunction(wx.XOR)
    self.dc.SetBrush(wx.Brush(self.color_selection, wx.TRANSPARENT))
    self.dc.SetPen(wx.Pen(self.color_selection, 1, wx.SOLID))
    self.leftMouseIsDown=True
    self.ctrl=self.ctrlIsDown
    txt=self.pointToTimeAndBandwidth(self.sel_start_point)
    self.parent.SetStatusText(txt, 2)

  # ###########################################################
  # Dragging bzw. Maus-Bewegung zwischen LeftDown und LeftUp.
  def onMotion(self, event):
    if self.leftMouseIsDown==True:
      # altes Auswahl-Objekt löschen bzw. weg-XOR'n
      self.drawSelectionLines(self.sel_start_point, self.sel_actu_point, self.ctrl)
      # neues Auswahl-Objekt darstellen
      self.sel_actu_point=event.GetPosition()
      self.ctrl=self.drawSelectionLines(self.sel_start_point, self.sel_actu_point, self.ctrlIsDown)

  # ###########################################################
  # Wird bei KeyDown aufgerufen und setzt self.ctrlIsDown gemäß
  # Gedrückt-Zustand der Strg-Taste.
  # Weiterhin werden die Cursor-Tasten abgefragt und ggf. eine
  # neue Datei geladen.
  def onKeyDown(self, event):
    self.ctrlIsDown=event.GetKeyCode()==wx.WXK_CONTROL
    if event.GetKeyCode()==wx.WXK_LEFT:
      self.changeFile(1)
    elif event.GetKeyCode()==wx.WXK_RIGHT:
      self.changeFile(2)
    self.onMotion(event)  # ggf. Auswahl neu zeichnen

  # ###########################################################
  # Wird bei KeyUp aufgerufen und setzt self.ctrlIsDown auf
  # False.
  def onKeyUp(self, event):
    self.ctrlIsDown=False
    self.onMotion(event)  # ggf. Auswahl neu zeichnen

  # ###########################################################
  # Bricht eine onLeftDown-onMotion-onLeftUp-Folge ab, wenn
  # mittendrin ein onContextMenu auftritt.
  def cancelSelection(self):
    self.leftMouseIsDown=False
    self.SetCursor(wx.StockCursor(wx.CURSOR_ARROW)) 
    self.dc.SetLogicalFunction(wx.COPY)
    self.paint()

  # ###########################################################
  # Lädt die vorige (direction=1) oder die nächste (direction=2)
  # Log-Datei im aktuellen Verzeichnis oder Archiv. In einem
  # Archiv dürfen sich nur Log-Dateien befinden.
  def changeFile(self, direction):
    if self.tarIsLoaded==True:
      filelst, curidx=self.getFilenameListFromArchive(self.filename[0], self.filename[1])
      if curidx==None:    # wenn aktuelle Datei nicht gefunden wurde
        return            # ...wurde gelöscht -> selber Schuld

      if direction==1:    # vorige Datei laden
        if curidx>0:
          self.loadAndShowFile((self.filename[0], filelst[curidx-1]))
      elif direction==2:  # nächste Datei laden
        if curidx<len(filelst)-1:
          self.loadAndShowFile((self.filename[0], filelst[curidx+1]))
    else:
      if self.filename[0]!="":
        basepth=os.path.dirname(self.filename[0])  # aktueller Pfad
        dirlst=os.listdir(basepth)        # alle Dateien im Pfad
        dirlst.sort()                     # Dateinamen sortieren
        filelst=[]
        idx=0
        curidx=None
        for f in dirlst:                  # über alle Dateien
          pf=os.path.join(basepth, f)     # Pfad + Dateiname
          if os.path.isfile(pf)==True:    # wenn Datei...
            ext=os.path.splitext(pf)[1]   # Extension abtrennen
            if ext.lower()==".log":       # wenn Extension == ".log"
              filelst.append(pf)          # merken
              if pf==self.filename[0]:    # wenn aktuelle Datei...
                curidx=idx                # Index in filelst merken
              idx+=1
        if curidx==None:    # wenn aktuelle Datei nicht gefunden wurde
          return            # ...wurde gelöscht -> selber Schuld

        if direction==1:    # vorige Datei laden
          if curidx>0:
            self.loadAndShowFile((filelst[curidx-1], ""))
        elif direction==2:  # nächste Datei laden
          if curidx<len(filelst)-1:
            self.loadAndShowFile((filelst[curidx+1], ""))

  # ###########################################################
  # Zeichnet ein Rechteck, bei dem die untere Waagerechte fehlt
  # oder Pfeile in die entspr. Verschiebe-Richtung.
  # Bei ctrl==True werden zwei senkrechte Linien gezeichnet.
  def drawSelectionLines(self, start_point, actu_point, ctrl):
    x1=min(start_point.x, actu_point.x)
    y1=max(start_point.y, actu_point.y)
    x2=max(start_point.x, actu_point.x)
    y2=min(start_point.y, actu_point.y)
    xm=start_point.x-actu_point.x
    ym=start_point.y-actu_point.y

    if ctrl==True:
      self.dc.DrawLine(x1, self.c_nul_y, x1, self.c_max_y)  # zwei senkrechte Linien
      self.dc.DrawLine(x2, self.c_nul_y, x2, self.c_max_y)
      return(ctrl)

    if abs(ym)<10 and abs(xm)>10:                 # waagerechter Pfeil
      if xm<0:                                    # Pfeil nach links
        self.dc.DrawLine(x1, y2, x2, y2)
        self.dc.DrawLine(x1, y2, x1+10, y2+10)
        self.dc.DrawLine(x1, y2, x1+10, y2-10)
      else:                                       # Pfeil nach rechts
        self.dc.DrawLine(x1, y2, x2, y2)
        self.dc.DrawLine(x2, y2, x2-10, y2+10)
        self.dc.DrawLine(x2, y2, x2-10, y2-10)
    elif abs(ym)>10 and abs(xm)<10:               # senkrechter Pfeil
      if ym<0:                                    # Pfeil nach oben
        self.dc.DrawLine(x1, y2, x1, y1)
        self.dc.DrawLine(x1, y2, x1-10, y2+10)
        self.dc.DrawLine(x1, y2, x1+10, y2+10)
      else:                                       # Pfeil nach unten
        self.dc.DrawLine(x1, y2, x1, y1)
        self.dc.DrawLine(x1, y1, x1-10, y1-10)
        self.dc.DrawLine(x1, y1, x1+10, y1-10)
    else:
      self.dc.DrawLine(x1, y1, x1, y2)            # unten offenes Rechteck
      self.dc.DrawLine(x1, y2, x2, y2)
      self.dc.DrawLine(x2, y2, x2, y1)
    return(ctrl)

  # ###########################################################
  # Ende einer Bereichs-Auswahl.
  # Das heißt Zoomen oder Bereich verschieben...
  def onLeftUp(self, event):
    if self.leftMouseIsDown==False:
      return

    self.leftMouseIsDown=False
    self.SetCursor(wx.StockCursor(wx.CURSOR_ARROW)) 
    self.dc.SetLogicalFunction(wx.COPY)

    xm=self.sel_start_point.x-self.sel_actu_point.x
    ym=self.sel_start_point.y-self.sel_actu_point.y

    # zu kleinen Bereich abfangen bzw. ignorieren
    if abs(xm)<10 and abs(ym)<10:
      return

    if self.ctrlIsDown==True:
      mode=2      # horizontal raus-zoomen
    elif abs(ym)<10:
      mode=3      # waagerecht verschieben
    elif abs(xm)<10:
      mode=4      # senkrecht verschieben
    else:
      mode=1      # sowohl X wie auch Y sind >10 also soll wohl rein-gezoomt werden

    self.setPresentationAreaVariables(mode, self.sel_start_point, self.sel_actu_point)
    self.paint()

  # ###########################################################
  # Aktualisiert das Fenster.
  def onPaint(self, event):
    self.paint()

  # ###########################################################
  # Aktualisiert das Fenster.
  def paint(self, save=False):
    self.dc=wx.AutoBufferedPaintDC(self)
    self.dc.SetBackground(wx.Brush(self.color_background))
    self.dc.SetTextForeground(self.color_text)
    self.dc.Clear()

    # der sichtbare Bereich liegt auf:
    #   horizontal  0 bis self.screen_width-1
    #   vertikal    0 bis self.screen_height-1
    self.screen_width, self.screen_height=self.GetSizeTuple()

    if self.logdata!=None:
      self.drawCurve()    # wenn Datei geladen -> Kurve anzeigen
    else:
      self.drawGrid()     # keine Datei -> nur Raster anzeigen

    if save==True:
      self.dc.SetTextForeground(self.color_grid)
      fnt=wx.Font(8, wx.FONTFAMILY_MODERN, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_LIGHT)
      self.dc.SetFont(fnt)
      txt="tun0graf"
      w, h=self.dc.GetTextExtent(txt)
      self.dc.DrawText(txt, self.c_max_x-w-3, self.screen_height-h)
      self.dc.SetTextForeground(self.color_text)

    self.setFilenameInStatusBar()
    self.parent.SetStatusText(str(len(self.zoom_history)), 1)
    self.parent.SetStatusText(self.getSumTupleText(), 3)

  # ###########################################################
  # Resize des Fensters.
  def onSize(self, event):
    self.parent.SetMinSize(wx.Size(150, 120))
    self.setPresentationAreaVariables()
    self.paint()
    self.Refresh()

  # ###########################################################
  # Fenter wird/wurde verschoben.
  # Da während des Verschiebens nicht immer Refreshed werden
  # soll, wird hier nur ein Timer restartet, so dass erst dann
  # der Refresh erfolgt, wenn 200ms lang keine Bewegung erkannt
  # wurde.
  def onMove(self, event):
    self.winMoveTimer.Start(200, oneShot=True)  # [Re]start timer

  # ###########################################################
  # Hier landet er, wenn nach einer Fenster-Verschiebung 200ms
  # keine weitere Verschiebung erkannt wurde.
  def onWinMoveTimer(self, event):
    # der Refresh ist nötig, da sonst nach einer Verschiebung
    # drawSelectionLines() nicht mehr korrekt zeichnet.
    self.paint()
    self.Refresh()

  # ###########################################################
  # Wird bei jedem Timer-Tick aufgerufen und prüft auf
  # Änderungen der angezeigten Datei.
  def onFileModTimer(self, event):
    if self.leftMouseIsDown==True:
      # wenn gerade gezoomt oder der Ausschnitt geändert wird, nix ändern
      return
    if self.logdata!=None and self.logdata.hasChanged()==True:
      # wenn sich Dateigröße geändert hat
      self.logdata.refresh()                        # neue Zeilen einlesen
      if self.justify_Y==True:                      # wenn gewünscht...
                                                    # ...neue lokale Maxima holen
        max_bw=self.getMaxBwForShownCurves(self.time_start, self.time_end)
        if max_bw>self.max_bw:                      # wenn höher als derzeit eingestellt
          self.max_bw=max_bw                        # neue Skala-Oberkante setzen
          self.setPresentationAreaVariables()       # abhängige Variablen setzen
      self.paint()                                # und neu darstellen

  # ###########################################################
  # Startet bzw. stoppt den Timer zur Änderungsüberwachung bei
  # Änderung der Einstellung im Menü.
  def men_followTimer(self, event):
    self.startStopFollowTimer()

  # ###########################################################
  # Startet bzw. stoppt den Timer zur Änderungsüberwachung
  # gemäß Einstellung im Menü.
  def startStopFollowTimer(self):
    if self.menue.IsEnabled(107)==True:   # wenn heutiges Log geladen ist
      if self.menue.IsChecked(107)==True: # und auf Änderung überwacht werden soll
        self.fileModTimer.Start(1000)     # dann im Sekundentakt prüfen
        return
    self.fileModTimer.Stop()              # Änderungen sollen nicht überwacht werden

  # ###########################################################
  # Schaltet nur die Variable self.justify_Y gemäß Menü um.
  def men_justifyY(self, event):
    self.justify_Y=self.menue.IsChecked(108)

  # ###########################################################
  # Setzt diverse Darstellungs-Variablen gemäß Fenstergröße
  # und Bereichs-Auswahl.
  # sel_start und sel_end sind vom Typ wx._core.Point
  def setPresentationAreaVariables(self, zoom_mode=0, sel_start=None, sel_end=None):
    self.border=5               # Abstand Grafiken zum Fenster-Rand
    self.s_height=20            # Höhe der Skala

    # Anzahl Linien für Skala
    self.s_count_hor_lines=4    # 25%, 50%, 75%, 100% (die 0%-Linie zählt hier nicht mit)
    self.s_count_ver_lines=24   # wird unten ggf. verkleinert (die 0%-Linie zählt hier nicht mit)

    # Ecken/Abmessung der Kurve
    self.c_nul_x=self.border                                      # links...
    self.c_nul_y=self.screen_height-self.s_height-self.border-1   # ...unten
    self.c_max_x=self.screen_width-self.border-1                  # rechts...
    self.c_max_y=self.border                                      # ...oben
    self.c_width=self.c_max_x-self.c_nul_x+1                      # Breite der Kurve
    self.c_height=self.c_nul_y-self.c_max_y+1                     # Höhe der Kurve
    # Ecken/Abmessung der Skala
    self.s_nul_x=self.c_nul_x                                     # links...
    self.s_nul_y=self.screen_height-self.border-1                 # ...unten
    self.s_max_x=self.c_max_x                                     # rechts...
    self.s_max_y=self.c_nul_y                                     # ...oben
    self.s_width=self.c_width                                     # Breite der Skala

    if zoom_mode==1:    # zoom-in
      self.zoom_history.append((self.time_start, self.time_end, self.max_bw))
      # Selektion ausserhalb der Kurve auf die Ecken der Kurve setzen
      sxs=min(self.c_width-1,   max(0, sel_start.x-self.c_nul_x))
      sxe=min(self.c_width-1,   max(0, sel_end.x  -self.c_nul_x))
      sys=min(self.c_height-1,  max(0, self.c_nul_y-sel_start.y))
      sye=min(self.c_height-1,  max(0, self.c_nul_y-sel_end.y))
      # (sxs,sys=links-unten) und (sxe,sye=rechts-oben) durchsetzen
      if sxe<sxs: sxs, sxe=(sxe, sxs)
      if sye<sys: sys, sye=(sye, sys)
      # sxs/sxe nach Zeitbereich wandeln
      time_start_merk =self.time_start
      self.time_start =self.time_start+int(sxs/self.scale_x)
      self.time_end   =min(TIME_MAX, time_start_merk+self.roundUp(sxe/self.scale_x))
      # sys/sye nach Bandbreite wandeln
      self.max_bw=int(sye/self.scale_y)
    elif zoom_mode==2:  # zoom-out
      self.zoom_history.append((self.time_start, self.time_end, self.max_bw))
      xm=sel_start.x-sel_end.x
      ts=abs(int(xm/self.scale_x))
      self.time_start=max(TIME_MIN, self.time_start-ts)
      self.time_end=min(TIME_MAX, self.time_end+ts)
    elif zoom_mode==3:  # scroll horizontal
      xm=sel_start.x-sel_end.x
      ts=abs(int(xm/self.scale_x))
      if xm<0:                      # nach links ziehen bzw. früher
        if self.time_start-ts<TIME_MIN:
          ts=self.time_start
        self.time_start-=ts
        self.time_end-=ts
      else:                         # nach rechts ziehen bzw. später
        if self.time_end+ts>TIME_MAX:
          ts=TIME_MAX-self.time_end
        self.time_start+=ts
        self.time_end+=ts
    elif zoom_mode==4:  # scroll vertikal
      ym=sel_start.y-sel_end.y
      ts=abs(int(ym/self.scale_y))
      if ym<0:                      # nach oben ziehen
        self.max_bw=min(self.max_bw+ts, BW_MAX)
      else:                         # nach unten ziehen
        self.max_bw=max(self.max_bw-ts, BW_MIN)

    # Daten für vertikale Skala-Linien bestimmen
    self.time_width=max(1, self.time_end-self.time_start)
    self.pixel_per_minute=self.c_width/float(self.time_width)

    s_minutes_per_ver_line=max(1, float(self.time_width)/(self.c_width/60.0))
    self.s_minutes_per_ver_line=self.roundUp(s_minutes_per_ver_line)

    self.s_pixel_per_ver_line=self.s_minutes_per_ver_line*self.pixel_per_minute
    self.s_count_ver_lines=int(1+self.c_width/self.s_pixel_per_ver_line)
    if (self.s_count_ver_lines*self.s_pixel_per_ver_line)>self.c_width:
      self.s_count_ver_lines-=1 # kann wegen des SetMinSize() nicht 0 werden

    # Daten für horizontale Skala-Linien bestimmen
    self.s_pixel_per_hor_line=float(self.c_height-1)/float(self.s_count_hor_lines)

    # Faktor Zeit bzw. Bandbreite zu Pixel
    self.scale_x=float(self.c_width-1)/float(self.time_width)
    self.scale_y=float(self.c_height-1)/float(self.max_bw)

  # ###########################################################
  # Liefert "val" als aufgerundeten int (ohne "import math").
  def roundUp(self, val):
    if val>int(val):
      return(int(val)+1)
    return(int(val))

  # ###########################################################
  # Liefert zu einem Punkt im Fenster die entsprechenden Werte
  # für Bandbreite und Zeit als String.
  def pointToTimeAndBandwidth(self, point):
    sx=min(self.c_width-1,   max(0, point.x-self.c_nul_x))
    sy=min(self.c_height-1,  max(0, self.c_nul_y-point.y))
    idx=self.time_start+int(sx/self.scale_x)
    return(prettySize(int(sy/self.scale_y))+" @ "+self.indexToTime(idx))

  # ###########################################################
  # Stellt die Farbe für Linien und Poygone ein.
  def setColor(self, color):
    self.dc.SetPen(wx.Pen(color))
    self.dc.SetBrush(wx.Brush(color))

  # ###########################################################
  # Stellt die Skala dar.
  def drawGrid(self):
    self.setColor(self.color_grid)
    # vertikale Linien
    for h in range(self.s_count_ver_lines+1):
      xpos=self.c_nul_x+h*self.s_pixel_per_ver_line
      self.dc.DrawLine(xpos, self.s_nul_y, xpos, self.c_max_y)
      if self.screen_width-xpos>40: # hinter der letzten Linie nur bei Platz die Zeit ausgeben
        t=self.time_start+h*self.s_minutes_per_ver_line
        ht=self.indexToTime(t)
        self.dc.DrawText(ht, xpos+3, self.s_max_y+3)
    if abs(xpos-self.s_max_x)>10:
      self.dc.DrawLine(self.s_max_x, self.s_nul_y, self.s_max_x, self.c_max_y)

    # horizontale Linien
    for y in range(self.s_count_hor_lines+1):
      ypos=self.s_max_y-y*self.s_pixel_per_hor_line
      self.dc.DrawLine(self.c_nul_x, ypos, self.c_max_x+1, ypos)
      if y>0:   # an der Null-Linie keinen Wert anzeigen
        txt=prettySize((self.max_bw/float(self.s_count_hor_lines))*y)
        self.dc.DrawText(txt, self.c_nul_x+3, ypos)

  # ###########################################################
  # Liefert zu einem Index in "logdata.getDict()" die Zeit.
  def indexToTime(self, t):
    hr, mi=divmod(t, 60)
    return("%02d:%02d"%(hr, mi))

  # ###########################################################
  # Zeichnet die eigentliche Kurve entsprechend der Daten.
  def drawCurve(self):
    data=self.logdata.getDict()

    # Startpunkte setzen (unten links)
    poli_rec_list=[(self.c_nul_x, self.c_nul_y)]  # Polygon Empfangen
    poli_snd_list=[(self.c_nul_x, self.c_nul_y)]  # Polygon Gesendet

    # Polygon mit Werten füllen
    x=x2=0
    for x in range(self.time_start, self.time_end+1):
      # Format: (x, y)
      poli_rec_list.append((self.c_nul_x+x2, self.c_nul_y-(data[x][0]*self.scale_y)))
      poli_snd_list.append((self.c_nul_x+x2, self.c_nul_y-(data[x][1]*self.scale_y)))

      # wenn nachfolgender Wert ==0 ist -> direkt auf Null gehen, um eine steile Flanke zu erhalten
      if x<self.time_end:
        if data[x+1][0]==0:
          poli_rec_list.append((self.c_nul_x+x2, self.c_nul_y))
        if data[x+1][1]==0:
          poli_snd_list.append((self.c_nul_x+x2, self.c_nul_y))
      x2+=self.scale_x

    # Polygone schließen (unten rechts)
    poli_rec_list.append((self.c_max_x, self.c_nul_y))
    poli_snd_list.append((self.c_max_x, self.c_nul_y))

    if self.show_overlap==True:
      poli_srd_list=self.overlapping(poli_rec_list, poli_snd_list) # Polygon Überlappung

    # Polygone so beschneiden, dass sie nicht über die Oberkante der Skala hinausgehen
    lst_rec=self.cutYToMax(poli_rec_list, self.c_max_y)
    lst_snd=self.cutYToMax(poli_snd_list, self.c_max_y)
    if self.show_overlap==True:
      lst_srd=self.cutYToMax(poli_srd_list, self.c_max_y)

    #if len(lst_rec)<10:
    #  print "lst_rec\n", self.formatListOfFloatTupel(lst_rec)

    self.dc.Clear()

    if self.show_receive==True:
      self.setColor(self.color_receive)
      self.dc.DrawPolygon(lst_rec)                    # Empfangskurve darstellen

    if self.show_send==True:
      self.setColor(self.color_send)
      self.dc.DrawPolygon(lst_snd)                    # Sendekurve darstellen

    if self.show_overlap==True:
      self.setColor(self.color_overlay)
      self.dc.DrawPolygon(lst_srd)                    # Überlagerungskurve darstellen

    if self.show_minutemarks==True:
      self.setColor(self.color_marker)
      for i in range(len(poli_rec_list)):
        self.dc.DrawLine( poli_rec_list[i][0], self.screen_height, 
                          poli_rec_list[i][0], self.screen_height-5)

    self.drawGrid()                                   # Raster darstellen

    # übertragene Bytes anzeigen
    txt=self.getSumTupleText(self.time_start, self.time_end, False)
    w, h=self.dc.GetTextExtent(txt)
    self.dc.DrawText(txt, self.c_max_x-w-3, self.c_max_y+3)

  # ###########################################################
  # Formatiert eine Liste aus Tupeln mit zwei float's in eine
  # Liste aus entspr. Strings um, bei denen die Werte auf eine
  # Nachkommastelle gekürzt sind. Fürs Debugging.
  def formatListOfFloatTupel(self, lst):
    l=[]
    for x, y in lst:
      l.append(("%.1f, %.1f"%(x, y)))
    return(l)

  # ###########################################################
  # Liefert die übertragenen Bytes der Kurve als String.
  def getSumTupleText(self, time_start=TIME_MIN, time_end=TIME_MAX, alwaysShowBoth=True):
    if self.logdata!=None:
      ch_r=u'\u2B07'  # utf8 Pfeil runter
      ch_s=u'\u2B06'  # utf8 Pfeil hoch
      if time_start==TIME_MIN and time_end==TIME_MAX:
        r, s=self.logdata.getSumTuple()
      else:
        r, s=self.logdata.getSumTupleForRange(time_start, time_end)
      if alwaysShowBoth==True or (self.show_send==True and self.show_receive==True):
        txt="%s%s  %s%s"%(ch_r, prettySize(r), ch_s, prettySize(s))
      elif self.show_receive==True:
        txt="%s%s"%(ch_r, prettySize(r))
      else:
        txt="%s%s"%(ch_s, prettySize(s))
    else:
      txt=""
    return(txt)

  # ###########################################################
  # Liefert zum Polygon "poli_lst" ein neues Polygon, dessen
  # Y-Werte den Wert "maxY" nicht unterschreiten bzw. außerhalb
  # der Skala landen würden. Die gelieferte Liste kann länger
  # als die Eingangsliste sein, weil ggf. zusätzliche Schnitt-
  # punkte mit der Oberkante der Skala eingefügt werden.
  def cutYToMax(self, poli_lst, maxY):
    lst=[]
    comesFromOutside=False
    for v in range(len(poli_lst)):
      if poli_lst[v][1]<maxY:       # aktueller Wert liegt außerhalb der Skala
        if comesFromOutside==True:  # wenn der vorige Wert auch schon außerhalb war
          lst.append((poli_lst[v][0], maxY))  # Y=maxY setzen
        else:
          # Schnittpunkt mit dem oberen Skalen-Ende berechnen
          p=self.intersectionPoint(poli_lst[v-1], poli_lst[v], 
                                   (self.c_nul_x, self.c_max_y), (self.c_max_x, self.c_max_y))
          if p==None: # kann eigentlich nicht vorkommen, weil Waagerechte hier nie landen
            lst.append((poli_lst[v][0], maxY))  # Y=maxY setzen
          else:
            lst.append(p)           # sonst Schnittpunkt statt gemessenen Punkt einfügen
          comesFromOutside=True
      else:                         # aktueller Wert liegt innerhalb der Skala
        if comesFromOutside==True:  # wenn voriger Wert außerhalb lag
          p=self.intersectionPoint(poli_lst[v-1], poli_lst[v], 
                                   (self.c_nul_x, self.c_max_y), (self.c_max_x, self.c_max_y))
          if p!=None: # auch hier ist es nie eine Waagerechte (also immer p!=None)
            lst.append(p)           # zusätzlichen Eintritts-Punkt einfügen
          lst.append(poli_lst[v])   # und auch den gemessenen Punkt
          comesFromOutside=False
        else:
          lst.append(poli_lst[v])   # alles gut -> Punkt kopieren
    return(lst)

  # ###########################################################
  # Liefert aus den zwei übergebenen Kurven eine neue Kurve,
  # die der Überlagerung der beiden Kurven entspricht.
  # l0 und l1 müssen gleich lang sein und jew. identische
  # X-Koordinaten haben.
  # Bei hoher Zoom-Stufe würde es "Nasen" an Wechselstellen
  # geben. Daher wird, wenn die kleinere Kurve wechselt, am
  # Schnittpunkt der Kurven ein zusätzlicher Punkt in die
  # Überlagerungs-Kurve eingefügt.
  def overlapping(self, l0, l1):
    s1=l0[1][1]>l1[1][1]          # Init der Wechsel-Erkennung
    d=[l0[0]]                     # Startpunkt 
    for x in range(1, len(l0)-1): # über alle ohne Start- und Schließ-Punkt
      s2=l0[x][1]>l1[x][1]        # aktuell größere Kurve
      if s1!=s2:                  # wenn Wechsel der größeren Kurve erkannt wurde
        ip=self.intersectionPoint(l0[x-1], l0[x], l1[x-1], l1[x]) # Schnittpunkt berechnen
        if ip!=None:              # None kann zwar nicht vorkommen...aber abprüfen schadet nix
          d.append(ip)            # zusätzlichen Punkt einfügen
      s1=s2                       # Wechsel-Kennung für nächsten Iterationsschritt umkopieren
      k=max(l0[x][1], l1[x][1])   # der größere Y-Wert beider Kurven...
      d.append((l0[x][0], k))     # ...ist die kleinere Kurve (weil 0,0 oben links ist)
    d.append(l0[len(l0)-1])       # Schließpunkt anhängen
    return(d)

  # ###########################################################
  # Liefert den Schnittpunkt der beiden Geraden xy12 und xy34
  # oder None, wenn die Geraden parallel verlaufen.
  # Quelle: http://en.wikipedia.org/wiki/Line%E2%80%93line_intersection
  def intersectionPoint(self, (x1, y1), (x2, y2), (x3, y3), (x4, y4)):
    t1=x1*y2-y1*x2
    t2=x3*y4-y3*x4
    n=(x1-x2)*(y3-y4)-(y1-y2)*(x3-x4)
    if n==0.0:  # parallel -> y3==y4 und y1==y2 -> n=0
      return(None)
    x=(t1*(x3-x4)-(x1-x2)*t2)/n
    y=(t1*(y3-y4)-(y1-y2)*t2)/n
    return((x, y))




# ###########################################################
# Der Fenster-Rahmen für das Hauptfenster.
class TunBwFrame(wx.Frame):
  def __init__(self, parent, pos=wx.DefaultPosition, size=wx.DefaultSize):
    style=wx.DEFAULT_FRAME_STYLE
    wx.Frame.__init__(self, None, wx.ID_ANY, "tun0graf "+VERSION, pos=pos, size=size, style=style)
    self.panel=TunBwWindow(self)




# ###########################################################
# Ein Dialog-Fenster mit einer Listbox.
class ListBoxDialog(wx.Dialog):
  def __init__(self, parent, title, size, sel_list):
    super(ListBoxDialog, self).__init__(parent, title=title, 
            size=size,
            style=wx.RESIZE_BORDER|wx.CLOSE_BOX)
    self.parent=parent
    self.sel_list=sel_list
    self.selectedIdx=None

    self.InitUI()
    self.Centre()

  # ###########################################################
  # GUI anlegen.
  def InitUI(self):
    self.lst=wx.ListBox(self, id=10)
    self.lst.Bind(wx.EVT_LISTBOX_DCLICK, self.onListSel)

    self.lst.SetItems(self.sel_list)
    self.lst.SetSelection(0)
    self.lst.Bind(wx.EVT_KEY_DOWN, self.onKeyDown)

    sizer=wx.BoxSizer(wx.VERTICAL)
    sizer.Add(self.lst, 1, wx.EXPAND, 5)

    self.SetSizer(sizer)
    self.lst.SetFocus()

  # ###########################################################
  # Wird bei Enter oder Doppelklick in ListBox aufgerufen.
  def onListSel(self, event):
    self.selectedIdx=self.lst.GetSelection()
    self.EndModal(wx.ID_OK)

  # ###########################################################
  # Liefert den Index des gewählten Eintrages an den Aufrufer.
  def getSelectedIdx(self):
    return(self.selectedIdx)

  # ###########################################################
  # Beendet den Dialog bei ESC-Taste.
  def onKeyDown(self, event):
    key=event.GetKeyCode()
    if key==wx.WXK_ESCAPE:
      self.EndModal(wx.ID_CANCEL)
    else:
      event.Skip()




# ###########################################################
# Setzt den ArgParser auf.
def setupArgParser():
  desc="tun0graf v"+VERSION+" - graphical display of a logfile created by tun0moni.py"
  usage="%(prog)s [-h] [logfilename [-s snapname]] [-z size] [-c S|R|O]"
  parser=argparse.ArgumentParser(description=desc, usage=usage)
  parser.add_argument('logfilename', nargs='?', 
                    help='name of the logfile to be loaded')
  parser.add_argument('-s', dest='snapname', nargs=1,
                    help='create snapshot of logfilename and save it as snapname.png')
  parser.add_argument('-z', dest='size', nargs=1, 
                    help='window size as X,Y with X>=150 and Y>=120')
  parser.add_argument('-c', dest='curve', nargs=1, 
                    choices=['SRO', 'SR', 'SO', 'S', 'RO', 'R', 'O'],
                    help='show curves (S=send, R=receive, O=overlap)')
  args=parser.parse_args()

  if args.snapname!=None and args.logfilename==None:
    parser.error('SNAPNAME can only be used with logfilename')

  if args.size!=None: # wenn -z angegeben -> Format prüfen
    bad=True
    if len(args.size)==1:         # genau ein Parameter
      s=args.size[0].split(",")   # getrennt durch Komma
      if len(s)==2:               # genau ein Komma
        if s[0].isdigit()==True and s[1].isdigit()==True: # nur Ziffern
          if int(s[0])>=150 or int(s[1])>=120:  # Mindestgröße
            args.size=(int(s[0]), int(s[1]))    # als Tupel von int übergeben
            bad=False
    if bad==True:
      parser.error('illegal size was specified. Format is x,y (minimum 150,120)')
    
  return(args)


# ###########################################################
# Main-Loop
if __name__=='__main__':
  args=setupArgParser()
  if args.logfilename!=None:
    filename_g=args.logfilename
  if args.snapname!=None:
    snapname_g=args.snapname[0]
  if args.curve!=None:
    curves_g=args.curve[0]

  fc=wx.FileConfig(localFilename=CFGFILE)
  spx=fc.ReadInt("pos_x", -1)
  spy=fc.ReadInt("pos_y", -1)
  ssx=fc.ReadInt("size_x", 1450)
  ssy=fc.ReadInt("size_y", 250)
  sp=(spx, spy) # (-1, -1) entspricht wx.DefaultPosition
  ss=(ssx, ssy) # (-1, -1) entspricht wx.DefaultSize
  del fc

  if args.size!=None:
    ss=args.size  # cmd-Parameter überstimmt cfg-File

  app=wx.App(False)
  frame=TunBwFrame(None, pos=sp, size=ss)
  frame.Show(True)
  app.MainLoop()

