#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Kommunikation mit der Rollladen-Steuerung. Version 3
#
# Detlev Ahlgrimm, 30.10.2015
#
# Quelle für Grundgerüst:
# http://www.ibm.com/developerworks/linux/tutorials/l-pysocks/#N10518
#
import socket
import select
import random
import hashlib
import time
import datetime

password_ATmega="Passw0rd_M"  # Challenge kommt vom ATmega
password_Python="Passw0rd_P"  # Challenge kommt vom Python-Script

DEBUG=False
#DEBUG=True

fn_zeiten_tabelle="/home/dede/daten/Rollotron-Zeiten.txt"
fn_logfile="/home/dede/rollotron_log.txt"

# ###########################################################
# Liefert die Zeiten aus der Zeitentabelle als String mit
# 12*8=96 Zeichen.
def getTimesList():
  with open(fn_zeiten_tabelle, 'r') as fl:
    lines=fl.readlines()

  m_str=""
  for ln in lines:
    lns=ln.strip()
    if lns=="" or lns.startswith("//"):
      continue
    lln=lns.replace(":", " ").replace("-", " ").split()
    for i in range(8):
      m_str+=chr(int(lln[i])+ord("0"))
  return(m_str)

# ###########################################################
# Liefert für eine gepackte Zeiten-Liste die entsprechend
# entpackte Version als Liste von 12 Strings.
def formatTimeList(strg):
  frmt=[":", "-", ":", "   ", ":", "-", ":", ""]
  lst=[]
  stro=""
  for i in xrange(len(strg)):
    stro+="%02d"%(ord(strg[i])-ord("0"),)+frmt[i%8]
    if i%8==7:
      lst.append(stro)
      stro=""
  return(lst)

# ###########################################################
# Liefert für zwei gepackte Zeiten-Listen ein Set von
# Integers der Monate mit Änderungen.
def findDifferences(strg1, strg2):
  ret_set=set()
  for i in range(len(strg1)):
    if strg1[i]!=strg2[i]:
      ret_set.add(i//8)   # 8 Werte pro Monat
  return(ret_set)

# ###########################################################
# Liefert für zwei gepackte Zeiten-Listen die Werte für die
# Monate, in denen die beiden Zeiten-Listen voneinander
# abweichen als Liste aus Strings.
def showDifferences(strgATmega, strgFile, indent=19):
  m=[ "Januar", "Februar", "März", "April", "Mai", "Juni", "Juli",
      "August", "September", "Oktober", "November", "Dezember"  ]
  ret_lst=[]
  d=findDifferences(strgATmega, strgFile)
  if len(d)>0:
    l1=formatTimeList(strgATmega)
    l2=formatTimeList(strgFile)
    if len(l1)==len(l2):
      for i in d:
        ret_lst.append("-"*25 + " " + m[i])
        ret_lst.append(l1[i] + "   (ATmega)")
        ret_lst.append(l2[i] + "   (Datei)")
  return(ret_lst)

# ###########################################################
# Liefert Datum+Uhrzeit immer als Winter- bzw. Normalzeit.
def getNormTime():
  now=datetime.datetime.now()
  if time.localtime().tm_isdst==1:
    now=now+datetime.timedelta(hours=-1)
  return(now.strftime("%Y.%m.%d %H:%M:%S"))

# ###########################################################
# Liefert die Differenz [in Sekunden] zwischen
# "datetime_strg" und der aktuellen Normal-Zeit.
def datetime_dif(datetime_strg):
  t1=datetime.datetime.strptime(datetime_strg, "%Y.%m.%d %H:%M:%S")
  t2=datetime.datetime.strptime(getNormTime(), "%Y.%m.%d %H:%M:%S")
  return((t2-t1).total_seconds())

# ###########################################################
# Liefert True, wenn "checksum" auf "strg" passt.
# Sonst False.
def testChecksum(strg, checksum):
  cs=0
  for c in strg:
    cs+=ord(c)
  if "%0.4x"%cs==checksum:
    return(True)
  return(False)

# ###########################################################
# Liefert die Checksumme zu "strg" als 4-Byte-Hex-String.
def getChecksum(strg):
  cs=0
  for c in strg:
    cs+=ord(c)
  return("%0.4x"%cs)




class SocketServer:
  def __init__(self, port):
    self.port=port;
    self.srvsock=socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    self.srvsock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    self.srvsock.bind(("", port))
    self.srvsock.listen(5)
    self.descriptors=[self.srvsock]
    print 'SocketServer started on port %s'%port
    self.indent=len(self.curTime())
    self.warn_str="!!!!!!!!!!! Warnung !!!!!!!!!!!"

  # ###########################################################
  # Hauptschleife des SocketServers.
  # Kann nur einen Client zur Zeit verarbeiten (sonst müsste
  # "state" pro Client verwaltet werden)!
  def run(self):
    state="WAIT"
    timeout=3  # 3 Sekunden ohne Kommunikation führt zu disconnect

    while 1:
      (sread, swrite, sexc)=select.select(self.descriptors, [], [], timeout)

      # - - - - - - - - - - - - - - - - -
      # Timeout verarbeiten
      # - - - - - - - - - - - - - - - - -
      if sread==[]:                         # Timeout
        if len(self.descriptors)>1:         # wenn Clients verbunden
          for sock in self.descriptors:     # über alle Clients
            if sock!=self.srvsock:          # ohne den Server selbst
              self.write_log("Timeout...")
              host, port=sock.getpeername()
              self.close_connection(sock, host, port)
              state="WAIT"
        continue

      for sock in sread:
        # - - - - - - - - - - - - - - - - -
        # neuen Connect verarbeiten
        # - - - - - - - - - - - - - - - - -
        if sock==self.srvsock:        # neuer Connect
          self.accept_new_connection()
        else:                         # Empfang von verbundenem Client 
          # - - - - - - - - - - - - - - - - -
          # empfangene Daten verarbeiten
          # - - - - - - - - - - - - - - - - -
          try:
            strg=sock.recv(1024)      # Kommando von Client empfangen
          except:
            strg=""                   # Client bei Fehler rausschmeißen
          host, port=sock.getpeername()

          # - - - - - - - - - - - - - - - - -
          # Disconnect verarbeiten
          # - - - - - - - - - - - - - - - - -
          if strg=='':                # Disconnect des Clients
            self.close_connection(sock, host, port)
            state="WAIT"
          else:                       # Client sendet Daten
            # - - - - - - - - - - - - - - - - -
            # Checksumme prüfen
            # - - - - - - - - - - - - - - - - -
            if DEBUG: print "raw:", strg
            checksum=strg[len(strg)-4:]   # die letzten 5 Zeichen sind ","+Checksum in Hex
            strg=strg[:len(strg)-5]
            if testChecksum(strg, checksum):
              if DEBUG: print "\nreceived(good):", strg
            else:
              if DEBUG: print "\nreceived(bad ):", strg
              self.send_string("BAD CHECKSUM")
              continue

            # - - - - - - - - - - - - - - - - -
            # Kommandos verarbeiten
            # - - - - - - - - - - - - - - - - -
            # in:"AUTH_REQ" -> out:challenge
            if state=="WAIT" and strg=="AUTH_REQ":
              self.rand_number=str(random.randint(100000000, 999999999))
              if DEBUG: print "\nsende challenge"
              self.send_string(self.rand_number)
              state="AUTH_REQ_CHALLENGE_SENT"

            # in:"UNAUTH" -> out:"OK"
            elif strg.startswith("UNAUTH"):
              self.send_string("OK")  # was zu senden spart drüben den Timeout
              state="WAIT"

            # in:MD5-Hash -> out:"[NOT] AUTHORIZED"
            elif state=="AUTH_REQ_CHALLENGE_SENT":
              self.md5hash=strg
              md5=hashlib.md5(self.rand_number+password_Python).hexdigest()
              if DEBUG: print "my md5 = ", md5
              if md5==self.md5hash:
                state="AUTHORIZED_GOOD_HASH_RECEIVED"
                self.send_string("AUTHORIZED")
              else:
                self.send_string("NOT AUTHORIZED")
                self.close_connection(sock, host, port)
                state="WAIT"

            # in:challenge -> out:MD5-Hash
            elif state=="AUTHORIZED_GOOD_HASH_RECEIVED":
              self.rand_number=strg
              md5=hashlib.md5(self.rand_number+password_ATmega).hexdigest()
              self.send_string(md5)
              state="AUTHORIZED_HASH_FOR_CHALLENGE_SENT"

            # in:"[NOT] AUTHORIZED" -> out:"FINE"|"BYE"
            elif state=="AUTHORIZED_HASH_FOR_CHALLENGE_SENT":
              if strg=="AUTHORIZED":
                self.send_string("FINE")
                state="AUTHORIZED"
                self.write_log("Autorisierung abgeschlossen")
              else:
                self.send_string("BYE")
                self.close_connection(sock, host, port)
                state="WAIT"

            # in:"TIME=YYYY.MM.DD HH:MM:SS" -> out:"OK"
            elif state=="AUTHORIZED" and strg.startswith("TIME="):
              v=strg.split("=", 1)
              if len(v)>1:
                try:
                  rtc_dif=datetime_dif(v[1])
                  tmp_str="Abweichung [sek]: %d"%(rtc_dif)
                  if rtc_dif>3:
                    self.warning_write_log_INT(tmp_str)
                  else:
                    self.write_log(tmp_str)
                except:
                  self.write_log(v[1])
                  self.warning_write_log_INT("Datum/Zeit ist illegal")
                self.send_string("OK")

            # in:"GET_TIME" -> out:Linux-time
            elif strg=="GET_TIME":    # die Zeit gibts auch ohne Autorisierung
              self.send_string(getNormTime())
              self.write_log("Datum/Zeit übermittelt")

            # in:"LIST=<96 Zeichen>" -> out:"LATEST"|"UPDATE"
            elif state=="AUTHORIZED" and strg.startswith("LIST="):
              v=strg.split("=", 1)
              if len(v)>1:
                t_str=getTimesList()
                if t_str==v[1]:
                  self.send_string("LATEST")
                  self.write_log("Zeiten-Tabelle ist aktuell")
                else:
                  self.send_string("UPDATE")
                  lst=showDifferences(v[1], t_str, self.indent)
                  for l in lst:
                    self.write_log(l)

            # in:"GET_TIMELIST" -> out:Linux-Zeiten-Liste
            elif state=="AUTHORIZED" and strg=="GET_TIMELIST":
              t_str=getTimesList()
              self.send_string(t_str)
              self.write_log("geänderte Zeiten-Tabelle übermittelt")

            # in:"WARNING=###" -> out:"OK"
            elif state=="AUTHORIZED" and strg.startswith("WARNING="):
              self.warning_write_log_EXT(strg)
              self.send_string("OK")

            # in:"VERSION=###" -> out:"OK"
            elif state=="AUTHORIZED" and strg.startswith("VERSION="):
              v=strg.split("=", 1)
              if len(v)>1:
                self.write_log("Firmware-Version: %s"%(v[1]))
              self.send_string("OK")

            # in:"VOLTAGE=###" -> out:"OK"
            elif state=="AUTHORIZED" and strg.startswith("VOLTAGE="):
              v=strg.split("=", 1)
              if len(v)>1:
                self.write_log("Akku-Spannung: %sV"%(v[1]))
              self.send_string("OK")

            # in:"INFO=###" -> out:"OK"
            elif strg.startswith("INFO="):
              v=strg.split("=", 1)
              if len(v)>1:
                self.write_log("Info: %s"%(v[1]))
              self.send_string("OK")

            # in:<etwas anderes> -> out:"ECHO:"+empfangener String
            else:
              self.send_string("ECHO:"+strg)
              self.write_log("unbekannte Anforderung: %s"%(strg))

  # ###########################################################
  # Sendet "strg" + "," + 4-Byte-Checksum an alle verbundenen
  # Clients (außer sich selbst).
  def send_string(self, strg):
    cs=getChecksum(strg)
    for sock in self.descriptors:
      if sock!=self.srvsock:
        sock.send(strg+","+cs)
    if DEBUG: print "sent:", strg+","+cs

  # ###########################################################
  # Nimmt einen neuen Client an und in die Liste der
  # verbundenen Clients auf.
  def accept_new_connection(self):
    newsock, (remhost, remport)=self.srvsock.accept()
    self.descriptors.append(newsock)
    self.write_log('Verbindung hergestellt %s (%s:%s)'% \
                  (self.get_hostname(remhost), remhost, remport))

  # ###########################################################
  # Schließt die Verbindung zum Client gemäß "sock".
  def close_connection(self, sock, host="", port=""):
    sock.close()
    self.descriptors.remove(sock)
    self.write_log('Verbindung geschlossen %s (%s:%s)\n'% \
                  (self.get_hostname(host), host, port))

  # ###########################################################
  # Liefert den Hostname zur IP-Adresse "ipaddr".
  def get_hostname(self, ipaddr):
    fqdn=socket.gethostbyaddr(ipaddr)
    if len(fqdn)>1:
      hn=fqdn[0].split('.')
      return(hn[0])
    return("")

  # ###########################################################
  # Liefert die Ortszeit als String.
  def curTime(self):
    return(datetime.datetime.now().strftime("%Y.%m.%d %H:%M:%S"))

  # ###########################################################
  # Gibt je nach Nummer in WARNING.### einen entsprechenden
  # Text aus.
  def warning_write_log_EXT(self, strg):
    msg_lst=[ (  1, "Schaltung hat neu gestartet"),
              (  2, "Falsche Prüfsumme in EEPROM-Bereich 1"), 
              (  4, "Falsche Prüfsumme in EEPROM-Bereich 2"),
              (  8, "RTC-Daten waren illegal, neu gesetzt"),
              ( 16, "RTC-Daten waren illegal, neu setzen nicht möglich")    ]
    w=strg.split("=", 1)
    if len(w)==2:
      if w[1].isdigit():
        wnr=int(w[1])
        for i in range(len(msg_lst)):
          if wnr&msg_lst[i][0]>0:
            self.warning_write_log_INT(msg_lst[i][1])

  # ###########################################################
  # Gibt "strg" zwischen zwei Zeilen mit dem Inhalt von
  # "self.warn_str" aus.
  def warning_write_log_INT(self, strg):
    self.write_log(self.warn_str)
    self.write_log(strg)
    self.write_log(self.warn_str)
    
  # ###########################################################
  # Schreibt Ortszeit und "strg" aufs Terminal.
  def write_log(self, strg):
    with open(fn_logfile, 'a') as fl:
      fl.write(self.curTime() + " " + strg + "\n")
    print self.curTime(), strg
    

if __name__=='__main__':
  myServer=SocketServer(2626)
  myServer.run()
