#!/usr/bin/env python3
# -*- coding: UTF-8 -*-

import os, sys
import time
import datetime
import io
import urllib3
import httplib2
from PIL import Image, ImageFilter, ImageMath
import numpy as np


# ###########################################################
# Liefert von dem Grauwert-Bild "arr" die Koordinaten des
# hellsten Bereiches.
# Das Bild wird dazu in 16 Bereiche gekachelt.
# Die Koordinaten (0, 0) liegen links oben im Bild.
def findBrightestArea(arr):
  #arr=np.asarray(img)
  arrsh=np.hsplit(arr, 4)       # horizontal in 4 Blöcke zerlegen
  m=list()
  for x in range(len(arrsh)):   # über alle 4 Blöcke
    m.append(np.sum(arrsh[x]))  # Summe des Blocks merken
  mh=m.index(max(m))            # Index des Blocks mit der größten Summe

  m=list()
  arrsv=np.vsplit(arrsh[mh], 4) # hellsten Block vertikal in 4 Blöcke zerlegen
  for y in range(len(arrsv)):
    m.append(np.sum(arrsv[y]))
  mv=m.index(max(m))
  return(mh, mv)

# ###########################################################
#
def findBrightestArea7x5(arr):
  cx=7  # Anzahl Elemente auf der X-Achse
  cy=5  # Anzahl Elemente auf der Y-Achse
  w=91  # Breite eines Elements
  h=96  # Höhe eines Elements

  max_seg=None
  max_seg_idx_x=None
  max_seg_sum=0
  for x in range(cx): # über die Segmente auf der X-Achse
    seg=arr[:, 1+x*w:1+x*w+w]
    tmp=np.sum(seg)
    if tmp>max_seg_sum:
      max_seg=seg       # hellstes Segment auf der X-Achse
      max_seg_idx_x=x   # X-Index merken
      max_seg_sum=tmp   # dessen Helligkeits-Wert

  max_seg_idx_y=None
  max_seg_sum=0
  for y in range(cy): # über die Segmente auf der Y-Achse
    seg=max_seg[y*h:y*h+h, :]
    tmp=np.sum(seg)
    if tmp>max_seg_sum:
      max_seg_sum=tmp
      max_seg_idx_y=y
  return(max_seg_idx_x, max_seg_idx_y)

# ###########################################################
# Liefert 35 Tupel aus (Grauwert, x, y) für das in eine 
# 7x5-Matrix zerlegte Bild "arr" als Liste.
# Die Rückgabe-Liste ist absteigend nach Grauwert sortiert.
def areasByBrightness(arr):
  cx=7  # Anzahl Elemente auf der X-Achse
  cy=5  # Anzahl Elemente auf der Y-Achse
  w=91  # Breite eines Elements
  h=96  # Höhe eines Elements

  ret_lst=list()
  for x in range(cx): # über die Segmente auf der X-Achse
    for y in range(cy): # über die Segmente auf der Y-Achse
      seg=arr[y*h:y*h+h, 1+x*w:1+x*w+w]
      br=np.sum(seg)
      ret_lst.append((br, x, y))
  f=lambda x:x[0]
  return(sorted(ret_lst, key=f, reverse=True))

# ###########################################################
#
class IP_Cam:
  def __init__(self):
    self.CAM_AUTH="camuser:pwd4Cam"
    self.snapshotURL="192.168.178.36/snapshot.cgi"
    self.ctrlURL="192.168.178.36/decoder_control.cgi?command=%d"

    self.http=urllib3.PoolManager()
    self.headers=urllib3.util.make_headers(basic_auth=self.CAM_AUTH)
    self.UP=0
    self.DOWN=2
    self.LEFT=4
    self.RIGHT=6

    self.STOP=1
    self.LEFT_UP=90
    self.RIGHT_UP=91
    self.LEFT_DOWN=92
    self.RIGHT_DOWN=93

  # ###########################################################
  #
  def setLight(self, v):
    url="192.168.178.36/set_misc.cgi?next_url=light.htm&led_mode=%d"
    r=self.http.request('GET', url%(v), headers=self.headers)

  # ###########################################################
  # Liefert einen Snapshot als Pillow-Image.
  def getImage(self):
    r=self.http.request('GET', self.snapshotURL, headers=self.headers)
    image=Image.open(io.BytesIO(r.data))
    return(image)

  # ###########################################################
  # Liefert einen Snapshot aus Grauwerten als NumPy-Array.
  def getImageGrayscale(self):
    try:
      img=self.getImage().convert('L')
    except:
      #print("error - retry")
      img=self.getImage().convert('L')
    return(np.asarray(img))

  # ###########################################################
  # Ersetzt in dem NumPy-Array "arr" alle Werte mit 0, die
  # entweder kleiner als "val" oder größer als 256-"val" sind.
  def removeNoise(self, arr, val=32):
    arr2=1*arr    # deep-copy erzeugen
    arr2[arr2>(256-val)]=0
    arr2[arr2<val]=0
    return(arr2)

  # ###########################################################
  # Bewegt die Kamera "duration" Sekunden lang in die Richtung
  # "direction".
  def gotoDirection(self, direction, duration):
    r=self.http.request('GET', self.ctrlURL%(direction), headers=self.headers)
    if duration<0:
      return
    elif duration>0:
      #print("sleep")
      time.sleep(duration)
    else:
      print("duration=%d"%(duration,))
    r=self.http.request('GET', self.ctrlURL%(1), headers=self.headers)

  # ###########################################################
  # Bewegt die Kamera in Grundstellung.
  def gotoCenter(self):
    r=self.http.request('GET', self.ctrlURL%(25), headers=self.headers)
    while not self.hasStopped():
      time.sleep(0.5)

  # ###########################################################
  # Speichert das Bild "imgAsArray" unter dem Dateinamen
  # "filename".
  def saveImage(self, imgAsArray, filename):
    img=Image.fromarray(imgAsArray)
    img.save("p/%s.jpg"%(filename,))

  # ###########################################################
  # Liefert für ein Grau-Bild in "imgAsArray" eine Liste mit
  # 64 Werten als Pattern. Die Werte entsprechen dem Mittelwert
  # des jeweiligen Bildausschnittes.
  def getImagePattern(self, imgAsArray):
    pat=np.empty([8, 8], dtype=np.int8) # Array anlegen
    arrsh=np.hsplit(imgAsArray, 8)      # Bild horizontal splitten
    for h in range(len(arrsh)):         # über 8 Teilbilder
      arrsv=np.vsplit(arrsh[h], 8)      # Teilbild vertikal splitten
      for v in range(len(arrsv)):       # über 8 Teilbilder
        #self.saveImage(arrsv[v], "b%d%d"%(h,v))
        pat[v, h]=np.mean(arrsv[v])     # Bild-Mittelwert ins Array[y,x]
    return(pat)                         # Array als Pattern liefern

  # ###########################################################
  # Vergleicht die zwei Pattern "pat1" und "pat2" und liefert
  # True, wenn der mittlerer Grauwert jedes Bildausschnitts
  # in beiden Pattern maximal um "maxdelta" voneinander
  # abweicht.
  def compareImagePattern(self, pat1, pat2, maxdelta=2):
    return(np.max(np.absolute(pat1-pat2))<=maxdelta)

  # ###########################################################
  # Vergleicht die zwei Pattern "pat1" und "pat2" und liefert
  # ein Tupel aus Summe der Abweichungen und höchter
  # Abweichung in einem Bildausschnitt.
  def compareImagePatternDetail(self, pat1, pat2):
    tmp=np.absolute(pat1-pat2)
    return(np.sum(tmp), np.max(tmp))

  # ###########################################################
  # Liefert True, wenn zwei Bilder, die 100ms nacheinander
  # von der Kamera gelesen wurden, annähernd gleich sind.
  def hasStopped(self):
    p1=cam.getImagePattern(cam.getImageGrayscale())
    time.sleep(0.1)
    p2=cam.getImagePattern(cam.getImageGrayscale())
    return(self.compareImagePattern(p1, p2))

  # ###########################################################
  #
  def gotoMatrixElement(self, x, y):
    #  0,0  1,0  2,0  3,0  4,0  5,0  6,0
    #  0,1  1,1  2,1  3,1  4,1  5,1  6,1
    #  0,2  1,2  2,2 (3,2) 4,2  5,2  6,2
    #  0,3  1,3  2,3  3,3  4,3  5,3  6,3
    #  0,4  1,4  2,4  3,4  4,4  5,4  6,4
    nam={0:"UP   ", 2:"DOWN ", 4:"LEFT ", 6:"RIGHT", None:""}

    if   y<2: dv=self.UP
    elif y>2: dv=self.DOWN
    else:     dv=None

    if   x<3: dh=self.LEFT
    elif x>3: dh=self.RIGHT
    else:     dh=None

    tv=0
    if not dv is None:
      if   y in (0, 4): tv=0.581
      elif y in (1, 3): tv=0.325

    th=0
    if not dh is None:
      if   x in (0, 6): th=2.63
      elif x in (1, 5): th=1.72
      elif x in (2, 4): th=0.93

    #print(x, y, nam[dv], nam[dh], tv, th)

    if not dv is None and not dh is None:
      # wenn beide Achsen bewegt werden sollen
      if dv==self.UP:
        if dh==self.LEFT:
          if tv>th: # mehr vertikal
            self.gotoDirection(self.LEFT_UP, th)
            self.gotoDirection(self.UP, tv-th)
          else:     # mehr horizontal
            self.gotoDirection(self.LEFT_UP, tv)
            self.gotoDirection(self.LEFT, th-tv)
        else: # dh==self.RIGHT
          if tv>th: # mehr vertikal
            self.gotoDirection(self.RIGHT_UP, th)
            self.gotoDirection(self.UP, tv-th)
          else:     # mehr horizontal
            self.gotoDirection(self.RIGHT_UP, tv)
            self.gotoDirection(self.RIGHT, th-tv)
      else: # dv==self.DOWN
        if dh==self.LEFT:
          if tv>th: # mehr vertikal
            self.gotoDirection(self.LEFT_DOWN, th)
            self.gotoDirection(self.DOWN, tv-th)
          else:     # mehr horizontal
            self.gotoDirection(self.LEFT_DOWN, tv)
            self.gotoDirection(self.LEFT, th-tv)
        else: # dh==self.RIGHT
          if tv>th: # mehr vertikal
            self.gotoDirection(self.RIGHT_DOWN, th)
            self.gotoDirection(self.DOWN, tv-th)
          else:     # mehr horizontal
            self.gotoDirection(self.RIGHT_DOWN, tv)
            self.gotoDirection(self.RIGHT, th-tv)
    else:
      if not dv is None:
        self.gotoDirection(dv, tv)
      if not dh is None:
        self.gotoDirection(dh, th)

# ###########################################################
# Liefert die Anzahl der Listenelemente aus "lst", die über
# dem Wert "above" liegen. Die Liste muss aus Dreier-Tupeln
# bestehen, der erste Wert des Tupels wird gegen "above"
# geprüft.
def countHigherThan(lst, above):
  cnt=0
  for a, x, y in lst:
    if a>above:
      cnt+=1
  return(cnt)

# ###########################################################
# Main
if __name__=="__main__":
  DEBUG=False
  nervous=True   # False für weniger Bewegungen
  cam=IP_Cam()

  img1=cam.getImageGrayscale()
  while True:
    movement_enable=True  # False für keine neue Positionierung aber DEBUG-Bearbeitung
    time.sleep(0.1)
    img2=cam.getImageGrayscale()

    diff=cam.removeNoise(img1-img2)
    diff_sum=np.sum(diff)

    if diff_sum<500000:   # keine Bewegung im Bild
      continue

    diff_sum_p="{:,}".format(diff_sum)
    t=datetime.datetime.utcnow().strftime("%Y%m%d-%H%M%S-%f")
    l=areasByBrightness(diff)
    c100K=countHigherThan(l, 100000)
    c5K=countHigherThan(l, 5000)
    a, x, y=l[0]

    if a<50000: # das hellste Element ist nicht hell genug
      continue

    if not nervous:
      if x in (1, 2, 3, 4, 5) and y in (1, 2, 3):
        movement_enable=False

    if diff_sum>4000000:  # zu viel "Bewegung" im Bild
      movement_enable=False

    if c100K>=12:         # in zu vielen Elementen ist Bewegung
      movement_enable=False

    if c5K>28: # zu viele Elemente oberhalb vom Grundrauschen
      movement_enable=False

    if DEBUG:
      print("\n%10s %d %d (%d, %d) %s"%(diff_sum_p, x, y, c100K, c5K, str(l)))
      cam.saveImage(img1, "d_%s-1"%(t))
      cam.saveImage(img2, "d_%s-2"%(t))
      cam.saveImage(diff, "d_%s-d(%d,%d)-%s-%d-%d"%(t, x, y, diff_sum_p, c100K, c5K))

    if movement_enable:
      cam.gotoMatrixElement(x, y)
      time.sleep(0.2)

    img2=cam.getImageGrayscale()
    img1=img2
