#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ###########################################################
# Ein Programm zur Kommunikation mit der
#   Zirkulationspumpen-Steuerung.
#
# Es wird ein Socket geöffnet und auf Anfragen gewartet.
# Die übermittelten Sensor-Daten werden in einer Datenbank
# gespeichert.
#
# Detlev Ahlgrimm, 2017
#

import sys
import os
import socket
import select
import time
import datetime
import sqlite3

FILENAME_LOGFILE    ="/home/dede/nzps_log.txt"
DATABASE_FILENAME   ="/nfs/nzps.sqlite"

# E: Warmwasser-Entnahme  (Rohr 1)
# L: Legionellenfunktion  (Rohr 2)
# Z: Zirkulation          (Rohr 3)
# R: Raumtemperatur       (Rohr 4)
sensors_adr={ "28ff94ba021704c6" : "E",
              "28ff01d20217042f" : "L",
              "28ff9e25031704ab" : "Z",
#              "28ff21e87316055a" : "R"  }
              "28ff28b9021703a3" : "R"  }
#28ff94ba021704c6,28ff9e25031704ab,28ff01d20217042f,28ff21e87316055a
#sensors_adr={ "28ff28b9021703a3" : "E",
#              "28ffd661751603eb" : "Z",
#              "28ffce3c751603ec" : "L",
#              "28ff291b8116055f" : "R"  }
#28ff28b9021703a3,28ffd661751603eb,28ffce3c751603ec,28ff291b8116055f - debug an WeMos-3

# maximal drei SSIDs mit den Nummer "0" bis "2"
ssids_nam={ "FreifunkWees01.0 (http://ffw)" :"0", 
            "FreifunkWees01.2 (http://ffw)" :"1", 
            "FreifunkWees01.3 (http://ffw)" :"2" }

socketServerIP_val={  "192.168.42.80"  :"0", 
                      ""               :"1" }

socketServerPort_val={  "2629"  :"0", 
                        "0"     :"1" }

syncTimes=[ ( 22,  15),     # 03:15 / 255 für "ungültig"
            ( 22,  20),     # die Anzahl der Elemente muss zu NUMBER_SYNCTIMES in der FW passen!
            ( 22,  30),
            (  3,  15),
            ( 22,  40),
            ( 22,  50),
            ( 25,  40),
            (  7,   0),
            (  8,   0),
            (  9,   0) ]

thresholds=[  45,   # 00: 45°C ab dieser Temperatur an Rohr Z kann die Pumpe im Normal-Modus abgeschaltet werden
              56,   # 01: 56°C ab dieser Temperatur an Rohr E wird Legionellen-Modus angenommen und die Pumpe eingeschaltet
              60,   # 02: 60°C ab dieser Temperatur an Rohr L wird Legionellen-Modus angenommen und die Pumpe eingeschaltet
              52,   # 03: 52°C ab dieser Temperatur an Rohr Z wird im Legionellen-Modus die Pumpe abgeschaltet
              0,    # 04:
              0,    # 05:
              0,    # 06:
              0,    # 07:
              0,    # 08:
              0,    # 09:
              0,    # 10:
              0,    # 11:
              0,    # 12:
              0,    # 13:
              0,    # 14:
              0 ]   # 15:

sensors_nam=          { v: k for k, v in sensors_adr.items() }
ssids_num=            { v: k for k, v in ssids_nam.items() }
socketServerIP_num=   { v: k for k, v in socketServerIP_val.items() }
socketServerPort_num= { v: k for k, v in socketServerPort_val.items() }



class Database():
  def __init__(self, dbname):
    if os.path.isfile(dbname)==False:
      initDB=True
    else:
      initDB=False

    self.connection=sqlite3.connect(dbname)
    self.cursor=self.connection.cursor()

    if initDB==True:
      self.createDatabase()

  # ###########################################################
  # Legt die Tabellen so an, dass LoggerView direkt darauf
  # zugreifen kann.
  def createDatabase(self):
    self.cursor.execute("CREATE TABLE  Host"                           \
                        " (id          INTEGER NOT NULL PRIMARY KEY,"  \
                        "  hostname    VARCHAR NOT NULL,"              \
                        "  ip_addr     VARCHAR NOT NULL,"              \
                        "  firmware    VARCHAR NOT NULL,"              \
                        "  sensor_cnt  INTEGER NOT NULL,"              \
                        "  last_reboot VARCHAR,"                       \
                        "  comment     VARCHAR)")
    self.cursor.execute("CREATE UNIQUE INDEX nodupe1 ON Host (hostname)")

    self.cursor.execute("CREATE TABLE  Location"                       \
                        " (id          INTEGER NOT NULL PRIMARY KEY,"  \
                        "  host        INTEGER NOT NULL,"              \
                        "  sensor      INTEGER NOT NULL,"              \
                        "  location    VARCHAR,"                       \
                        "  valid_from  VARCHAR,"                       \
                        "  valid_till  VARCHAR,"                       \
                        "  FOREIGN KEY(host) REFERENCES Host(id))")

    self.cursor.execute("CREATE TABLE  Dump"                           \
                        " (id          INTEGER NOT NULL PRIMARY KEY,"  \
                        "  location    INTEGER NOT NULL,"              \
                        "  timestamp   INTEGER,"                       \
                        "  temperature REAL,"                          \
                        "  light       INTEGER,"                       \
                        "  FOREIGN KEY(location) REFERENCES Location(id))")
    self.cursor.execute("CREATE UNIQUE INDEX nodupe2 ON Dump (location, timestamp)")
    self.cursor.execute("CREATE INDEX speed1 ON Dump (timestamp)")


    self.cursor.execute("CREATE VIEW HostLoc AS" \
                        " SELECT l.id AS locid, h.hostname, l.sensor, l.location" \
                        " FROM Host as h, Location as l WHERE h.id=l.host")

    self.cursor.execute("CREATE VIEW DumpD AS" \
                        " SELECT id, location, datetime(timestamp, 'unixepoch', 'localtime') AS timestamp, temperature, light" \
                        " FROM Dump" \
                        " ORDER BY timestamp, location")

    self.cursor.execute("CREATE VIEW Hours AS" \
                        " SELECT hl.hostname, hl.sensor, hl.location, substr(d.timestamp, 1, 13) || ':00' AS timestamp, count(*) AS minutes" \
                        " FROM DumpD AS d, HostLoc AS hl" \
                        " WHERE hl.locid=d.location" \
                        " GROUP BY hl.hostname, hl.sensor, hl.location, substr(d.timestamp, 1, 13)")

    self.cursor.execute('INSERT INTO Host (id, hostname, ip_addr, firmware, sensor_cnt, last_reboot, comment) VALUES (?, ?, ?, ?, ?, ?, ?)',
                        (1, "zps", "egal", "2.0", 4, "egal", "egal"))
    self.cursor.execute('INSERT INTO Location (host, sensor, location) VALUES (?, ?, ?)', (1, 1, "Entnahme"))
    self.cursor.execute('INSERT INTO Location (host, sensor, location) VALUES (?, ?, ?)', (1, 2, "Legionellen"))
    self.cursor.execute('INSERT INTO Location (host, sensor, location) VALUES (?, ?, ?)', (1, 3, "Zirkulation"))
    self.cursor.execute('INSERT INTO Location (host, sensor, location) VALUES (?, ?, ?)', (1, 4, "Raum"))

    self.connection.commit()

  # ###########################################################
  # Legt die Tabellen an.
  def createDatabase_alt(self):
    self.cursor.execute("CREATE TABLE  Dump"                           \
                        " (id          INTEGER NOT NULL PRIMARY KEY,"  \
                        "  timestamp   INTEGER NOT NULL,"              \
                        "  pump_on     INTEGER,"                       \
                        "  temp_R      REAL,"                          \
                        "  temp_E      REAL,"                          \
                        "  temp_Z      REAL,"                          \
                        "  temp_L      REAL)")

    self.cursor.execute("CREATE VIEW DumpD AS"                                        \
                        " SELECT id, datetime(timestamp, 'unixepoch') AS timestamp,"  \
                        " pump_on, temp_R, temp_E, temp_Z, temp_L,"                   \
                        " round(temp_E-temp_Z, 2) as EZdelta FROM Dump")

    self.connection.commit()


  # ###########################################################
  # Änderungen festschreiben.
  def commit(self):
    self.connection.commit()


  # ###########################################################
  # Fügt die übergebenen Daten in die Tabelle Dump ein.
  def insertDump(self, timestamp, temp_E, temp_L, temp_Z, temp_R, pump_on):
    if pump_on>0:
      pump_on=100
    try:
      self.cursor.execute('INSERT INTO Dump (location, timestamp, temperature, light) VALUES (?, ?, ?, ?)', (1, timestamp, temp_E, pump_on))
      self.cursor.execute('INSERT INTO Dump (location, timestamp, temperature, light) VALUES (?, ?, ?, ?)', (2, timestamp, temp_L, 0))
      self.cursor.execute('INSERT INTO Dump (location, timestamp, temperature, light) VALUES (?, ?, ?, ?)', (3, timestamp, temp_Z, 0))
      self.cursor.execute('INSERT INTO Dump (location, timestamp, temperature, light) VALUES (?, ?, ?, ?)', (4, timestamp, temp_R, 0))
    except Exception, e:
      print "Error: %s"%(str(e))


class PowerOnBitmap():
  def __init__(self):
    # self.power_on_times[7][36] anlegen und mit 0/AUS füllen
    self.power_on_times=[[0 for x in range(36)] for y in range(7)]

  # ###########################################################
  # Python-Weekday kompatibel zum C-Weekday machen.
  #
  # datetime.weekday():  Monday is 0 and Sunday is 6
  # Time.h            :  Day of the week (1-7), Sunday is day 1
  #     Py   C
  #  Mo 0 -> 2
  #  Di 1 -> 3
  #  Mi 2 -> 4
  #  Do 3 -> 5
  #  Fr 4 -> 6
  #  Sa 5 -> 7
  #  So 6 -> 1
  def __weekday_c(self, dt):
    if dt.weekday()==6:
      return(1)
    return(dt.weekday()+2)

  # ###########################################################
  # bit_nr [0, ..., 7]
  def __getBit(self, byte, bit_nr):
    return((byte>>bit_nr)&1)

  # ###########################################################
  #
  def getPowerOnStateForTime(self, dt):
    bit_full_day=dt.hour*12+(dt.minute-dt.minute%5)/5
    byte_adr=bit_full_day/8
    bit_num=bit_full_day-byte_adr*8
    return(self.__getBit(self.power_on_times[self.__weekday_c(dt)-1][byte_adr], bit_num))

  # ###########################################################
  #
  def setPowerOnStateForDatetime(self, dt, state):
    bit_full_day=dt.hour*12+(dt.minute-dt.minute%5)/5
    byte_adr=bit_full_day/8
    bit_num=bit_full_day-byte_adr*8
    if state==0:
      self.power_on_times[self.__weekday_c(dt)-1][byte_adr]&=((1<<bit_num)^0xff)
    elif state==1:
      self.power_on_times[self.__weekday_c(dt)-1][byte_adr]|=(1<<bit_num)

  # ###########################################################
  #
  def setPowerOnStateForTime(self, weekday, hour, minute, state):
    bit_full_day=hour*12+(minute-minute%5)/5
    byte_adr=bit_full_day/8
    bit_num=bit_full_day-byte_adr*8
    if state==0:
      self.power_on_times[weekday-1][byte_adr]&=((1<<bit_num)^0xff)
    elif state==1:
      self.power_on_times[weekday-1][byte_adr]|=(1<<bit_num)

  # ###########################################################
  #
  def printBitmap(self):
    print "       ",
    for i in range(0, 23, 2):
      print "%02d:00   "%(i,),
    print
    for i in range(len(self.power_on_times)):
      print "day=%d :"%(i,),
      for j in range(len(self.power_on_times[i])):
        print "%02x"%(self.power_on_times[i][j],),
      print

  # ###########################################################
  #
  def initDefault(self):
    self.setPowerOnStateForTime(2,  3, 30, 1)
    for wd in [2, 3, 4, 5, 6]: # Montag - Freitag
      self.setPowerOnStateForTime(wd,  5, 45, 1)
      self.setPowerOnStateForTime(wd,  5, 50, 1)
      self.setPowerOnStateForTime(wd,  5, 55, 1)
      self.setPowerOnStateForTime(wd,  6, 30, 1)
      self.setPowerOnStateForTime(wd,  7,  0, 1)
      self.setPowerOnStateForTime(wd,  7, 30, 1)
      self.setPowerOnStateForTime(wd,  8,  0, 1)
      self.setPowerOnStateForTime(wd, 10,  0, 1)
      self.setPowerOnStateForTime(wd, 13,  0, 1)
      self.setPowerOnStateForTime(wd, 13, 30, 1)
      self.setPowerOnStateForTime(wd, 14,  0, 1)
      self.setPowerOnStateForTime(wd, 16,  0, 1)
      self.setPowerOnStateForTime(wd, 18,  0, 1)
      self.setPowerOnStateForTime(wd, 19,  0, 1)
      self.setPowerOnStateForTime(wd, 19, 30, 1)
      self.setPowerOnStateForTime(wd, 20,  0, 1)
      self.setPowerOnStateForTime(wd, 20, 30, 1)
      self.setPowerOnStateForTime(wd, 21,  0, 1)
      self.setPowerOnStateForTime(wd, 21, 30, 1)
      self.setPowerOnStateForTime(wd, 22,  0, 1)
      self.setPowerOnStateForTime(wd, 22, 30, 1)
    for wd in [1, 7]: # Sonntag, Samstag
      self.setPowerOnStateForTime(wd,  6,  0, 1)
      self.setPowerOnStateForTime(wd,  6, 30, 1)
      self.setPowerOnStateForTime(wd,  7,  0, 1)
      self.setPowerOnStateForTime(wd,  7, 30, 1)
      self.setPowerOnStateForTime(wd,  8,  0, 1)
      self.setPowerOnStateForTime(wd,  8, 30, 1)
      self.setPowerOnStateForTime(wd, 10,  0, 1)
      self.setPowerOnStateForTime(wd, 13,  0, 1)
      self.setPowerOnStateForTime(wd, 14,  0, 1)
      self.setPowerOnStateForTime(wd, 18,  0, 1)
      self.setPowerOnStateForTime(wd, 20,  0, 1)
      self.setPowerOnStateForTime(wd, 20, 30, 1)
      self.setPowerOnStateForTime(wd, 21,  0, 1)
      self.setPowerOnStateForTime(wd, 21, 30, 1)
      self.setPowerOnStateForTime(wd, 22,  0, 1)
      self.setPowerOnStateForTime(wd, 22, 30, 1)
    self.setPowerOnStateForTime(7, 23,  0, 1)

    
  # ###########################################################
  #
  def getAsString(self):
    ret_str=""
    for wd in range(7):
      for bp in range(36):
        ret_str+=chr(self.power_on_times[wd][bp])
    return(ret_str)





# ###########################################################
# Der Socket-Server.
# Also Socket öffnen, lauschen, Verbindungen annehmen, 
# verarbeiten und Verbindungen wieder schließen.
class SocketServer:
  def __init__(self, port):
    self.port=port;
    self.srvsock=socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    self.srvsock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    self.srvsock.bind(("", port))
    self.srvsock.listen(5)
    self.descriptors=[self.srvsock]

    self.hostinfo=dict()

    self.db=Database(DATABASE_FILENAME)
    self.pob=PowerOnBitmap()
    self.pob.initDefault()

    self.log=Logging()


  # ###########################################################
  # Hauptschleife des SocketServers.
  def run(self):
    timeout=15  # 5 Sekunden ohne Kommunikation führt zu disconnect

    while True:
      (sread, swrite, sexc)=select.select(self.descriptors, [], [], timeout)
      if sread==[]:   # wenn sich "timeout" Sekunden lang niemand gemeldet hat
        if len(self.descriptors)>1:         # und wenn Clients verbunden
          for sock in self.descriptors:     # über alle Clients
            if sock!=self.srvsock:          # ohne den Server selbst
              try:
                host, port=sock.getpeername() # identifizieren
                self.close_connection(sock, host) # rausschmeißen
                del self.hostinfo[sock]
                self.log.write_warning("timeout")
              except:
                pass
        continue                            # zurück zu select.select

      for sock in sread:
        if sock==self.srvsock:
          self.accept_new_connection()
        else:
          try:
            strg=sock.recv(2048)
          except:
            strg=""
            self.log.write("Verbindung wird geschlossen\n")

          try:
            host, port=sock.getpeername()
          except:
            host, port=("?", "?")

          if strg=="":
            try:
              self.log.write("disconnect")
              self.close_connection(sock, host)
              del self.hostinfo[sock]
            except:
              pass
          else:
            status, data=parseReceivedData(strg)
            if status==True:                          # wenn Checksum gepasst hat
              self.process_command(sock, host, data)  # Kommando verarbeiten
            else:
              self.sock_send_with_checksum(sock, "BAD\x00")


  # ###########################################################
  # Nimmt einen neuen Client an und in die Liste der
  # verbundenen Clients auf.
  def accept_new_connection(self):
    newsock, (host, port)=self.srvsock.accept()
    self.descriptors.append(newsock)


  # ###########################################################
  # Sendet "strg" samt Checksum an "sock".
  def sock_send_with_checksum(self, sock, strg):
    #print "sock_send_with_checksum len=", len(strg)
    try:
      sock.send(buildDataWithBinChecksum(strg))
    except:
      pass
      self.log.write("Fehler bei sock_send_with_checksum()")


  # ###########################################################
  # Schließt die Verbindung zum Client gemäß "sock".
  def close_connection(self, sock, host=""):
    try:
      sock.close()
      self.descriptors.remove(sock)
    except:
      pass


  # ###########################################################
  # Liefert den Hostname zur IP-Adresse "ipaddr".
  def get_hostname(self, ipaddr):
    try:
      fqdn=socket.gethostbyaddr(ipaddr)
    except:
      return("")
    if len(fqdn)>1:
      hn=fqdn[0].split('.')
      return(hn[0])
    return("")


  # ###########################################################
  # Verarbeitet die Anweisungen des Clients.
  # Das Protokoll ist folgendens:
  #   in:   "GET_TIME"
  #   out:  Ortszeit als String           Format: "YYYY.MM.DD hh:mm:ss"
  #
  #   in:   "SSID.n=" + data              mit n=["1", "2", "3"]
  #   out:  "OK" oder "NEW,<neue SSID>"
  #
  #   in:   "SOCKSRV_IPADR.n" + data      mit n=["1", "2"]
  #   out:  "OK" oder "NEW,xxx.xxx.xxx.xxx"
  #
  #   in:   "SOCKSRV_PORT.n" + data       mit n=["1", "2"]
  #   out:  "OK" oder "NEW,yyyyy"
  #
  #   in:   "SENSORS=" + data             mit data="12345678,12345678,12345678,12345678"
  #   out:  "BAD" oder "LEZR"
  #
  #   in:   "GET_POWER_ON_BITMAP"
  #   out:  Bitmap in der Länge von 252 Byte
  #
  #   in:   "GET_SYNC_TIMES"
  #   out:  String in der Länge 8
  #
  #   in:   "GET_THRESHOLDS"
  #   out:  String in der Länge 16
  #
  #   in:   "SENSORHOUR.hh=" + data       mit data="4 Byte, binärer Timestamp"+"60*7 Byte, binäre Sensordaten"
  #   out:  "OK"
  #
  #   in:   "INFO=" + data
  #   out:  "OK"
  #
  def process_command(self, sock, host, cmd):
    if cmd.startswith("GET_TIME"):
      t=curTime(True)
      self.sock_send_with_checksum(sock, t)
      self.log.write("GET_TIME (%s)"%(t,))

    elif cmd.startswith("SSID."):
      nr=cmd[5]                     # nr=X        SSID.X
      ssid=cmd[7:]                  # ssid=Y..Y   SSID.X=Y..Y
      #print ">>>", cmd
      if ssids_nam.get(ssid, "-")==nr:
        self.sock_send_with_checksum(sock, "OK")
      else:
        self.sock_send_with_checksum(sock, "NEW,"+ssids_num[nr])
      
    elif cmd.startswith("SOCKSRV_IPADR."):
      nr=cmd[14]
      ipadr=cmd[16:]
      #print ">>>", cmd
      if socketServerIP_val.get(ipadr, "-")==nr:
        self.sock_send_with_checksum(sock, "OK")
      else:
        self.sock_send_with_checksum(sock, "NEW,"+socketServerIP_num[nr])

    elif cmd.startswith("SOCKSRV_PORT."):
      nr=cmd[13]
      port=cmd[15:]
      #print ">>>", cmd
      if socketServerPort_val.get(port, "-")==nr:
        self.sock_send_with_checksum(sock, "OK")
      else:
        self.sock_send_with_checksum(sock, "NEW,"+socketServerPort_num[nr])

    elif cmd.startswith("SENSORS="):
      v=cmd.split("=", 1)
      if len(v)>1:
        self.log.write("Sensors: %s"%(v[1]))
        # v[1]="28ff94ba021704c6,28ff9e25031704ab,28ff01d20217042f,28ff21e87316055a"
        s_nam=""
        s_lst=v[1].split(",") # die Sensor-Adressen einzeln als Liste
        unknown=0
        unknown_char="?"
        for s in s_lst:       # unbekannte/neue Sensoren suchen
          if s not in sensors_adr:
            unknown+=1
          else:
            s_nam+=sensors_adr[s]
        if unknown==1:        # wenn genau ein Sensor unbekannt/neu ist...
          # kann das maschinell gefixt werden
          for s in "LEZR":
            if s not in s_nam:
              unknown_char=s
              self.log.write_warning("Sensor %c wurde offenbar getauscht! Sensor-Daten sind anzupassen!"%(s,))
        elif unknown>1:
          self.log.write_warning("Mehr als einen neuen Sensor erkannt!")
        s_nam=""
        for s in s_lst:
          s_nam+=sensors_adr.get(s, unknown_char)
        self.sock_send_with_checksum(sock, s_nam)
      else:
        self.sock_send_with_checksum(sock, "BAD")

    elif cmd.startswith("GET_POWER_ON_BITMAP"):
      pob=self.pob.getAsString()
      #print "len(pob)=", len(pob)
      self.sock_send_with_checksum(sock, pob)

    elif cmd.startswith("GET_SYNC_TIMES"):
      strg=""
      for (h, m) in syncTimes:
        strg+=chr(h)
        strg+=chr(m)
      self.sock_send_with_checksum(sock, strg)

    elif cmd.startswith("GET_THRESHOLDS"):
      strg=""
      for i in thresholds:
        strg+=chr(i)
      self.sock_send_with_checksum(sock, strg)

    elif cmd.startswith("SENSORHOUR."):
      v=cmd.split("=", 1)
      print ">", v[0], "-"*40
      #hexdump(v[1], len(v[1]))
      tsh, ha=getSensorDataFromHourBlock(v[1])
      print tsh.strftime("%Y.%m.%d %H:%M:%S")
      for m in range(60):
        if ha[m][5]==0:
          #print m, hex(ha[m][0]), hex(ha[m][1]), hex(ha[m][2]), hex(ha[m][3]), ha[m][4]
          tsm=datetime.datetime(tsh.year, tsh.month, tsh.day, tsh.hour, m)
          #tsm=int((tsm-datetime.datetime(1970, 1, 1)).total_seconds())
          tsm=int(tsm.strftime("%s"))
          tsms=datetime.datetime.fromtimestamp(tsm).strftime("%Y.%m.%d %H:%M:%S")
          print "%s  %4.1f  %4.1f  %4.1f  %4.1f"%(tsms, ha[m][0]/16.0, ha[m][1]/16.0, ha[m][2]/16.0, ha[m][3]/16.0), ha[m][4]
          self.db.insertDump(tsm, ha[m][0]/16.0, ha[m][1]/16.0, ha[m][2]/16.0, ha[m][3]/16.0, ha[m][4])
      self.db.commit()
      self.sock_send_with_checksum(sock, "OK")

    elif cmd.startswith("INFO="):
      v=cmd.split("=", 1)
      if len(v)>1:
        self.log.write("Info: %s"%(v[1]))
      self.sock_send_with_checksum(sock, "OK")

    else:
      self.sock_send_with_checksum(sock, "ECHO:"+cmd)
      self.log.write_warning("unbekannte Anforderung: %s"%(cmd,))






# ###########################################################
# Kapselung der Funktionen zum Logging.
class Logging():
  def __init__(self):
    pass


  # ###########################################################
  # Schreibt Ortszeit und "strg" mit vorangestelltem und farblich
  # hervorgehobenem Text " Warnung : " aufs Terminal und in die
  # Log-Datei.
  def write_warning(self, strg):
    with open(FILENAME_LOGFILE, 'a') as fl:
      fl.write("%s \x1b[1;32;41m Warnung \x1b[0m : %s\n"%(curTime(), strg))
    print "%s \x1b[1;32;41m Warnung \x1b[0m : %s"%(curTime(), strg,)

    
  # ###########################################################
  # Schreibt Ortszeit und "strg" aufs Terminal und in die
  # Log-Datei.
  def write(self, strg):
    with open(FILENAME_LOGFILE, 'a') as fl:
      fl.write(curTime() + " " + strg + "\n")
    print curTime(), strg




# ###########################################################
# Die drei Byte Sensor-Daten enthalten drei Werte. Der erste
# Wert belegt 12 Bit, der zweite 10 Bit und der dritte 1 Bit.
# Sind dies die Nibbles der drei Byte: ab cd ef
# bildet sich der Temeraturwert aus:   d ab  (&0xFFF)
# der Helligkeitswert aus:             ef c  (&0x3FF)
# und das BAD-Flag aus:                e     (&0x4)
# oder auch als:
#   temp=0x6bc   -> 0xbc 0x06 0x00
#   ldr =0x345   -> 0x00 0x50 0x34
#   bad =1       -> 0x00 0x00 0x40
def getSensorDataFromHourBlock2(strg):
  lst=list()
  for m in range(60):
    p=4+3*m
    elem=strg[p:p+3]
    temp=ord(elem[0]) + ((ord(elem[1])&0x0F)<<8)
    if (temp&0x800)>0:  # Vorzeichen-Bit
      temp=-(0x7FF-(temp&0x7FF))
    ldr=((ord(elem[1])&0xF0)>>4) + ((ord(elem[2])&0x3F)<<4)
    bad=(ord(elem[2])&0x40)>>6
    #print hex(ord(elem[0])), hex(ord(elem[1])), hex(ord(elem[2])), (round(temp/16.0, 1), ldr, bad)
    lst.append((round(temp/16.0, 1), ldr, bad))
  return(lst)

# ###########################################################
# Zerlegt einen StundenBlock in "strg" in seine Komponenten.
# Die ersten vier Byte sind der Timestamp.
# Für 2017.10.6 9:58:4  ==  59 d7 53 ac
# würde geliefert werden    ac 53 d7 59
# Danach folgen 60 Blocke von je 7 Byte Länge.
# Die Variablen eines solchen Blocks sind:
#   temp_E(12), temp_L(12), temp_Z(12), tempR(12), pump_on(1), bad(1)
#   123         456         789         abc        x           y
# Sie werden geliefert als:
#   23 61 45 89 c7 ab 0(00yx)
#   00 01 02 03 04 05 06
def getSensorDataFromHourBlock(strg):
  ts=int("%02x%02x%02x%02x"%(ord(strg[3]), ord(strg[2]), ord(strg[1]), ord(strg[0])), 16)
  tsdt=datetime.datetime.utcfromtimestamp(ts)
  #print tsdt.strftime("%Y.%m.%d %H:%M:%S")
  ha=list()
  for m in range(60):
    md=strg[4+m*7:11+m*7]
    temp_E=((ord(md[1])&0x0f)<<8)+ord(md[0])          # 23 -1 -> 1 23
    temp_L=((ord(md[1])&0xf0)>>4)+(ord(md[2])<<4)     # 6- 45 -> 4 56
    temp_Z=((ord(md[4])&0x0f)<<8)+ord(md[3])          # 89 -7 -> 7 89
    temp_R=((ord(md[4])&0xf0)>>4)+(ord(md[5])<<4)     # c- ab -> a bc
    pump_on=ord(md[6])&0x01
    bad=(ord(md[6])&0x02)>>1
    ha.append((temp_E, temp_L, temp_Z, temp_R, pump_on, bad))
  return((tsdt, ha))


# ###########################################################
# Die Nutzdaten eines ein-Stunden-Blocks sind 4+3*60=184=0xB8
# Byte lang. Zusammen mit der Prüfsumme von zwei Byte ist der
# ankommende Block 186=0xBA Byte lang. Die Prüfsumme wurde
# aus den 184 Byte Nutzdaten gebildet.
def isValidChecksumForHourBlock(strg):
  cs=0
  for c in strg[:-2]:
    cs+=ord(c)
    cs+=1
  return((cs&0xFFFF)==(ord(strg[-2:][0])+ord(strg[-2:][1])*256))


# ###########################################################
# Zerlegt einen vier Byte langen Timestamp in seine
# Komponenten.
# Eine long-Variable mit dem Inhalt 0x01234567 wird
# übertragen als: 67 45 23 01
# Somit sieht der Aufbau ziemlich durcheinander aus:
#    MMYYYYYY  hDDDDDMM  mmmmhhhh  ssssssmm
def convertTimestamp(strg):
  YY=  ord(strg[0])&0b00111111
  MM=((ord(strg[0])&0b11000000)>>6) + ((ord(strg[1])&0b00000011)<<2)
  DD= (ord(strg[1])&0b01111100)>>2
  hh=((ord(strg[1])&0b10000000)>>7) + ((ord(strg[2])&0b00001111)<<1)
  mm=((ord(strg[2])&0b11110000)>>4) + ((ord(strg[3])&0b00000011)<<4)
  ss= (ord(strg[3])&0b11111100)>>2
  return(YY+2000, MM, DD, hh, mm, ss)


# ###########################################################
# Liefert die Ortszeit als String.
# Wird "precise" mit True übergeben, wartet die Funktion vor
# der Rückgabe der Zeit bis zum Beginn einer neuen Sekunde.
def curTime(precise=False):
  if precise:
    n=datetime.datetime.now().second
    while datetime.datetime.now().second==n:
      time.sleep(0.00001)
    # jetzt hat die neue Sekunde gerade frisch angefangen
  return(datetime.datetime.now().strftime("%Y.%m.%d %H:%M:%S"))


# ###########################################################
# Trennt die Checksum von "strg" ab und liefert ein Tupel
# (True, strg), sofern die Checksum passte. Wenn nicht, wird
# (False, "") geliefert.
def parseReceivedData(strg):
  if len(strg)<3:
    return((False, ""))
  cs=ord(strg[-2:][0])+ord(strg[-2:][1])*256
  strg=strg[:-2]
  if getBinChecksum(strg)!=cs:
    return((False, ""))
  return((True, strg))


# ###########################################################
# Liefert "strg" mit angehängter Checksum.
def buildDataWithBinChecksum(strg):
  cst=getBinChecksum(strg)
  cs=chr(cst&0xFF) + chr((cst&0xFF00)/256)
  return(strg+cs)


# ###########################################################
# Liefert die Checksumme zu "strg" als 2-Byte Integer.
def getBinChecksum(strg):
  cs=0
  for c in strg:
    cs+=ord(c)
  return(cs&0xFFFF)


# ###########################################################
# Gibt einen Hexdump von "buf" in der Länge "lng" aus.
def hexdump(buf, lng):
  p=0
  while p<lng:
    sys.stdout.write("%04X  "%(p))
    for i in range(min(16, (lng-p))):
      sys.stdout.write("%02x "%ord(buf[p+i]))
    sys.stdout.write(" ")
    if (lng-p)<16:
      sys.stdout.write(" "*(3*(16-(lng-p))))
    for i in range(min(16, (lng-p))):
      c=ord(buf[p+i])
      if c>31 and c<128:
        sys.stdout.write(chr(c))
      else:
        sys.stdout.write('.')
    p+=16
    print


if __name__=='__main__':
  myServer=SocketServer(2629)
  myServer.run()
