#include "wlan_comm.h"

WiFiClient sockSrv_client;


/* -------------------------------------------------------------------
 *  Versucht nacheinander eine Verbindung zu den in SSID[3][33] 
 *  abgelegten WLAN-Namen herzustellen und liefert true, wenn es bei
 *  mindestens einem WLAN geklappt hat.
 */
int connectKnownWLAN(void) {
  int i, wlan_cnt, wlan_nr, retries;
  char ssid[33];  // maxLen(SSID):  https://serverfault.com/a/45509/430599

  WiFi.forceSleepWake();
  wlan_cnt=WiFi.scanNetworks();
  if(wlan_cnt>0) {
    for(i=0; i<3; i++) {
      if(strlen(SSID[i])==0) {
        continue;
      }
      for(wlan_nr=0; wlan_nr<wlan_cnt; ++wlan_nr) {
        WiFi.SSID(wlan_nr).toCharArray(ssid, 33);
        //Serial.print("strcmp(");  Serial.print(ssid); Serial.print(", "); Serial.print(SSID[i]);  Serial.println(")");
        if(strcmp(ssid, SSID[i])==0) {
          Serial.print(ssid); Serial.print("  "); Serial.println(WiFi.RSSI(wlan_nr));
          WiFi.begin(ssid);
          for(retries=0; retries<20 && WiFi.status()!=WL_CONNECTED; retries++) {  // max. 10 Sekunden probieren
            delay(500);
          }
          if(WiFi.status()==WL_CONNECTED) {
            return(1);
          }
        }
      }
    }
  }
  return(0);
}

/* -------------------------------------------------------------------
 *  Stellt eine Verbindung zum WLAN her und liefert true, wenn es
 *  geklappt hat.
 */
int connectWLAN(void) {
  int retries;

  WiFi.forceSleepWake();
  WiFi.begin("FreifunkWees01.0 (http://ffw)", "");
  retries=20;
  while(WiFi.status()!=WL_CONNECTED && retries>0) { // 10 Sek. lang probieren
    delay(500);
    retries--;
  }
  return(WiFi.status()==WL_CONNECTED);
}

/* -------------------------------------------------------------------
 *  Trennt die Verbindung zum WLAN.
 */
void disconnectWLAN(void) {
  WiFi.disconnect();
  delay(100);   // dem disconnect etwas Zeit geben, bevor abgeschatet wird
  WiFi.forceSleepBegin();
}

/* -------------------------------------------------------------------
 *  Stellt eine Verbindung zum übergebenen SocketServer her und 
 *  liefert true, wenn es geklappt hat.
 */
int connectSocketServer(char *ipaddr, int portnr) {
  return(sockSrv_client.connect(ipaddr, portnr));
}
//int connectSocketServer(void) {
//  return(sockSrv_client.connect("192.168.42.80", 2629));  // ss=.80, ffw=.99
//}

/* -------------------------------------------------------------------
 *  Versucht nacheinander eine Verbindung zu einem der in 
 *  SocketServer_adr[][] und SocketServer_port[] abgelegten
 *  SocketServer herzustellen und liefert true, wenn es geklappt hat.
 */
int connectKnownSocketServer(void) {
  int ss, retries;

  for(ss=0; ss<2; ss++) {   // zwei mögliche SocketServer in SocketServer_adr[][]
    if(strlen(SocketServer_adr[ss])==0) {
      continue;
    }
    for(retries=0; retries<5; retries++) {
      Serial.print("verbinde mit SocketServer @");
      Serial.print(SocketServer_adr[ss]);
      Serial.print(":");
      Serial.println(SocketServer_port[ss], DEC);
      if(connectSocketServer(SocketServer_adr[ss], SocketServer_port[ss])) {
        Serial.println("verbunden mit SocketServer");
        return(true);
      }
    }
  }
  return(false);
}


/* -------------------------------------------------------------------
 *  Liefert true, wenn die Verbindung noch steht.
 */
int isConnectedToSocketServer(void) {
  return(sockSrv_client.connected());
}

/* -------------------------------------------------------------------
 *  Trennt die Verbindung zum SocketServer.
 */
void disconnectSocketServer(void) {
  sockSrv_client.stop();
}

/* -------------------------------------------------------------------
 *  Liefert eine 16 Bit lange Prüfsumme für den Speicherbereich ab
 *  "buf" in der Länge "len".
 */
uint16_t getBinChecksum(uint8_t *buf, uint16_t len) {
  uint16_t cs=0;
  for(; len>0; len--)  {
    cs+=*buf++;
  }
  return(cs);
}

/* -------------------------------------------------------------------
 *  Sendet den Inhalt des Speicherbereiches ab "buf" in der Länge
 *  "len" samt einer Prüfsumme an den SocketServer.
 *  Die Verbindung zum SocketServer muss offen sein.
 */
void sendSocketServer(uint8_t *buf, uint32_t len) {
  uint8_t *buf_ptr;
  uint16_t cs=getBinChecksum(buf, len);

  buf_ptr=(uint8_t *)malloc(len+2);
  if(buf_ptr!=NULL) {
    memcpy(buf_ptr, buf, len);
    *(buf_ptr+len)=cs&0xff;
    *(buf_ptr+len+1)=(cs>>8)&0xff;
    sockSrv_client.write((const uint8_t *)buf_ptr, len+2);
    free(buf_ptr);
  }
}

/* -------------------------------------------------------------------
 *  Empfängt bis zu "maxlen" Byte vom SocketServer und legt sie ab
 *  "buf" im Speicher ab. Passt die Checksum gemäß der letzten zwei
 *  Byte zu den davor liegenden Byte, wird die Länge der empfangenen
 *  Daten (ohne die letzten zwei Byte für die Checksum)
 *  zurückgeliefert. Ansonsten wird 0 geliefert.
 *  Die Verbindung zum SocketServer muss offen sein.
 */
uint32_t receiveSocketServer(uint8_t *buf, uint32_t maxlen, uint32_t timeout_ms=2000) {
  uint32_t start_ms=millis();
  uint32_t idx, len;
  uint16_t cs;

  while(sockSrv_client.available()==0 && (millis()-start_ms)<timeout_ms) {
    delay(10);
  }
  idx=0;
  while(sockSrv_client.available()>0 && idx<maxlen && (millis()-start_ms)<timeout_ms) {
    buf[idx++]=sockSrv_client.read();
  }
  if(idx>=3 && (millis()-start_ms)<timeout_ms) {
    // Mindestens ein Byte Nutzdaten empfangen und kein Timeout aufgetreten
    len=idx-2;
    cs=getBinChecksum(buf, len);
    if(buf[len]==(uint8_t)cs&0xff && buf[len+1]==(uint8_t)(cs>>8)&0xff) {
      buf[len]='\0';  // String draus machen...falls es keiner sein sollte
      return(len);
    } else {
      Serial.println("checksum fehler");
      Serial.println(len);
    }
  }
  return(0);
}

