#include <stdlib.h>
#include <stdint.h>
#include <Time.h>
#include <ESP8266WiFi.h>

#include "wlan_comm.h"
#include "ds18b20.h"
#include "rtc_eeprom.h"
#include "debug.h"
#include "globals.h"
#include "esp_eeprom.h"
#include "sensor.h"

#define REC_BUF_SIZE 1024
uint8_t rec_buf[REC_BUF_SIZE];


time_t newMinute;
time_t newHour;
time_t blockSync;
time_t nextSync;

int leg_mode, leg_time, leg_date;
int pump_on;
int led_rt, led_bl;

/* -------------------------------------------------------------------
 *  Setzt die interne Uhr gemäß des Timestamp-Strings "datetime_str".
 */
void setSysTimeFromString(uint8_t *datetime_str) {
  // Einzelstrings aus Timestamp bauen
  // 2015.10.20 19:48:08
  // 01234567890123456789
  //     X  X  X  X  X
  *(datetime_str+4)='\0';
  *(datetime_str+7)='\0';
  *(datetime_str+10)='\0';
  *(datetime_str+13)='\0';
  *(datetime_str+16)='\0';
  setTime(atoi((char*)datetime_str+11), atoi((char*)datetime_str+14), atoi((char*)datetime_str+17),
          atoi((char*)datetime_str+8), atoi((char*)datetime_str+5), atoi((char*)datetime_str));
}


/* -------------------------------------------------------------------
 *  Init
 *    Sensor-Adressen lesen
 *    loop
 *      Zeit holen und als Systemzeit einstellen
 *      Sensor-Adressen senden
 *      Sensor-Zuordnung empfangen
 *      
 */
void setup() {
  int i, rc, new_data_received;
  time_t t;

  error_g=0;
  WiFi.forceSleepBegin();
  Serial.begin(115200);

  pinMode(RELAIS_PIN, OUTPUT);
  digitalWrite(RELAIS_PIN, LOW);  // Relais aus
  RTC_EEPROM_init();
  pinMode(LED_BL_PIN, OUTPUT);
  pinMode(LED_RT_PIN, OUTPUT);
  digitalWrite(LED_BL_PIN, HIGH);  // LED aus
  digitalWrite(LED_RT_PIN, HIGH);  // LED aus

  clearSensorData_subMinuteArray();
  clearSensorDataArray();

  rc=copyVarsFromEEPROM();
  //rc=false;
  if(!rc) {
    // Wenn die Daten im EEPROM nicht valide sind, wurden sie
    // voraussichtlich noch nie gesetzt. Damit der erste Connect
    // mit dem SocketServer klappen kann, müssen sie dann hier
    // gesetzt werden.
    strcpy(SSID[0], "FreifunkWees01.0 (http://ffw)");
    strcpy(SSID[1], "FreifunkWees01.2 (http://ffw)");
    strcpy(SSID[2], "");
    strcpy(SocketServer_adr[0], "192.168.42.80");
    strcpy(SocketServer_adr[1], "");
    SocketServer_port[0]=2629;
    SocketServer_port[1]=0;
    Sync_time[0][0]=3;
    Sync_time[0][1]=15;
    for(i=1; i<NUMBER_SYNCTIMES; i++) {
      Sync_time[i][0]=255;
      Sync_time[i][1]=255;
    }
    Serial.println("\n\nFactoryDefaults eingestellt!");
    //error_g|=xx;
  }

  rc=discoverOneWireDevices();
  if(rc==0) {
    Serial.println("crc-error in discoverOneWireDevices()");
  }
/*  if(sensor_count_g==0) {
    strncpy((char*)ROMcode_g[0], "\x28\xff\x94\xba\x02\x17\x04\xc6", 8);  // Fake-Sensoren fürs Debugging auf WeMos
    strncpy((char*)ROMcode_g[1], "\x28\xff\x9e\x25\x03\x17\x04\xab", 8);  // ohne den ",8" würden 9 Byte (+'\0') kopiert werden
    strncpy((char*)ROMcode_g[2], "\x28\xff\x01\xd2\x02\x17\x04\x2f", 8);
    strncpy((char*)ROMcode_g[3], "\x28\xff\x21\xe8\x73\x16\x05\x5a", 8);
    sensor_count_g=4;
  }*/
  Serial.print("\n\nsensor_count_g=");  Serial.println(sensor_count_g);
  if(sensor_count_g!=MAX_SENSOR_COUNT) {
    Serial.println("Falsche Anzahl von Sensoren erkannt!");
    // error_g|=xx;
  }
  for(i=0; i<sensor_count_g; i++) {     // Dummy-read zwecks Init aller erkannten Sensoren
    readDS18B20_CRC(i);
  }

  doFullSync(true);

  // die per WLAN geholte Zeit muss [auch] in die RTC geschrieben werden.
  // weiterhin ist die Funktion der RTC zu prüfen und ggf. eine Warnung zu senden
  //if(reconnectIfNeeded()) {
  //  getSetDateTime();
    /*sendSocketServer((uint8_t*)"GET_TIME", 8);  // Zeit anfragen
    Serial.println("GET_TIME gesendet");
    if(receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000)>=19) {  // Zeit empfangen z.B.: 2017.09.21 20:17:06
      Serial.println((char*)rec_buf);
      setRTC((char*)rec_buf);
      setSysTimeFromString(rec_buf);
      t=getRTC();
      //Serial.print("RTC="); Serial.print(t); Serial.print("   SYS="); Serial.println(now());
      if(abs(now()-t)>1) {  // bei mehr als einer Sekunde Abweichnung ist was faul
        Serial.println("SYS != RTC !!!");
        // error_g|=xx;
      }
    }*/
    /*new_data_received=false;
    new_data_received|=syncConnectionData();
    new_data_received|=syncSensorMetaData();
    new_data_received|=getPowerOnBitmap();
    new_data_received|=getSyncTimes();
    new_data_received|=getThresholds();
    disconnectSocketServer();
    disconnectWLAN();
    if(new_data_received) {  // wenn Änderungen empfangen wurden...
      rc=copyVarsToEEPROM();
      Serial.print("copyVarsToEEPROM()=");      Serial.println(rc);
    }
  } else {
    // die Kommunikation mit dem SocketServer hat nicht geklappt
    t=getRTC();
    setTime(hour(t), minute(t), second(t), day(t), month(t), year(t));
    Serial.println("Der Sync war nicht erfolgreich!");
    Serial.print("Die SystemZeit wurde gemäß RTC eingestellt:");
    printRTC(t);  Serial.println();
  }*/

  t=now();
  newMinute=t;  // zur Wechsel-Erkennung
  newHour=t;
  blockSync=0;
  nextSync=0;
  pump_on=0;
  nextSensorHour(t);
  led_rt=led_bl=0;
  leg_mode=leg_time=leg_date=0;
}


/* -------------------------------------------------------------------
 *  Liefert true, wenn eine Verbindung zum SocketServer steht.
 */
int reconnectIfNeeded(void) {
  if(isConnectedToSocketServer()) {           // wenn die Verbindung zum SocketServer steht
    return(true);                             // ist alles gut
  }
  if(WiFi.status()!=WL_CONNECTED) {           // wenn WLAN-Connect nicht mehr steht
    if(connectKnownWLAN()) {                  // im WLAN anmelden und wenns geklappt hat
      return(connectKnownSocketServer());     // beim SocketServer anmelden und Erfolgsstatus liefern
    }
  } else {                                    // wenn WLAN-Connect steht
    return(connectKnownSocketServer());       // beim SocketServer anmelden und Erfolgsstatus liefern
  }
  return(false);
}

/* -------------------------------------------------------------------
 *  Fragt beim SocketServer die aktuelle Zeit an und stellt diese
 *  in der RTC und als SystemZeit ein. Liefert true, wenn die Zeit
 *  eingestellt wurde.
 */
int getSetDateTime(void) {
  time_t t;
  sendSocketServer((uint8_t*)"GET_TIME", 8);  // Zeit anfragen
  if(receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000)>=19) {  // Zeit empfangen z.B.: 2017.09.21 20:17:06
    setRTC((char*)rec_buf);
    setSysTimeFromString(rec_buf);
    t=getRTC();
    if(abs(now()-t)>1) {  // bei mehr als einer Sekunde Abweichnung ist was faul
      Serial.println("SYS != RTC !!!");
      // error_g|=xx;
    }
  } else {
    return(false);
  }
  return(true);
}

/* -------------------------------------------------------------------
 *  
 */
int doFullSync(int fromSetup) {
  int rc, new_data_received=false;
  time_t t;

  digitalWrite(LED_BL_PIN, LOW);  // blaue LED an
  if(reconnectIfNeeded()) {
    getSetDateTime();
    new_data_received|=syncConnectionData();    // WLAN-SSID, IP-Adresse, Portnummer
    if(fromSetup) {                             // nur, wenn aus setup() aufgerufen...
      new_data_received|=syncSensorMetaData();  // ...Sensor-Adress-Zuordnung
    }
    new_data_received|=getPowerOnBitmap();      // Einschaltzeiten
    new_data_received|=getSyncTimes();          // Sync-Zeiten
    new_data_received|=getThresholds();         // Schwellwerte / Deltas
    if(!fromSetup) {                            // nur, wenn aus loop() aufgerufen...
      sendSensorHours();                        // ...Sensor-Daten senden
    }
    disconnectSocketServer();
    disconnectWLAN();
    if(new_data_received) {  // wenn Änderungen empfangen wurden...
      rc=copyVarsToEEPROM();
      Serial.print("copyVarsToEEPROM()=");      Serial.println(rc);
    }
    rc=true;
  } else {
    // die Kommunikation mit dem SocketServer hat nicht geklappt
    t=getRTC();
    setTime(hour(t), minute(t), second(t), day(t), month(t), year(t));
    Serial.println("Der Sync war nicht erfolgreich!");
    Serial.print("Die SystemZeit wurde gemäß RTC eingestellt:");
    printRTC(t);  Serial.println();
    rc=false;
  }
  digitalWrite(LED_BL_PIN, HIGH);  // blaue LED aus
  return(rc);
}

/* -------------------------------------------------------------------
 *  Sendet den Inhalt der Variablen
 *    SSID[][]
 *    SocketServer_adr[][]
 *    SocketServer_port[]
 *  an den SocketServer.
 *  Bei der Antwort "OK" passiert nix weiter, bei "NEW,..." wird
 *  die entsprechende Variable (im RAM) auf den übermittelten Wert 
 *  gesetzt.
 *  Liefert true, wenn mindestens eine Variable neu gesetzt wurde.
 */
int syncConnectionData(void) {
  int nr, was_changed;

  was_changed=false;
  for(nr=0; nr<3; nr++) {
    strcpy((char*)rec_buf, "SSID.");
    snprintf((char*)rec_buf+strlen((char*)rec_buf), 34, "%d=%s", nr, SSID[nr]);
    sendSocketServer(rec_buf, strlen((char*)rec_buf));
    receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000);
    Serial.print("syncSSIDs "); Serial.println((char*)rec_buf);
    if(strcmp((char*)rec_buf, "OK")!=0) {
      was_changed=true;
      strncpy(SSID[nr], (char*)rec_buf+4, 33);
    }
  }
  for(nr=0; nr<2; nr++) {
    strcpy((char*)rec_buf, "SOCKSRV_IPADR.");
    snprintf((char*)rec_buf+strlen((char*)rec_buf), 34, "%d=%s", nr, SocketServer_adr[nr]);
    sendSocketServer(rec_buf, strlen((char*)rec_buf));
    receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000);
    Serial.print("syncSOCKSRV_IPADRs "); Serial.println((char*)rec_buf);
    if(strcmp((char*)rec_buf, "OK")!=0) {
      was_changed=true;
      strncpy(SocketServer_adr[nr], (char*)rec_buf+4, 33);
    }
  }
  for(nr=0; nr<2; nr++) {
    strcpy((char*)rec_buf, "SOCKSRV_PORT.");
    snprintf((char*)rec_buf+strlen((char*)rec_buf), 34, "%d=%d", nr, SocketServer_port[nr]);
    sendSocketServer(rec_buf, strlen((char*)rec_buf));
    receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000);
    Serial.print("syncSOCKSRV_PORTs "); Serial.println((char*)rec_buf);
    if(strcmp((char*)rec_buf, "OK")!=0) {
      was_changed=true;
      SocketServer_port[nr]=atoi((char*)rec_buf+4);
    }
  }
  return(was_changed);
}

/* -------------------------------------------------------------------
 *  Schreibt die Adresse des Sensors "sn" als String nach "buf".
 */
void formatDS18b20Address(int sn, char *buf) {
  snprintf(buf, 20, "%02x%02x%02x%02x%02x%02x%02x%02x,",
    ROMcode_g[sn][0], ROMcode_g[sn][1], ROMcode_g[sn][2], ROMcode_g[sn][3],
    ROMcode_g[sn][4], ROMcode_g[sn][5], ROMcode_g[sn][6], ROMcode_g[sn][7]);
}

/* -------------------------------------------------------------------
 *  Sendet die Sensor-Adressen und empfängt deren Funktions-Zuordnung.
 *  Liefert true, wenn eine geänderte Sensor-Zuordnung nach
 *  ROMindex_g[] geschrieben wurde.
 */
int syncSensorMetaData(void) {
  int s;

  strcpy((char*)rec_buf, "SENSORS=");
  for(s=0; s<sensor_count_g; s++) {
    formatDS18b20Address(s, (char*)rec_buf+strlen((char*)rec_buf)); // Sensor-Adressen hochmelden
  }
  rec_buf[strlen((char*)rec_buf)-1]=0; // das letzte Komma löschen
  sendSocketServer(rec_buf, strlen((char*)rec_buf));
  // Funktions-Zuordnung empfangen
  if(receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000)>0) {
    if(strcmp((char*)rec_buf, "BAD")==0) {
      Serial.println("syncSensorMetaData() hat BAD empfangen!");
      return(false);
    }
    if(memcmp(rec_buf, ROMindex_g, 4)==0) {
      return(false);  // keine Änderungen empfangen
    }
    for(s=0; s<strlen((char*)rec_buf) && s<MAX_SENSOR_COUNT; s++) {
      ROMindex_g[s]=rec_buf[s];
    }
    Serial.println((char*)rec_buf);
    return(true);
  }
  return(false);
}


/* -------------------------------------------------------------------
 *  Empfängt die Schaltzeiten-Bitmap für die Zirkulationspumpe. 
 *  Wurden Änderungen erkannt, werden die Daten nach
 *  Power_on_time_g[][] kopiert und true zurückgeliefert.
 */
int getPowerOnBitmap(void) {
  int len;

  strcpy((char*)rec_buf, "GET_POWER_ON_BITMAP");
  sendSocketServer(rec_buf, strlen((char*)rec_buf));
  len=receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000);
  if(len==7*36) {
    if(memcmp(rec_buf, Power_on_time_g, 7*36)==0) {
      return(false);  // keine Änderungen empfangen
    }
    memcpy(Power_on_time_g, rec_buf, 7*36);
    return(true);
  } else {
    Serial.print("len(getPowerOnBitmap)=");    Serial.println(len);
  }
  return(false);
}

/* -------------------------------------------------------------------
 *  Liefert 1, wenn für Wochtag+Uhrzeit gemäß "t" eine 1 in der
 *  Bitmap steht.
 */
int getPowerOnStateForTime(time_t t) {
  int bit_full_day, byte_adr, bit_num;

  bit_full_day=hour(t)*12+(minute(t)-minute(t)%5)/5;
  byte_adr=bit_full_day/8;
  bit_num=bit_full_day-byte_adr*8;
  //Serial.print("bit_full_day=");    Serial.print(bit_full_day);
  //Serial.print("     byte_adr=");   Serial.print(byte_adr);
  //Serial.print("     bit_num=");    Serial.println(bit_num);
  return((Power_on_time_g[weekday(t)-1][byte_adr]>>bit_num)&1);
}

/* -------------------------------------------------------------------
 *  Empfängt NUMBER_SYNCTIMES Sync-Zeiten.
 *  Wurden Änderungen erkannt, werden die Daten nach
 *  Sync_time[][] kopiert und true zurückgeliefert.
 */
int getSyncTimes(void) {
  int len;

  strcpy((char*)rec_buf, "GET_SYNC_TIMES");
  sendSocketServer(rec_buf, strlen((char*)rec_buf));
  len=receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000);
  if(len==NUMBER_SYNCTIMES*2) {
    if(memcmp(rec_buf, Sync_time, NUMBER_SYNCTIMES*2)==0) {
      return(false);  // keine Änderungen empfangen
    }
    memcpy(Sync_time, rec_buf, NUMBER_SYNCTIMES*2);
    return(true);
  } else {
    Serial.print("len(getSyncTimes)=");    Serial.println(len);
  }
  return(false);
}

/* -------------------------------------------------------------------
 *  Empfängt die Schwellwerte.
 *  Liefert true, wenn geänderte Schwellwerte nach Threshold[]
 *  geschrieben wurde.
 */
int getThresholds(void) {
  int len;

  strcpy((char*)rec_buf, "GET_THRESHOLDS");
  sendSocketServer(rec_buf, strlen((char*)rec_buf));
  len=receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000);
  if(len==16) {
    if(memcmp(rec_buf, Threshold, 16)==0) {
      return(false);  // keine Änderungen empfangen
    }
    memcpy(Threshold, rec_buf, 16);
    return(true);
  } else {
    Serial.print("len(syncThresholds)=");    Serial.println(len);
  }
  return(false);
}

/* -------------------------------------------------------------------
 *  Sendet alle validen Daten-Sätze in "sensor_day_g" an den
 *  SocketServer. Gesendete Sätze werden auf invalide gesetzt.
 *  Liefert true, wenn mindestens ein Satz gesendet und bestätigt
 *  wurde.
 */
int sendSensorHours(void) {
  int h, m;
  int len;
  int has_valid_data, ack=false;
  char *ptr;

  for(h=0; h<26; h++) {
    has_valid_data=false;
    for(m=0; m<60; m++) {   // prüfen, ob es mindestens eine gültige Daten-Minute in der Stunde gibt
      if(sensor_day_g[h].data[m].bad==0) {
        has_valid_data=true;
        break;
      }
    }
    if(has_valid_data) {
      snprintf((char*)rec_buf, REC_BUF_SIZE, "SENSORHOUR.%02d=", hour(sensor_day_g[h].timestamp));
      len=strlen((char*)rec_buf);
      memcpy((char*)rec_buf+len, &sensor_day_g[h], sizeof(SensorHour));
      sendSocketServer(rec_buf, len+sizeof(SensorHour));
      receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000);
      if(strcmp((char*)rec_buf, "OK")==0) {
        ack=true;
        clearSensorDataHour(h);
      }
    }
  }
  return(ack);
}





/* -------------------------------------------------------------------
 *  
 */
void loop() {
  static int rc, i, cnt, pps;
  int temp_prob;  // debug-dummy
  static int temp[MAX_SENSOR_COUNT];
  static time_t sysTime;

  // ------------------------------------------------------------
  // Zeit pro Durchlauf fix halten
  // ------------------------------------------------------------
  sysTime=now();
  Serial.print("SYS:"); printRTC(sysTime);  Serial.println();

  // ------------------------------------------------------------
  // Minuten- und Stunden-Umschaltung für die Daten-Speicherung
  // ------------------------------------------------------------
  if(minute(sysTime)!=minute(newMinute)) {
    storeSensorData(newMinute);
    clearSensorData_subMinuteArray();
    Serial.println("neue Minute ---------------------------");
    newMinute=sysTime;
  }
  if(hour(sysTime)!=hour(newHour)) {
    nextSensorHour(sysTime);
    Serial.println("neue Stunde ###########################");
    newHour=sysTime;
  }

  // ------------------------------------------------------------
  // Z-Pumpe gemäß Zeiten-Bitmap einschalten
  // ------------------------------------------------------------
  pps=getPowerOnStateForTime(sysTime);
  if(pump_on==0 && pps==1) {
    // wenn die Z-Pumpe aus ist und laut Zeiten-Bitmap jetzt an sein soll
    pump_on=1;    // Z-Pumpe an
  }
  Serial.print("pump_on=");   Serial.println(pump_on);

  // ------------------------------------------------------------
  // Sensoren lesen und gemäß Funktions-Zuordnung speichern
  // ------------------------------------------------------------
  for(i=0; i<sensor_count_g; i++) {
    if(ROMindex_g[i]=='E')      temp[0]=readDS18B20_CRC(i);
    else if(ROMindex_g[i]=='L') temp[1]=readDS18B20_CRC(i);
    else if(ROMindex_g[i]=='Z') temp[2]=readDS18B20_CRC(i);
    else if(ROMindex_g[i]=='R') temp[3]=readDS18B20_CRC(i);
  }
  storeSensorData_subMinute(temp[0], temp[1], temp[2], temp[3], pump_on);

  // ------------------------------------------------------------
  // Z-Pumpe gemäß Zeiten-Bitmap und Temperatur ausschalten
  // ------------------------------------------------------------
  if(pump_on==1 && pps!=1) {
    // wenn die Z-Pumpe an ist und laut Zeiten-Bitmap jetzt nicht mehr an sein muss
    if(leg_mode==0 && (temp[2]/16.0)>Threshold[0]) {
      // wenn nicht der Legionellen-Modus läuft und Sensor(Z)>45°C
      pump_on=0;    // Z-Pumpe aus
    }
  }

  // ------------------------------------------------------------
  // Z-Pumpe entspr. Legionellen-Funktion schalten
  // ------------------------------------------------------------
  if(sysTime>leg_date && ((temp[0]/16.0)>Threshold[1] || (temp[2]/16.0)>Threshold[2])) {
    // Legionellen-Modus war heute noch nicht an und (Sensor(E)>56°C oder Sensor(L)>60°C)
    pump_on=1;                  // Z-Pumpe ein
    leg_mode=1;                 // Legionellen-Modus ein
    leg_time=sysTime+60*60;     // in einer Stunde frühestens Z-Pumpe ausschalten
    leg_date=sysTime+60*60+24;  // frühestens morgen wieder Legionellen-Modus aktivieren
  }
  if(leg_mode==1 && sysTime>leg_time && (temp[2]/16.0)>Threshold[3]) {
    // Legionellen-Modus ist an und bereits eine Stunde gelaufen und
    // Sensor(Z)>52°C
    leg_mode=0;   // Legionellen-Modus aus
    pump_on=0;    // Z-Pumpe aus
  }
  if(leg_mode==1 && sysTime>(leg_time+60*60)) {
    // Legionellen-Modus ist an und bereits zwei Stunde gelaufen
    leg_mode=0;   // Legionellen-Modus aus
    pump_on=0;    // Z-Pumpe aus
  }

  // ------------------------------------------------------------
  // Debug
  // ------------------------------------------------------------
  Serial.print("Sensoren: E="); Serial.print(temp[0]/16.0);
  Serial.print("   L="); Serial.print(temp[1]/16.0);
  Serial.print("   Z="); Serial.print(temp[2]/16.0);
  Serial.print("   R="); Serial.println(temp[3]/16.0);
  /*if(temp[0]/16.0>25.0) {   // Debug-Dummy
    temp_prob=1;
  } else {
    temp_prob=0;
  }*/
  temp_prob=0;

  // ------------------------------------------------------------
  // Sync-Steuerung
  // ------------------------------------------------------------
  for(i=0; i<NUMBER_SYNCTIMES; i++) {
    if(hour(sysTime)==Sync_time[i][0] && minute(sysTime)==Sync_time[i][1] && sysTime>blockSync) {
      // wenn jetzt SyncZeit ist und in dieser Minute noch nicht hierher verzweigt wurde
      nextSync=sysTime;
      blockSync=sysTime+60;  // für den Rest dieser Minute hier das Setzen von nextSync blocken
      Serial.print("nextSync=");  printTimestamp(nextSync);  Serial.println();
      break;
    }
  }
  if(nextSync>0 && sysTime>=nextSync) {
    if(doFullSync(false)) {
      nextSync=0;     // Sync war erfolgreich
    } else {
      nextSync+=5*60; // Sync war nicht erfolgreich - in fünf Minuten erneut probieren
    }
  }

  // ------------------------------------------------------------
  // Debug
  // ------------------------------------------------------------
  Serial.print("SyncTimes=[");
  for(i=0; i<NUMBER_SYNCTIMES; i++) {
    Serial.print("(");
    Serial.print(Sync_time[i][0], DEC);
    Serial.print(", ");
    Serial.print(Sync_time[i][1], DEC);
    Serial.print("), ");
  }
  Serial.println(" ]");
  Serial.print("nextSync=");  printTimestamp(nextSync);  Serial.println();
  Serial.println();

  // ------------------------------------------------------------
  // Relais Ansteuerung
  // ------------------------------------------------------------
  if(pump_on==1) {
    digitalWrite(RELAIS_PIN, HIGH);   // Relais ein
  } else {
    digitalWrite(RELAIS_PIN, LOW);    // Relais aus
  }

  // ------------------------------------------------------------
  // LED Ansteuerung
  // ------------------------------------------------------------
  cnt=0;
  led_rt=led_bl=HIGH;
  while(now()<sysTime+5) {
    if((pump_on==1 && cnt%10==0) || (pump_on==0 && cnt%50==0)) {
      led_bl^=1;
    }
    if((temp_prob==1 && cnt%5==0)) {
      led_rt^=1;
    }
    digitalWrite(LED_RT_PIN, led_rt);
    digitalWrite(LED_BL_PIN, led_bl);
    led_bl=HIGH;
    cnt++;
    delay(50);  // 5x = 1/4 Sek., 20x = 1 Sek.
  }

}

