#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import sys
import sqlite3
import datetime
import math

class Database():
  def __init__(self, dbname):
    if os.path.isfile(dbname)==False:
      initDB=True
    else:
      initDB=False

    self.connection=sqlite3.connect(dbname)
    self.cursor=self.connection.cursor()

    if initDB==True:
      self.createDatabase()

  # ###########################################################
  # Legt die Tabellen an.
  def createDatabase(self):
    # die Tabelle für die netxml-Daten
    self.cursor.execute('CREATE TABLE network'                              \
                        ' (bssid      VARCHAR PRIMARY KEY NOT NULL UNIQUE,' \
                        '  name       VARCHAR NOT NULL,'                    \
                        '  crypto     VARCHAR,'                             \
                        '  channel    VARCHAR,'                             \
                        '  first_seen VARCHAR,'                             \
                        '  last_seen  VARCHAR)')

    # die Tabelle für die gpsxml-Daten
    self.cursor.execute('CREATE TABLE gpspoint' \
                        ' (id         INTEGER NOT NULL PRIMARY KEY,'  \
                        '  lat        VARCHAR NOT NULL,'              \
                        '  lon        VARCHAR NOT NULL,'              \
                        '  bssid      VARCHAR NOT NULL,'              \
                        '  dbm        VARCHAR,'                       \
                        '  last_seen  VARCHAR)')
    self.cursor.execute('CREATE UNIQUE INDEX nodupe1 ON gpspoint (lat, lon, bssid)')
    self.cursor.execute('CREATE INDEX refidx1 ON gpspoint (bssid)')

    # die Tabelle für die berechneten Daten
    self.cursor.execute('CREATE TABLE ap_loc' \
                        ' (id           INTEGER NOT NULL PRIMARY KEY,'  \
                        '  lat          VARCHAR NOT NULL,'              \
                        '  lon          VARCHAR NOT NULL,'              \
                        '  bssid        VARCHAR NOT NULL,'              \
                        '  calc_points  INTEGER,'                       \
                        '  last_calc    VARCHAR)')
    self.cursor.execute('CREATE UNIQUE INDEX refidx2 ON ap_loc (bssid)')

    self.cursor.execute('CREATE VIEW AccessPoints AS' \
                          ' SELECT a.lat, a.lon, a.bssid, n.name, n.channel, n.crypto, a.calc_points, n.first_seen, n.last_seen'  \
                          ' FROM network as n, ap_loc as a WHERE n.bssid=a.bssid')

    self.connection.commit()

  # ######################################################################
  def commit(self):
    self.connection.commit()

  # ######################################################################
  def getAllAPs(self):
    c=self.connection.cursor()
    c.execute('SELECT * FROM AccessPoints')
    fs=c.fetchall()
    return(fs)

  # ######################################################################
  def insertNetwork(self, bssid, essid, chan, enc, fseen):
    c=self.connection.cursor()
    c.execute('SELECT bssid, name, crypto, channel, first_seen, last_seen FROM network WHERE bssid=?', (bssid,))
    fs=c.fetchone()
    update=False
    if fs is not None:        # wenn BSSID schon in der DB enthalten ist
      if fs[4]<fseen:         # wenn einzufügender Satz neuer als first_seen ist
        if fs[5] is not None: # wenn last_seen bereits gesetzt ist
          if fs[5]<fseen:     # wenn einzufügender Satz neuer als last_seen ist
            update=True       # UPDATE, weil last_seen älter als "fseen" ist
        else:
          update=True         # UPDATE, weil last_seen noch unbelegt ist
      if update:
        self.cursor.execute('UPDATE network SET name=?, crypto=?, channel=?, last_seen=? WHERE bssid=?', (essid, enc, chan, fseen, bssid))
#        print("UPDATE network")
    else:
      try:
        self.cursor.execute('INSERT INTO network (bssid, name, crypto, channel, first_seen) VALUES (?, ?, ?, ?, ?)', (bssid, essid, enc, chan, fseen))
#        print("INSERT network")
      except Exception as e:
        #print("error, insertNetwork", str(e))
        pass

  # ######################################################################
  def insertGPSpoint(self, bssid, lat, lon, dbm, seen):
    c=self.connection.cursor()
    c.execute('SELECT count(*) FROM network WHERE bssid=?', (bssid,))
    fs=c.fetchone()
    if fs[0]==0:  # BSSID existiert nicht in network
      return

    c.execute('SELECT lat, lon, bssid, dbm, last_seen FROM gpspoint WHERE lat=? AND lon=? AND bssid=?', (lat, lon, bssid))
    fs=c.fetchone()
    if fs is not None:  # wenn Koordinate zur BSSID schon in der DB enthalten ist
      if fs[4]<seen:    # wenn einzufügender Satz neuer ist
        self.cursor.execute('UPDATE gpspoint SET dbm=?, last_seen=? WHERE lat=? AND lon=? AND bssid=?', (dbm, seen, lat, lon, bssid))
#        print("UPDATE gpspoint")
    try:
      self.cursor.execute('INSERT INTO gpspoint (lat, lon, bssid, dbm, last_seen) VALUES (?, ?, ?, ?, ?)', (lat, lon, bssid, dbm, seen))
#      print("INSERT gpspoint")
    except Exception as e:
      #print("error, insertGPSpoint", str(e))
      pass

  # ######################################################################
  def getAPloc(self, bssid):
    c=self.connection.cursor()
    c.execute('SELECT lat, lon, bssid, calc_points, last_calc FROM ap_loc WHERE bssid=?', (bssid,))
    fs=c.fetchone()
    return(fs)

  # ######################################################################
  def insertAPloc(self, bssid, lat, lon, pts, dt):
    c=self.connection.cursor()
    c.execute('SELECT count(*) FROM ap_loc WHERE bssid=?', (bssid,))
    fs=c.fetchone()
    if fs[0]>0:   # BSSID existiert bereits in ap_loc
      self.cursor.execute('UPDATE ap_loc SET lat=?, lon=?, calc_points=?, last_calc=? WHERE bssid=?', (lat, lon, pts, dt, bssid))
    else:
      try:
        self.cursor.execute('INSERT INTO ap_loc (lat, lon, bssid, calc_points, last_calc) VALUES (?, ?, ?, ?, ?)', (lat, lon, bssid, pts, dt))
      except Exception as e:
        #print("error, insertAPloc", str(e))
        pass

  # ######################################################################
  def youngestPointTimestamp(self, bssid):
    c=self.connection.cursor()
    c.execute('SELECT max(last_seen) FROM (SELECT last_seen FROM gpspoint WHERE bssid=? GROUP BY last_seen)', (bssid,))
    fs=c.fetchone()
    if fs is not None:
      return(fs[0])
    return(None)

  # ######################################################################
  def needsCalculation(self, bssid, yp):
    #yp=self.youngestPointTimestamp(bssid)
    ap=self.getAPloc(bssid)   # lat, lon, bssid, calc_points, last_calc
    if ap is not None:  # BSSID befindet sich bereits in ap_loc
      print(ap)
      if ap[4]<yp:      # ap_loc-Satz ist älter als jüngster gpspoint-Satz
        return(True)
      else:             # ap_loc-Satz ist genauso alt oder jünger als jüngster gpspoint-Satz
        return(False)
    else:               # BSSID noch nicht in ap_loc
      return(True)

  # ######################################################################
  def getAllBSSIDs(self):
    c=self.connection.cursor()
    c.execute('SELECT bssid, name, crypto, channel, first_seen, last_seen FROM network')
    fs=c.fetchall()
    return(fs)

  # ######################################################################
  def getGPSpoints(self, bssid):
    #print("getGPSpoints", bssid)
    c=self.connection.cursor()
    #c.execute('SELECT lat, lon, bssid, dbm, last_seen FROM gpspoint WHERE bssid=?', (bssid,))
    c.execute('SELECT lat, lon, dbm FROM gpspoint WHERE bssid=?', (bssid,))
    fs=c.fetchall()
    return(fs)


# ######################################################################
# a.lat, a.lon, a.bssid, n.name, n.channel, n.crypto, a.calc_points, n.first_seen, n.last_seen
if __name__ == '__main__':
  db=Database("/home/dede/daten/wardriving.sqlite")
  aps=db.getAllAPs()
  print('<?xml version="1.0" encoding="UTF-8" standalone="yes" ?>')
  print('<gpx version="1.1" creator="RasPi Tracker" xmlns="http://www.topografix.com/GPX/1/1" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd">')

  #aps=db.getGPSpoints("34:81:C4:37:3F:53")
  for ap in aps:
    print('  <wpt lat="%s" lon="%s"><desc>(%s (%s), chan=%s, crypt=%s, pts=%s)</desc></wpt>'%(ap[0], ap[1], ap[3], ap[2], ap[4], ap[5], ap[6]))
    #print('  <wpt lat="%s" lon="%s"><desc>%s</desc></wpt>'%(ap[0], ap[1], ap[2]))
  #  print("(%s, %s),"%(ap[0], ap[1]))


  print('</gpx>')
