#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import sys
import sqlite3
import datetime
import math

MACS_TO_IGNORE=["B8:27:EB:65:BA:64"]

class Database():
  def __init__(self, dbname):
    if os.path.isfile(dbname)==False:
      initDB=True
    else:
      initDB=False

    self.connection=sqlite3.connect(dbname)
    self.cursor=self.connection.cursor()

    if initDB==True:
      self.createDatabase()

  # ###########################################################
  # Legt die Tabellen an.
  def createDatabase(self):
    # die Tabelle für die netxml-Daten
    self.cursor.execute('CREATE TABLE network'                              \
                        ' (bssid      VARCHAR PRIMARY KEY NOT NULL UNIQUE,' \
                        '  name       VARCHAR NOT NULL,'                    \
                        '  crypto     VARCHAR,'                             \
                        '  channel    VARCHAR,'                             \
                        '  first_seen VARCHAR,'                             \
                        '  last_seen  VARCHAR)')

    # die Tabelle für die gpsxml-Daten
    self.cursor.execute('CREATE TABLE gpspoint' \
                        ' (id         INTEGER NOT NULL PRIMARY KEY,'  \
                        '  lat        VARCHAR NOT NULL,'              \
                        '  lon        VARCHAR NOT NULL,'              \
                        '  bssid      VARCHAR NOT NULL,'              \
                        '  dbm        VARCHAR,'                       \
                        '  last_seen  VARCHAR)')
    self.cursor.execute('CREATE UNIQUE INDEX nodupe1 ON gpspoint (lat, lon, bssid)')
    self.cursor.execute('CREATE INDEX refidx1 ON gpspoint (bssid)')

    # die Tabelle für die berechneten Daten
    self.cursor.execute('CREATE TABLE ap_loc' \
                        ' (id           INTEGER NOT NULL PRIMARY KEY,'  \
                        '  lat          VARCHAR NOT NULL,'              \
                        '  lon          VARCHAR NOT NULL,'              \
                        '  bssid        VARCHAR NOT NULL,'              \
                        '  calc_points  INTEGER,'                       \
                        '  last_calc    VARCHAR)')
    self.cursor.execute('CREATE UNIQUE INDEX refidx2 ON ap_loc (bssid)')

    self.cursor.execute('CREATE VIEW AccessPoints AS' \
                          ' SELECT a.lat, a.lon, a.bssid, n.name, n.channel, n.crypto, a.calc_points, n.first_seen, n.last_seen'  \
                          ' FROM network as n, ap_loc as a WHERE n.bssid=a.bssid')

    self.connection.commit()

  # ######################################################################
  def commit(self):
    self.connection.commit()

  # ######################################################################
  def getAllAPs(self):
    c=self.connection.cursor()
    c.execute('SELECT * FROM AccessPoints')
    fs=c.fetchall()
    return(fs)

  # ######################################################################
  def insertNetwork(self, bssid, essid, chan, enc, fseen):
    c=self.connection.cursor()
    c.execute('SELECT bssid, name, crypto, channel, first_seen, last_seen FROM network WHERE bssid=?', (bssid,))
    fs=c.fetchone()
    update=False
    if fs is not None:        # wenn BSSID schon in der DB enthalten ist
      if fs[4]<fseen:         # wenn einzufügender Satz neuer als first_seen ist
        if fs[5] is not None: # wenn last_seen bereits gesetzt ist
          if fs[5]<fseen:     # wenn einzufügender Satz neuer als last_seen ist
            update=True       # UPDATE, weil last_seen älter als "fseen" ist
        else:
          update=True         # UPDATE, weil last_seen noch unbelegt ist
      if update:
        self.cursor.execute('UPDATE network SET name=?, crypto=?, channel=?, last_seen=? WHERE bssid=?', (essid, enc, chan, fseen, bssid))
    else:
      try:
        self.cursor.execute('INSERT INTO network (bssid, name, crypto, channel, first_seen) VALUES (?, ?, ?, ?, ?)', (bssid, essid, enc, chan, fseen))
      except Exception as e:
        #print("error, insertNetwork", str(e))
        pass

  # ######################################################################
  def insertGPSpoint(self, bssid, lat, lon, dbm, seen):
    c=self.connection.cursor()
    c.execute('SELECT count(*) FROM network WHERE bssid=?', (bssid,))
    fs=c.fetchone()
    if fs[0]==0:  # BSSID existiert nicht in network
      return

    c.execute('SELECT lat, lon, bssid, dbm, last_seen FROM gpspoint WHERE lat=? AND lon=? AND bssid=?', (lat, lon, bssid))
    fs=c.fetchone()
    if fs is not None:  # wenn Koordinate zur BSSID schon in der DB enthalten ist
      if fs[4]<seen:    # wenn einzufügender Satz neuer ist
        self.cursor.execute('UPDATE gpspoint SET dbm=?, last_seen=? WHERE lat=? AND lon=? AND bssid=?', (dbm, seen, lat, lon, bssid))
    try:
      self.cursor.execute('INSERT INTO gpspoint (lat, lon, bssid, dbm, last_seen) VALUES (?, ?, ?, ?, ?)', (lat, lon, bssid, dbm, seen))
    except Exception as e:
      #print("error, insertGPSpoint", str(e))
      pass

  # ######################################################################
  def getAPloc(self, bssid):
    c=self.connection.cursor()
    c.execute('SELECT lat, lon, bssid, calc_points, last_calc FROM ap_loc WHERE bssid=?', (bssid,))
    fs=c.fetchone()
    return(fs)

  # ######################################################################
  def insertAPloc(self, bssid, lat, lon, pts, dt):
    c=self.connection.cursor()
    c.execute('SELECT count(*) FROM ap_loc WHERE bssid=?', (bssid,))
    fs=c.fetchone()
    if fs[0]>0:   # BSSID existiert bereits in ap_loc
      self.cursor.execute('UPDATE ap_loc SET lat=?, lon=?, calc_points=?, last_calc=? WHERE bssid=?', (lat, lon, pts, dt, bssid))
    else:
      try:
        self.cursor.execute('INSERT INTO ap_loc (lat, lon, bssid, calc_points, last_calc) VALUES (?, ?, ?, ?, ?)', (lat, lon, bssid, pts, dt))
      except Exception as e:
        #print("error, insertAPloc", str(e))
        pass

  # ######################################################################
  def newestPointTimestamp(self, bssid):
    c=self.connection.cursor()
    c.execute('SELECT max(last_seen) FROM (SELECT last_seen FROM gpspoint WHERE bssid=? GROUP BY last_seen)', (bssid,))
    fs=c.fetchone()
    if fs is not None:
      return(fs[0])
    return(None)

  # ######################################################################
  def needsCalculation(self, bssid, yp):
    #yp=self.newestPointTimestamp(bssid)
    ap=self.getAPloc(bssid)   # lat, lon, bssid, calc_points, last_calc
    if ap is not None:  # BSSID befindet sich bereits in ap_loc
      if ap[4]<yp:      # ap_loc-Satz ist älter als neuster gpspoint-Satz
        return(True)
      else:             # ap_loc-Satz ist gleich oder neuer als neuster gpspoint-Satz
        return(False)
    else:               # BSSID noch nicht in ap_loc
      return(True)

  # ######################################################################
  def getAllBSSIDs(self):
    c=self.connection.cursor()
    c.execute('SELECT bssid, name, crypto, channel, first_seen, last_seen FROM network')
    fs=c.fetchall()
    return(fs)

  # ######################################################################
  def getGPSpoints(self, bssid):
    #print("getGPSpoints", bssid)
    c=self.connection.cursor()
    #c.execute('SELECT lat, lon, bssid, dbm, last_seen FROM gpspoint WHERE bssid=?', (bssid,))
    c.execute('SELECT lat, lon, dbm FROM gpspoint WHERE bssid=?', (bssid,))
    fs=c.fetchall()
    return(fs)


# ######################################################################
# Quelle: http://stackoverflow.com/a/14115494/3588613
class PolygonCentroid():
  def bottommost_index_for_polygon(self, polygon):
      bottommost_index = 0
      for index, point in enumerate(polygon):
          if (point['y'] < polygon[bottommost_index]['y']):
              bottommost_index = index
      return bottommost_index

  def angle_for_vector(self, start_point, end_point):
      y = end_point['y'] - start_point['y']
      x = end_point['x'] - start_point['x']
      angle = 0
      if (x == 0):
          if (y > 0):
              angle = 90.0
          else:
              angle = 270.0
      elif (y == 0):
          if (x > 0):
              angle = 0.0
          else:
              angle = 180.0
      else:
          angle = math.degrees(math.atan((y+0.0)/x))
          if (x < 0):
              angle += 180
          elif (y < 0):
              angle += 360
      return angle

  def convex_hull_for_polygon(self, polygon):
      starting_point_index = self.bottommost_index_for_polygon(polygon)
      convex_hull = [polygon[starting_point_index]]
      polygon_length = len(polygon)
      hull_index_candidate = 0 #arbitrary
      previous_hull_index_candidate = starting_point_index
      previous_angle = 0
      cnt=0
      while True:
          cnt+=1
          if cnt>(len(polygon)*2):  # manchmal frisst er sich fest
            return(None)            # daher lieber abwürgen und den Mittelpunkt anders bestimmen

          smallest_angle = 360
          for j in range(0,polygon_length):
              if (previous_hull_index_candidate == j):
                  continue
              current_angle = self.angle_for_vector(polygon[previous_hull_index_candidate], polygon[j])
              if (current_angle < smallest_angle and current_angle > previous_angle):
                  hull_index_candidate = j
                  smallest_angle = current_angle

          if (hull_index_candidate == starting_point_index): # we've wrapped all the way around
              break
          else:
              convex_hull.append(polygon[hull_index_candidate])
              previous_angle = smallest_angle
              previous_hull_index_candidate = hull_index_candidate
      return convex_hull

  def area_for_polygon(self, polygon):
      result = 0
      imax = len(polygon) - 1
      for i in range(0,imax):
          result += (polygon[i]['x'] * polygon[i+1]['y']) - (polygon[i+1]['x'] * polygon[i]['y'])
      result += (polygon[imax]['x'] * polygon[0]['y']) - (polygon[0]['x'] * polygon[imax]['y'])
      return result / 2.

  def centroid_for_polygon(self, polygon):
      area = self.area_for_polygon(polygon)
      imax = len(polygon) - 1
      result_x = 0
      result_y = 0
      for i in range(0,imax):
          result_x += (polygon[i]['x'] + polygon[i+1]['x']) * ((polygon[i]['x'] * polygon[i+1]['y']) - (polygon[i+1]['x'] * polygon[i]['y']))
          result_y += (polygon[i]['y'] + polygon[i+1]['y']) * ((polygon[i]['x'] * polygon[i+1]['y']) - (polygon[i+1]['x'] * polygon[i]['y']))
      result_x += (polygon[imax]['x'] + polygon[0]['x']) * ((polygon[imax]['x'] * polygon[0]['y']) - (polygon[0]['x'] * polygon[imax]['y']))
      result_y += (polygon[imax]['y'] + polygon[0]['y']) * ((polygon[imax]['x'] * polygon[0]['y']) - (polygon[0]['x'] * polygon[imax]['y']))
      result_x /= (area * 6.0)
      result_y /= (area * 6.0)
      return {'x': result_x, 'y': result_y}

  def getCenter(self, coords):
    data=[]
    for lat, lon, dbm in coords:
      if int(dbm)<-90:
        continue
      flat=float(lat)
      flon=float(lon)
      if (flat, flon) not in data:
        data.append({"x":flat, "y":flon})
    d=self.convex_hull_for_polygon(data)
    if d is None:
      return(None)
    d2=self.centroid_for_polygon(d)
    return(d2["x"], d2["y"])

# ######################################################################
#
def getCenterSimple(coords):
  if len(coords)==0:  # coords==[] würde zu "division by zero" führen
    return(None)
  sum_fak=sum_lat=sum_lon=0
  nodupes=[]
  for lat, lon, dbm in coords:
    flat=float(lat)
    flon=float(lon)
    if (flat, flon) not in nodupes:
      nodupes.append((flat, flon))
      sum_fak+=1
      sum_lat+=flat
      sum_lon+=flon
  return(sum_lat/sum_fak, sum_lon/sum_fak)


# ######################################################################
# Liefert zum ersten Auftreten von "tag" in "xml" den Inhalt zwischen
# dem öffnenden und dem schließenden Tag, inklusive dieser zwei Tags.
# Als zweiter Rückgabewert wird die Position in "xml" hinter dem
# schließenden Tag geliefert.
def xml_get(xml, tag, start_pos=0):
  p1=xml.find("<%s"%(tag,), start_pos)
  if p1>=0:
    end_tag="</%s>"%(tag,)
    p2=xml.find(end_tag, start_pos)
    if p2>=0:
      return(xml[p1:p2+len(end_tag)], p1, p2+len(end_tag))
  return(None, 0, 0)

# ######################################################################
# Liefert zum ersten Auftreten von "tag" in "xml" den Inhalt zwischen
# dem öffnenden und dem schließenden Tag.
# Beispiel:
#      xml='...me><extensions><speedkmh>1.37</speedkmh><sat>10</sa...'
#      tag='speedkmh'
# Rückgabe='1.37'
def xml_get_simple_attr(xml, tag, start_pos=0):
  if xml is None:
    return("", 0)
  start_tag="<%s>"%(tag,)
  end_tag="</%s>"%(tag,)
  p1=xml.find(start_tag, start_pos)
  if p1>=0:
    p2=xml.find(end_tag, p1)
    if p2>0:
      return(xml[p1+len(start_tag):p2], p2)
  return("", 0)

# ######################################################################
# Liefert zum ersten Auftreten von "tag" in "xml" den Inhalt zwischen
# "tag" und ">". Das Zeichen ">" kann somit im späteren Rückgabewert
# nicht vorkommen - auch nicht in zwischen Anführungszeichen.
# Beispiel:
#      xml='<trkpt lat="54.803600000" lon="9.524133333"><time....'
#      tag='trkpt'
# Rückgabe='lat="54.803600000" lon="9.524133333"'
def xml_get_tag_attr(xml, tag):
  start_tag="<%s "%(tag,)
  p1=xml.find(start_tag)
  if p1>=0:
    p2=xml.find(">")
    if p2>0:
      return(xml[p1+len(start_tag):p2])
  return("")

# ######################################################################
#
def xml_get_compl_attr(xml, tag):
  if xml is None:
    return("")
  start_tag="<%s "%(tag,)
  end_tag="</%s>"%(tag,)
  p1=xml.find(start_tag)
  if p1>=0:
    p2=xml.find(end_tag)
    if p2>0:
      p3=xml.find(">", p1)
      if p3>=0:
        return(xml[p3+1:p2])
  return("")


# ######################################################################
#
if __name__ == '__main__':
  if len(sys.argv)<2:
    print("Aufruf: %s netxml-Datei"%(sys.argv[0],))
    sys.exit()

  if not os.path.isfile(sys.argv[1]):
    print("keine Datei:", sys.argv[1])
    sys.exit()

  full_pth=os.path.abspath(sys.argv[1])
  fn=os.path.splitext(full_pth)
  if len(fn)==2:
    if fn[1]==".netxml":
      netxml=full_pth
      gpsxml=fn[0]+".gpsxml"
    elif fn[1]==".gpsxml":
      netxml=fn[0]+".netxml"
      gpsxml=full_pth
  else:
    sys.exit()

  if not os.path.isfile(netxml):
    print(".netxml-Datei nicht gefunden!")
    sys.exit()
  if not os.path.isfile(gpsxml):
    print(".gpsxml-Datei nicht gefunden!")
    sys.exit()

  db=Database("/home/dede/daten/wardriving.sqlite")

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Netzwerke auslesen und in DB schreiben
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  fl=open(netxml, "r")
  filedata=fl.read()
  fl.close()
  p2=1
  while p2>0:
    network, p1, p2=xml_get(filedata, "wireless-network", p2)
    if p2>0:
      bssid, egal=xml_get_simple_attr(network, "BSSID")
      ssid_block, q1, q2=xml_get(network, "SSID")
      enc, s1=xml_get_simple_attr(ssid_block, "encryption")
      while s1>0:
        enc2, s1=xml_get_simple_attr(ssid_block, "encryption", s1)
        if enc2.strip()!="":
          enc+=", " + enc2
      essid=xml_get_compl_attr(ssid_block, "essid")
      chan, egal=xml_get_simple_attr(network, "channel")
      #<seen-time>Sat Jul  9 18:39:39 2016</seen-time>
      seen, egal=xml_get_simple_attr(network, "seen-time")
      seen_dt=datetime.datetime.strptime(seen, '%a %b %d %H:%M:%S %Y')
      if bssid not in MACS_TO_IGNORE:
        if essid!="":
          db.insertNetwork(bssid, essid, chan, enc, seen_dt.strftime("%Y-%m-%d %H:%M:%S"))

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # GPSpoints auslesen und in DB schreiben
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  fl=open(gpsxml, "r")
  filedata=fl.read()
  fl.close()
  for ln in filedata.split("\n"):
    if ln.strip().startswith("<gps-run"):
      time_sp=ln.find('start-time="')
      if time_sp>=0:
        time_ep=ln.find('"', time_sp+12)
        seen=ln[time_sp+12:time_ep]
        seen_dt=datetime.datetime.strptime(seen, '%a %b %d %H:%M:%S %Y')
        stime=seen_dt.strftime("%Y-%m-%d %H:%M:%S")
      break

  for ln in filedata.split("\n"):
    if ln.strip().startswith("<gps-point"):
      bssid=lat=lon=dbm=""

      bss_sp=ln.find('bssid="')
      if bss_sp>=0:
        bss_ep=ln.find('"', bss_sp+7)
        bssid=ln[bss_sp+7:bss_ep]

      lat_sp=ln.find('lat="')
      if lat_sp>=0:
        lat_ep=ln.find('"', lat_sp+5)
        lat=ln[lat_sp+5:lat_ep]
      
      lon_sp=ln.find('lon="')
      if lon_sp>=0:
        lon_ep=ln.find('"', lon_sp+5)
        lon=ln[lon_sp+5:lon_ep]

      dbm_sp=ln.find('signal_dbm="')
      if dbm_sp>=0:
        dbm_ep=ln.find('"', dbm_sp+12)
        dbm=ln[dbm_sp+12:dbm_ep]
      if bssid not in MACS_TO_IGNORE:
        if bssid!="" and lat!="" and lon!="" and dbm!="":
          db.insertGPSpoint(bssid, lat, lon, dbm, stime)

  db.commit()
  # SELECT bssid, count(*) FROM gpspoint group by bssid

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Mittelpunkte berechnen
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  pc=PolygonCentroid()
  #dt=datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")

  n=db.getAllBSSIDs() # bssid, name, crypto, channel, first_seen, last_seen
  for i in n:
    bssid=i[0]
    dt=db.newestPointTimestamp(bssid)
    if dt is not None and db.needsCalculation(bssid, dt):
      p=db.getGPSpoints(bssid)  # lat, lon, bssid, dbm, last_seen
      cent=None
      if len(p)>3 and len(p)<200:
        cent=pc.getCenter(p)
        if cent is None:
          print("abgewürgt")
      if cent is None:  #len(p)>0:
        cent=getCenterSimple(p)
      if cent is not None:
        db.insertAPloc(bssid, cent[0], cent[1], len(p), dt)
    else:
      pass
      #print("übersprungen", bssid, dt)
  db.commit()
