#!/usr/bin/env python3
# -*- coding: UTF-8 -*-

# Ein Script zum Einlesen von .netxml- und .gpsxml-Dateien in eine
# Datenbank und anschließender Mittelpunkts-Berechnung der einzelen
# <gps-point>-Sätze pro BSSID.
#
# Detlev Ahlgrimm 07.2016
#
# v1.0    14.07.2016  erste Version


import os
import sys
from lxml import etree    # python3-lxml-3.3.5-2.1.3.x86_64
import datetime

from Database4wd        import Database
from PolygonCentroid4wd import PolygonCentroid

DATABASE_FILE="/home/dede/daten/wardriving.sqlite"
MACS_TO_IGNORE=["B8:27:EB:65:BA:64"]  # MAC des RaspAP

# ######################################################################
# Liefert zu einer Liste aus Koordinaten mit Dämpfung im Format
# [(lat, lon, dbm), ...] ein Tupel (lat, lon) mit den Mittelwerten.
def getCenterSimple(coords):
  sum_fak=sum_lat=sum_lon=0
  nodupes=list()
  for lat, lon, dbm in coords:      # über alle Koordinaten
    try:
      flat=float(lat)
      flon=float(lon)
    except:
      return(None)
    if (flat, flon) not in nodupes: # Duplikate ausblenden
      nodupes.append((flat, flon))
      sum_fak+=1
      sum_lat+=flat                 # jeweilige Summe bilden
      sum_lon+=flon
  if sum_fak>0: # "division by zero" abfangen
    return(sum_lat/sum_fak, sum_lon/sum_fak)  # und final durch Anzahl teilen
  return(None)

# ######################################################################
# Liefert zu den Script-Übergabe-Parametern die Dateinamen in zwei
# Listen.
def getInputFiles():
  netxml=list()
  gpsxml=list()
  if len(sys.argv)==2:                    # nur ein Parameter übergeben
    if os.path.isfile(sys.argv[1]):       # einzelner Dateiname übergeben
      fn, ext=os.path.splitext(sys.argv[1])
      if ext.lower() in [".netxml", ".gpsxml"]: # Extension soll schon stimmen
        netxml.append(fn+".netxml")
        gpsxml.append(fn+".gpsxml")
    elif os.path.isdir(sys.argv[1]):      # Verzeichnisname übergeben
      fl=os.listdir(sys.argv[1])
      for f in fl:                        # über alle Dateien im Verzeichnis
        if f.lower().startswith("kismet"):# die passenden rauspicken
          if f.lower().endswith(".netxml"):
            netxml.append(f)
          elif f.lower().endswith(".gpsxml"):
            gpsxml.append(f)
  else:                                   # mehrere Parameter bzw. Wildcard übergeben
    for f in range(1, len(sys.argv)):     # übergebene Parameter auf die Listen verteilen
      if sys.argv[f].lower().endswith(".netxml"):
        netxml.append(sys.argv[f])
      elif sys.argv[f].lower().endswith(".gpsxml"):
        gpsxml.append(sys.argv[f])

  netxml2=list()
  gpsxml2=list()
  for fn in netxml:         # in den Ergebnis-Listen sollen nur vorhandene Dateinen stehen
    if os.path.isfile(fn):
      netxml2.append(fn)
  for fn in gpsxml:
    if os.path.isfile(fn):
      gpsxml2.append(fn)
  netxml2.sort()            # Listen so sortieren, dass die ältesten Dateien vorne stehen
  gpsxml2.sort()
  return(netxml2, gpsxml2)


# ######################################################################
#
if __name__ == '__main__':
  if len(sys.argv)<2:
    print("Aufruf: %s <verz> | <datei> [<datei>] ... [<datei>]"%(sys.argv[0],))
    sys.exit()

  netxmls, gpsxmls=getInputFiles()

  if len(netxmls)!=len(gpsxmls):
    print(".netxml- und .gpsxml-Dateien müssen immer paarig auftreten!")
    sys.exit()
  if len(netxmls)==0:
    print("Keine .netxml- oder .gpsxml-Dateien gefunden.")
    sys.exit()

  db=Database(DATABASE_FILE)

  cnt_net_in=cnt_net_out=cnt_net_bad=cnt_net_ign=0
  for netxml in netxmls:
    print("Bearbeite Datei", netxml)
    with open(netxml, "r") as fl:
      filedata=fl.read()
    root=etree.fromstring(filedata.encode())
    for network in root.iter("wireless-network"):   # über alle networks
      if network.get("type")=="infrastructure":     # wenn aktuelles network vom richtigen Typ ist
        cnt_net_in+=1
        enc=""
        last_seen=datetime.datetime.strptime(network.get("last-time"), '%a %b %d %H:%M:%S %Y')
        lvl1=network.getchildren()                  # eine Ebene runter
        for e1 in lvl1:                             # über alle Tags vom network-Satz
          if e1.tag.lower()=="channel":
            chan=e1.text
          elif e1.tag.lower()=="bssid":
            bssid=e1.text
          elif e1.tag.lower()=="ssid":              # unter SSID kommt eine weitere Ebene
            lvl2=e1.getchildren()
            for e2 in lvl2:                         # über alle Tags vom SSID-Satz
              if e2.tag.lower()=="essid":
                essid=e2.text
              elif e2.tag.lower()=="encryption":    # encryption kann mehrmals vorkommen
                if enc=="": enc=e2.text
                else:       enc+=", "+e2.text
        if bssid not in MACS_TO_IGNORE:
          if essid is not None and essid!="":
            cnt_net_out+=1
            db.insertNetwork(bssid, essid, chan, enc, last_seen.strftime("%Y-%m-%d %H:%M:%S"))
          else:
            cnt_net_bad+=1
        else:
          cnt_net_ign+=1
  db.commit()

  cnt_pt_in=cnt_pt_out=cnt_pt_bad=cnt_pt_ign=0
  for gpsxml in gpsxmls:
    print("Bearbeite Datei", gpsxml)
    with open(gpsxml, "r") as fl:
      filedata=fl.read()
    root=etree.fromstring(filedata.encode())
    for point in root.iter("gps-run"):              # da sollte es nur einen von geben, aber wer weiss
      lvl1=point.getchildren()                      # eine Ebene runter
      for e in lvl1:                                # über alle gps-point's
        cnt_pt_in+=1
        if e.tag.lower()=="gps-point":
          bssid=e.get("bssid")
          if bssid=="GP:SD:TR:AC:KL:OG":            # Dummy-BSSIDs ausblenden
            cnt_pt_ign+=1
            continue
          try:
            stime=datetime.datetime.fromtimestamp(int(e.get("time-sec")))
          except:
            stime=None
          lat=e.get("lat")
          lon=e.get("lon")
          dbm=e.get("signal_dbm")
          if bssid is None or lat is None or lon is None or dbm is None or stime is None:
            cnt_pt_bad+=1
            continue
          if bssid not in MACS_TO_IGNORE:
            cnt_pt_out+=1
            db.insertGPSpoint(bssid, lat, lon, dbm, stime.strftime("%Y-%m-%d %H:%M:%S"))
          else:
            cnt_pt_ign+=1
  db.commit()

  pc=PolygonCentroid()
  print("Berechne die Standorte der AccessPoints")
  n=db.getAllBSSIDs() # bssid, name, crypto, channel, first_seen, last_seen
  cnt_calc1=cnt_calc2=cnt_nocalc=cnt_err=0
  for i in n:
    bssid=i[0]
    dt=db.newestPointTimestamp(bssid)
    if dt is not None and db.needsCalculation(bssid, dt):
      p=db.getGPSpoints(bssid)  # lat, lon, dbm
      cent=None
      if len(p)>=4: # der PolygonCentroid() braucht mindestens vier Eckpunkte
        cent=pc.getCenter(p)
        if cent is not None:
          cnt_calc1+=1
      if cent is None:
        cnt_calc2+=1
        cent=getCenterSimple(p)
      if cent is None:
        cnt_err+=1
      if cent is not None:
        db.insertAPloc(bssid, cent[0], cent[1], len(p), dt)
    else:
      cnt_nocalc+=1
      pass
  db.commit()
  print("-"*51)
  print("Vorgefundene Sätze vom Typ <wireless-network>     :", cnt_net_in)
  print("In die DB geschriebene <wireless-network>-Sätze   :", cnt_net_out)
  print("Fehlerhafte / ignorierte <wireless-network>-Sätze :", cnt_net_bad)
  print("Ausgeblendete <wireless-network>-Sätze            :", cnt_net_ign)
  print("- "*26)
  print("Vorgefundene Sätze vom Typ <gps-point>            :", cnt_pt_in)
  print("In die DB geschriebene <gps-point>-Sätze          :", cnt_pt_out)
  print("Fehlerhafte / ignorierte <gps-point>-Sätze        :", cnt_pt_bad)
  print("Ausgeblendete <gps-point>-Sätze                   :", cnt_pt_ign)
  print("- "*26)
  print("Per Polygon-Schwerkunkt berechnete AP-Standorte   :", cnt_calc1)
  print("Per Koordinaten-Mittelwert berechnete AP-Standorte:", cnt_calc2)
  print("Unveränderte AP-Standorte                         :", cnt_nocalc)
  print("Nicht bestimmbare AP-Standorte                    :", cnt_err)

