#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Ein Sub-Script für das Wardriving.
# Detlev Ahlgrimm 07.2016
#
# v1.0    14.07.2016  erste Version

import os
import sqlite3

class Database():
  def __init__(self, dbname):
    if os.path.isfile(dbname)==False:
      initDB=True
    else:
      initDB=False

    self.connection=sqlite3.connect(dbname)
    self.cursor=self.connection.cursor()

    if initDB==True:
      self.createDatabase()

  # ###########################################################
  # Legt die Tabellen an.
  def createDatabase(self):
    # die Tabelle für die netxml-Daten
    self.cursor.execute('CREATE TABLE network'                        \
                        ' (id         INTEGER NOT NULL PRIMARY KEY,'  \
                        '  bssid      VARCHAR NOT NULL,'              \
                        '  name       VARCHAR NOT NULL,'              \
                        '  channel    VARCHAR,'                       \
                        '  first_seen VARCHAR,'                       \
                        '  last_seen  VARCHAR,'                       \
                        '  enc        INTEGER,'                       \
                        '  FOREIGN KEY(enc) REFERENCES encryptions(id))')
    self.cursor.execute('CREATE UNIQUE INDEX nodupe1 ON network (bssid)')

    # die Tabelle für die Verschlüsselungs-Kombinationen
    self.cursor.execute('CREATE TABLE encryptions'                    \
                        ' (id         INTEGER NOT NULL PRIMARY KEY,'  \
                        '  crypto     VARCHAR NOT NULL)')
    self.cursor.execute('CREATE UNIQUE INDEX nodupe2 ON encryptions (crypto)')

    # network mit crypto (als Klartext)
    self.cursor.execute('CREATE VIEW network_enc AS'                                                \
                          ' SELECT n.bssid, n.name, e.crypto, n.channel, n.first_seen, n.last_seen' \
                          ' FROM network as n, encryptions as e WHERE n.enc=e.id')

    # die Tabelle für die gpsxml-Daten
    self.cursor.execute('CREATE TABLE gpspoint'                       \
                        ' (id         INTEGER NOT NULL PRIMARY KEY,'  \
                        '  lat        VARCHAR NOT NULL,'              \
                        '  lon        VARCHAR NOT NULL,'              \
                        '  bssid      VARCHAR NOT NULL,'              \
                        '  dbm        VARCHAR,'                       \
                        '  last_seen  VARCHAR)')
    self.cursor.execute('CREATE UNIQUE INDEX nodupe3 ON gpspoint (lat, lon, bssid)')
    self.cursor.execute('CREATE INDEX refidx1 ON gpspoint (bssid)')

    # die Tabelle für die berechneten Daten
    self.cursor.execute('CREATE TABLE ap_loc'                           \
                        ' (id           INTEGER NOT NULL PRIMARY KEY,'  \
                        '  lat          VARCHAR NOT NULL,'              \
                        '  lon          VARCHAR NOT NULL,'              \
                        '  bssid        VARCHAR NOT NULL,'              \
                        '  calc_points  INTEGER,'                       \
                        '  last_calc    VARCHAR)')
    self.cursor.execute('CREATE UNIQUE INDEX refidx2 ON ap_loc (bssid)')

    # AP's mit Koordinaten
    self.cursor.execute('CREATE VIEW AccessPoints AS'                                                                             \
                          ' SELECT a.lat, a.lon, a.bssid, n.name, n.channel, n.crypto, a.calc_points, n.first_seen, n.last_seen'  \
                          ' FROM network_enc as n, ap_loc as a WHERE n.bssid=a.bssid')

    self.connection.commit()

  # ######################################################################
  # Änderungen festschreiben.
  def commit(self):
    self.connection.commit()

  # ######################################################################
  # Nimmt ein Netzwerk auf oder aktualisiert es, wenn es schon in der DB ist.
  def insertNetwork(self, bssid, essid, chan, enc, fseen):
    self.cursor.execute('SELECT first_seen, last_seen FROM network_enc WHERE bssid=?', (bssid,))
    fs=self.cursor.fetchone()
    update=False
    if fs is not None:        # wenn BSSID schon in der DB enthalten ist
      if fs[0]<fseen:         # wenn einzufügender Satz neuer als first_seen ist
        if fs[1] is not None: # wenn last_seen bereits gesetzt ist
          if fs[1]<fseen:     # wenn einzufügender Satz neuer als last_seen ist
            update=True       # UPDATE, weil last_seen älter als "fseen" ist
        else:
          update=True         # UPDATE, weil last_seen noch unbelegt ist
      if update:
        encID=self.__getSetEncryption(enc)
        self.cursor.execute('UPDATE network SET name=?, channel=?, last_seen=?, enc=? WHERE bssid=?', (essid, chan, fseen, encID, bssid))
    else:
      encID=self.__getSetEncryption(enc)
      self.cursor.execute('INSERT INTO network (bssid, name, channel, first_seen, enc) VALUES (?, ?, ?, ?, ?)', (bssid, essid, chan, fseen, encID))

  # ######################################################################
  # Liefert die id zu "enc". Sofern "enc" noch nicht existiert, wird der
  # Satz angelegt und dann dessen id geliefert.
  def __getSetEncryption(self, enc):
    self.cursor.execute('SELECT id FROM encryptions WHERE crypto=?', (enc,))
    fs=self.cursor.fetchone()
    if fs is None:
      self.cursor.execute('INSERT INTO encryptions (crypto) VALUES (?)', (enc,))
      self.cursor.execute('SELECT id FROM encryptions WHERE crypto=?', (enc,))
      fs=self.cursor.fetchone()
    return(fs[0])

  # ######################################################################
  # Nimmt einen Messpunkt auf.
  # Die BSSID muss sich in der Tabelle network befinden!
  def insertGPSpoint(self, bssid, lat, lon, dbm, seen):
    self.cursor.execute('SELECT count(*) FROM network WHERE bssid=?', (bssid,))
    fs=self.cursor.fetchone()
    if fs[0]==0:  # BSSID existiert nicht in network
      return

    self.cursor.execute('SELECT last_seen FROM gpspoint WHERE lat=? AND lon=? AND bssid=?', (lat, lon, bssid))
    fs=self.cursor.fetchone()
    if fs is not None:  # wenn Koordinate zur BSSID schon in der DB enthalten ist
      if fs[0]<seen:    # wenn einzufügender Satz neuer ist
        self.cursor.execute('UPDATE gpspoint SET dbm=?, last_seen=? WHERE lat=? AND lon=? AND bssid=?', (dbm, seen, lat, lon, bssid))
    else:
      self.cursor.execute('INSERT INTO gpspoint (lat, lon, bssid, dbm, last_seen) VALUES (?, ?, ?, ?, ?)', (lat, lon, bssid, dbm, seen))

  # ######################################################################
  # Fügt eine AP-Koordinate ein.
  def insertAPloc(self, bssid, lat, lon, pts, dt):
    self.cursor.execute('SELECT count(*) FROM ap_loc WHERE bssid=?', (bssid,))
    fs=self.cursor.fetchone()
    if fs[0]>0:   # BSSID existiert bereits in ap_loc
      self.cursor.execute('UPDATE ap_loc SET lat=?, lon=?, calc_points=?, last_calc=? WHERE bssid=?', (lat, lon, pts, dt, bssid))
    else:
      self.cursor.execute('INSERT INTO ap_loc (lat, lon, bssid, calc_points, last_calc) VALUES (?, ?, ?, ?, ?)', (lat, lon, bssid, pts, dt))

  # ######################################################################
  # Liefert zur BSSID den Timestamp des neusten Messpunktes.
  def newestPointTimestamp(self, bssid):
    self.cursor.execute('SELECT max(last_seen) FROM (SELECT last_seen FROM gpspoint WHERE bssid=? GROUP BY last_seen)', (bssid,))
    fs=self.cursor.fetchone()
    if fs is not None:
      return(fs[0])
    return(None)

  # ######################################################################
  # Liefert True, wenn die Koodinaten der BSSID vor "yp" berechnet wurden.
  def needsCalculation(self, bssid, yp):
    #yp=self.newestPointTimestamp(bssid)
    ap=self.getAPloc(bssid)   # lat, lon, bssid, calc_points, last_calc
    if ap is not None:  # BSSID befindet sich bereits in ap_loc
      if ap[4]<yp:      # ap_loc-Satz ist älter als neuster gpspoint-Satz
        return(True)
      else:             # ap_loc-Satz ist gleich oder neuer als neuster gpspoint-Satz
        return(False)
    else:               # BSSID noch nicht in ap_loc
      return(True)

  # ######################################################################
  # Liefert alle Netzwerke als Liste.
  # (bssid, name, crypto, channel, first_seen, last_seen)
  def getAllBSSIDs(self):
    self.cursor.execute('SELECT bssid, name, crypto, channel, first_seen, last_seen FROM network_enc')
    fs=self.cursor.fetchall()
    return(fs)

  # ######################################################################
  # Liefert alle Messpunkte zu einer BSSID als Liste.
  # (lat, lon, dbm)
  def getGPSpoints(self, bssid):
    self.cursor.execute('SELECT lat, lon, dbm FROM gpspoint WHERE bssid=?', (bssid,))
    fs=self.cursor.fetchall()
    return(fs)

  # ######################################################################
  # Liefert die Koordinaten zu einer BSSID.
  # (lat, lon, bssid, calc_points, last_calc)
  def getAPloc(self, bssid):
    self.cursor.execute('SELECT lat, lon, bssid, calc_points, last_calc FROM ap_loc WHERE bssid=?', (bssid,))
    fs=self.cursor.fetchone()
    return(fs)

  # ######################################################################
  # Liefert alle APs mit ihren Koordinaten.
  # (lat, lon, bssid, name, channel, crypto, calc_points, first_seen, last_seen)
  def getAllAPs(self):
    self.cursor.execute('SELECT lat, lon, bssid, name, channel, crypto, calc_points, first_seen, last_seen FROM AccessPoints')
    fs=self.cursor.fetchall()
    return(fs)
