#!/usr/bin/env python
# -*- coding: UTF-8 -*-

# Ein Python-Script zum Schreiben von gpx-Tracks.
# Detlev Ahlgrimm 06./07.2016
#
# v1.0    26.06.2016  erste Version
# v1.1    29.06.2016  GPS_wait_and_set_systemtime() zugefügt und einige strukturelle Umbauten
# v1.2    03.07.2016  Ermittlung der Geschwindigkeit und locking auf gps_g zugefügt
# v1.2a   04.07.2016  bei Script-Start warten, bis GPS-Modul verfügbar
# v1.3    12.07.2016  start von kismet_server nach setzen der Zeit + Verarbeitung von "STOP KISMET"
# v1.4    30.07.2016  Übermittlung der aktuell gesehenen WLANs an Status-Seite

import os
import sys
import time
import threading
import gps
import SocketServer
import socket
import json
import subprocess
import datetime
from dateutil import tz
from math import radians, cos, sin, asin, sqrt


TRACK_POINT_INTERVAL=5    # Frequenz, mit der Track-Punkte geschrieben werden [in Sekunden]

HOME_DIR=os.path.expanduser('~')
MOUNTPOINT_USB_STICK=os.path.join(HOME_DIR, "usbstick")       # hier ist der USB-Stick eingehängt
LOG_FILE_DIR=os.path.join(MOUNTPOINT_USB_STICK, "recordings") # hier landen die gpx-Tracks
LOG_FILE_DIR_BACKUP=HOME_DIR  # hier landen die gpx-Tracks, wenn der USB-Stick nicht gesteckt ist


run_tracker_g=True        # beendet bei False die Hauptschleife und somit das Script
trackpoints_written_g=0   # die Anzahl der bisher geschriebenen Track-Punkte
tracker_is_running_g=0    # auf 1, wenn TrackWriter() läuft und fehlerlos schreibt

async_get_wlans_g=None
counter_WLAN_g=dict()

# ######################################################################
# Eine Datenstruktur für einen Punkt mit Zeit, Geschwindigkeit und
# Anzahl genutzter Satelliten.
class GPS_position():
  def __init__(self):
    self.lat=0      # gps.fix.latitude
    self.lon=0      # gps.fix.longitude
    self.utc=""     # gps.utc
    self.lti=None   # gps.utc in Ortszeit als datetime-Objekt
    self.spd=0      # die Geschwindigkeit in km/h zwischen der letzten und aktuellen Position
    self.sat=0      # gps.satellites mit used==True

gps_g=GPS_position()      # nimmt die Daten der gps-Klasse auf und wird per Thread aktualisiert
gps_pos_g=GPS_position()  # die letzte valide Position


# ######################################################################
# Quelle: https://pypi.python.org/pypi/haversine
AVG_EARTH_RADIUS = 6371  # in km
def haversine(point1, point2, miles=False):
    """ Calculate the great-circle distance bewteen two points on the Earth surface.

    :input: two 2-tuples, containing the latitude and longitude of each point
    in decimal degrees.

    Example: haversine((45.7597, 4.8422), (48.8567, 2.3508))

    :output: Returns the distance bewteen the two points.
    The default unit is kilometers. Miles can be returned
    if the ``miles`` parameter is set to True.

    """
    # unpack latitude/longitude
    lat1, lng1 = point1
    lat2, lng2 = point2

    # convert all latitudes/longitudes from decimal degrees to radians
    lat1, lng1, lat2, lng2 = map(radians, (lat1, lng1, lat2, lng2))

    # calculate haversine
    lat = lat2 - lat1
    lng = lng2 - lng1
    d = sin(lat * 0.5) ** 2 + cos(lat1) * cos(lat2) * sin(lng * 0.5) ** 2
    h = 2 * AVG_EARTH_RADIUS * asin(sqrt(d))
    if miles:
        return h * 0.621371  # in miles
    else:
        return h  # in kilometers

# ######################################################################
# Liefert für die beiden Koordinate/Zeit-Tupel in "p1" und "p2" die
# zwischen diesen Punkten gefahrene Geschwindigkeit in km/h.
def speed(p1, p2):
  km=haversine((p1[0], p1[1]), (p2[0], p2[1]))
  if p2[2]>p1[2]: td=p2[2]-p1[2]
  else:           td=p1[2]-p2[2]
  if td.seconds<=0: # Division durch Null abfangen
    return(0.0)
  return(km/(td.seconds/3600.0))


# ######################################################################
# Liefert für einen Timestamp im Format "2016-06-25T10:40:31.000Z"
# ein entsprechendes datetime-Objekt in Ortszeit.
def utcToLocal(timestamp):
  utc=datetime.datetime.strptime(timestamp, '%Y-%m-%dT%H:%M:%S.%fZ')
  utc=utc.replace(tzinfo=tz.tzutc())
  return(utc.astimezone(tz.tzlocal()))


# ######################################################################
# Schreibt "strg" ins syslog.
def writeLog(strg):
  subprocess.call(["logger", "GPS_tracker: %s"%(strg,)])


# ######################################################################
# Führt die Kommandos gemäß der Liste "cmds" aus und liefert dessen
# Rückgabe als String.
def execute(cmds):
  process=subprocess.Popen(cmds, stdin=subprocess.PIPE,
                  stdout=subprocess.PIPE, stderr=open(os.devnull, 'w'))
  rc=process.communicate()[0]
  return(rc)


# ######################################################################
# Schreibt das gpx-File im Verzeichnis "log_file_dir".
class TrackWriter():
  def __init__(self, log_file_dir):
    global tracker_is_running_g
    writeLog("TrackWriter gestartet")
    self.logfile=os.path.join(log_file_dir, time.strftime("%Y_%m_%d_%H_%M_%S")+".gpx")
    strg= '<?xml version="1.0" encoding="UTF-8" standalone="yes" ?>\n' \
          '<gpx version="1.1" creator="RasPi Tracker" xmlns="http://www.topografix.com/GPX/1/1"' \
          ' xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"' \
          ' xsi:schemaLocation="http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd">\n' \
          '  <trk>\n' \
          '    <trkseg>\n'
    self.failed=False
    try:
      fobj=open(self.logfile, "a")
      fobj.write(strg)    
      fobj.close()
      tracker_is_running_g=1
    except:
      self.failed=True
      tracker_is_running_g=0
    self.strg='      <trkpt lat="%11.9f" lon="%11.9f"><time>%s</time>' \
              '<extensions><speedkmh>%.2f</speedkmh>' \
              '<sat>%d</sat></extensions></trkpt>\n'

  # ######################################################################
  # Schreibt eine Position samt Zeit und Geschwindigkkeit in das gpx-File.
  def write(self):
    global tracker_is_running_g
    if not self.failed:
      try:
        fobj=open(self.logfile, "a")
        fobj.write(self.strg%(gps_pos_g.lat, gps_pos_g.lon, gps_pos_g.utc, gps_pos_g.spd, gps_pos_g.sat))
        fobj.close()
        tracker_is_running_g=1
        return(True)
      except:
        tracker_is_running_g=0
    return(False)

  # ######################################################################
  # Schließt das gpx-File.
  def close(self):
    if not self.failed:
      try:
        fobj=open(self.logfile, "a")
        fobj.write('    </trkseg>\n  </trk>\n</gpx>')
        fobj.close()
        writeLog("TrackWriter beendet")
      except:
        pass


# ######################################################################
# Liefert der WebSeite GPS-Daten und reagiert auf Kommandos von dort.
class MyTCPHandler(SocketServer.BaseRequestHandler):
  def handle(self):
    global run_tracker_g
    global counter_WLAN_g
    global async_get_wlans_g
    inp=self.request.recv(1024)
    if inp.decode()=='GET DATA':
      snd="%11.9f;%11.9f;%d;%d;%d;%.2f"%(
            gps_pos_g.lat,
            gps_pos_g.lon,
            gps_pos_g.sat,
            trackpoints_written_g,
            tracker_is_running_g,
            gps_pos_g.spd   )
      self.request.sendall(snd)
    elif inp.decode()=='GET WLANS':
      snd=list()
      if async_get_wlans_g is not None:
        wlans=async_get_wlans_g.get()
        for bssid, (nam, dbms) in wlans.items():
          if bssid in counter_WLAN_g:
            counter_WLAN_g[bssid]+=len(dbms)
          else:
            counter_WLAN_g.update({bssid:len(dbms)})
          snd.append((counter_WLAN_g[bssid], nam, dbms))
        if len(snd)==0:
          snd=[(0, "keine AccessPoints", "")]
      self.request.sendall(json.dumps(snd))
    elif inp.decode()=='STOP TRACKER':
      self.request.sendall("OK")
      run_tracker_g=False
    elif inp.decode()=='STOP KISMET':
      self.request.sendall("OK")
      async_get_wlans_g.die()
      async_get_wlans_g=None
      os.system("pkill kismet_server")
      writeLog("kismet_server gestoppt")
class SocketSrv(threading.Thread):
  def __init__(self):
    threading.Thread.__init__(self)
    self.server=SocketServer.TCPServer(("127.0.0.1", 42424), MyTCPHandler)

  # ######################################################################
  # Startet den Thread und damit den SocketServer.
  def run(self):
    writeLog("SocketServer gestartet")
    self.server.serve_forever()
    writeLog("SocketServer beendet")

  # ######################################################################
  # Stoppt den SocketServer und damit den Thread.
  def die(self):
    self.server.shutdown()


 
# ######################################################################
# Eine Klasse, die ständig die aktuellen Daten des GPS-Moduls in der
# globalen Variable "gps_g" ablegt bzw. "gps_g" aktuell hält.
class GpsLatest(threading.Thread):
  def __init__(self):
    threading.Thread.__init__(self)
    global gps_g
    self.gps=gps.gps(mode=gps.WATCH_ENABLE)
    self.is_running=True
    self.died=False
    self.lock=threading.Lock()

  # ######################################################################
  # Aktualisiert endlos "gps_g" gemäß GPS.
  # Der Schreibzugriff auf "gps_g" erfolgt nur bei freiem "self.lock".
  def run(self):
    global gps_g
    while self.is_running:
      self.gps.next()
      with self.lock:
        gps_g.lat=self.gps.fix.latitude
        gps_g.lon=self.gps.fix.longitude 
        gps_g.utc=self.gps.utc
        gps_g.sat=self.__numberOfUsedSatellites()
    self.died=True

  # ######################################################################
  # Beendet die endlose Aktualisierung von "gps_g".
  def die(self):
    self.is_running=False
    while not self.died:
      time.sleep(0.01)

  # ######################################################################
  # Sperren des Schreibzugriffs aus "gps_g".
  def acquire_lock(self):
    self.lock.acquire()

  # ######################################################################
  # Freigeben des Schreibzugriffs aus "gps_g".
  def release_lock(self):
    self.lock.release()

  # ######################################################################
  # Liefert die Anzahl der Satelliten, bei denen das Attribut "used"
  # auf True steht.
  def __numberOfUsedSatellites(self):
    cnt=0
    for c in self.gps.satellites:
      if c.used:
        cnt+=1
    return(cnt)


# ######################################################################
# Wartet, bis vom GPS Datum und Uhrzeit empfangen wurde und stellt dann
# die Systemzeit entsprechend. Liefert True, wenn die aktuelle Zeit
# eingestellt wurde - False, wenn ein Fehler auftrat.
def GPS_wait_and_set_systemtime():
  writeLog("warte auf Datum+Zeit vom GPS")
  time_set=False
  wcnt=0
  while not time_set and run_tracker_g:
    try:
      if gps_g.utc is not None and len(gps_g.utc)==24:
        dtl=utcToLocal(gps_g.utc)
        execute(["sudo", "date", "-s", dtl.strftime("%Y-%m-%d %H:%M:%S")])
        time_set=True
        writeLog("Zeit gesetzt")
      else:
        wcnt+=1
        if wcnt%10==0:
          writeLog("%d mal gewartet"%(wcnt,)) # alle 30 Sekunden melden
        time.sleep(3)
    except Exception, e:
      writeLog("Ausnahme aufgetreten %s"%(str(e),))
      return(False)
  return(True)



# ######################################################################
# Eine Klasse, um mit dem kismet_server zu kommunizieren.
class KismetSrv():
  def __init__(self, address=("127.0.0.1", 2501)):
    self.data=socket.create_connection(address).makefile("w", 1)
    self.cmdnr=0
    self.setup()

  # ######################################################################
  # Dem kismet_server sagen, welche Infos er senden soll.
  def setup(self):
    self.sendCmd("ENABLE BSSID bssid,signal_dbm")
    self.__getData()
    self.sendCmd("ENABLE SSID mac,ssid")

  # ######################################################################
  # Das Kommando "cmd" mit vorangestellter fortlaufender Nummer an den
  # kismet_server senden.
  def sendCmd(self, cmd):
    snd="!%d %s\n"%(self.cmdnr, cmd)
    self.data.write(snd)
    self.cmdnr+=1

  # ######################################################################
  # Verfügbare Zeilen vom kismet_server lesen und als Liste zurückliefern.
  def __getData(self):
    lns=self.data.readline().rstrip("\n").split("\x01")
    return(lns)

  # ######################################################################
  # Liefert die letzten beim kismet_server verfügbaren Zeilen, deren
  # erste Zeile mit "prefix" beginnt. Andere Zeilen werden überlesen.
  def __getDataWithPrefix(self, prefix):
    lns=["-"]
    while not lns[0].startswith(prefix):
      lns=self.__getData()
    return(lns)

  # ######################################################################
  # Liefert die letzt-verfügbare BSSID + AccessPoint-Namen mit Dämpfung
  # vom kismet_server oder "", wenn ein Fehler aufgetreten ist.
  def getAccessPointInfo(self):
    ln1=self.__getDataWithPrefix("*BSSID:") # ['*BSSID: B8:27:EB:65:BA:64 -17 ']
    ln2=self.__getDataWithPrefix("*SSID:")  # ['*SSID: B8:27:EB:65:BA:64 ', 'RaspAP', ' ']
    try:
      if ln1[0].startswith("*BSSID:") and ln2[0].startswith("*SSID:"):
        bssid=ln1[0].split()  # ['*BSSID:', 'B8:27:EB:65:BA:64', '-17']
        ssid=ln2[0].split()   # ['*SSID:', 'B8:27:EB:65:BA:64']
        if ln2[1]!="" and bssid[1]==ssid[1]:
          return(bssid[1], ln2[1], bssid[2])  # ('B8:27:EB:65:BA:64', 'RaspAP', '-17')
    except:
      pass
    return("")


WLAN_COLLECT_MAX=20
DBMS_COLLECT_MAX=9
# ######################################################################
# Eine Klasse zum Sammeln von AccessPoint-Namen mit jew. Dämpfung.
# Es wird so lange gesammelt, bis die Daten abgefragt werden.
# Durch die Abfrage werden die bisher gesammelten Daten gelöscht.
class CurrentWLANs(threading.Thread):
  def __init__(self):
    threading.Thread.__init__(self)
    self.wlans=dict()
    self.ks=KismetSrv()
    self.is_running=False
    self.died=False
    self.lock=threading.Lock()

  # ######################################################################
  # Sammelt asynchron AccessPoint-Namen und dbm-Werte in einem Dictionary.
  # Key ist die BSSID, Value ist ein Tupel aus AccessPoint-Name und einer
  # Liste von Integers mit dbm-Werten.
  def run(self):
    writeLog("CurrentWLANs gestartet")
    self.is_running=True
    while self.is_running:
      if len(self.wlans)>WLAN_COLLECT_MAX:
        self.wlans=dict() # wenn es zu viele werden -> alle killen
      data=self.ks.getAccessPointInfo()
      if len(data)>2:
        bssid, nam, dbm=data
        with self.lock:
          if bssid in self.wlans:
            nams, dbms=self.wlans[bssid]
            dbms.append(int(dbm))
            if len(dbms)>DBMS_COLLECT_MAX:
              dbms.pop(0) # ältesten wegschmeissen
            self.wlans.update({bssid:(nam, dbms)})
          else:
            self.wlans.update({bssid:(nam, [int(dbm)])})
    self.died=True
    writeLog("CurrentWLANs beendet")

  # ######################################################################
  # Stoppt die Lese-Schleife und damit den Thread.
  def die(self):
    self.is_running=False
    while not self.died:
      time.sleep(0.01)

  # ######################################################################
  # Liefert ein Dictionary mit den gesammelten AccessPoints und leert
  # das lokale Sammel-Dictionary.
  def get(self):
    with self.lock:
      d=self.wlans.copy()
      self.wlans=dict()
    return(d)



# ######################################################################
#
if __name__ == '__main__':
  socksrv=SocketSrv()   # Kommunikation mit der WebSeite starten
  socksrv.start()

  while not os.path.exists("/dev/gps0"):  # GPS-Modul gesteckt?
    if not run_tracker_g:   # StopTracker über WebSeite angefordert?
      socksrv.die()
      writeLog("Script beendet!")
      sys.exit()
    time.sleep(1) # ohne GPS-Modul kann man ... nur darauf warten

  gpsl=GpsLatest()    # GPS-Modul Auto-Aktualisierung starten
  gpsl.start()

  if not GPS_wait_and_set_systemtime(): # Systemzeit gem. GPS setzen
    writeLog("keine Zeit empfangen -> reboot")
    subprocess.call(["sudo", "init", "6"])

  rc=execute(["df", "-hT"])
  if rc.find(MOUNTPOINT_USB_STICK)>=0:  # ist der USB-Stick eingehängt?
    track=TrackWriter(LOG_FILE_DIR)
  else:
    track=TrackWriter(LOG_FILE_DIR_BACKUP)

  os.system("/usr/local/bin/kismet_server --daemonize")
  writeLog("kismet_server gestartet")
  time.sleep(1) # dem kismet_server etwas Zeit geben, um Verbindungen anzunehmen
  async_get_wlans_g=CurrentWLANs()
  async_get_wlans_g.start()

  previous_pos_and_time=None  # letzte Position+Zeit - zur Geschwindigkeits-Ermittlung
  try:
    while run_tracker_g:  # solange auf der WebSeite nicht "Stop Tracker" gewählt wurde
      gps_pos_g=GPS_position()  # Werte zurücksetzen
      if gps_g.lat>0 and gps_g.lon>0 and gps_g.utc!="": # GPS-Daten valide?
        # Daten rauskopieren, damit der Thread nicht mittendrin einzelne Attribute ändert und
        # lat/lon ggf. nicht mehr zu utc passt - was zu einer falschen Geschwindigkeit führen würde.
        try:
          gpsl.acquire_lock()       # erstmal Schreibzugriff auf "gps_g" sperren
          gps_pos_g.lat=gps_g.lat
          gps_pos_g.lon=gps_g.lon
          gps_pos_g.utc=gps_g.utc
          gps_pos_g.sat=gps_g.sat
          gpsl.release_lock()       # Schreibzugriff auf "gps_g" freigeben
          gps_pos_g.lti=utcToLocal(gps_pos_g.utc)
        except:
          gpsl.release_lock()
        if previous_pos_and_time is not None: # ab dem zweiten Track-Punkt...Geschwindigkeit ermitteln
          try:
            gps_pos_g.spd=speed(previous_pos_and_time, (gps_pos_g.lat, gps_pos_g.lon, gps_pos_g.lti))
          except:
            gps_pos_g=GPS_position()
        if track.write(): # bei Fehler wird tracker_is_running_g=0 von track.write() gesetzt
          trackpoints_written_g+=1
        previous_pos_and_time=(gps_pos_g.lat, gps_pos_g.lon, gps_pos_g.lti)
      time.sleep(TRACK_POINT_INTERVAL)
  except (KeyboardInterrupt, SystemExit):
    pass

  if async_get_wlans_g is not None:
    async_get_wlans_g.die()
  os.system("pkill kismet_server")
  writeLog("kismet_server gestoppt")
  track.close()
  socksrv.die()
  gpsl.die()
  writeLog("Script beendet!")
