#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ###########################################################
# Ein Programm zur Kommunikation mit den Logger-Schaltungen.
#
# Es wird ein Socket geöffnet und auf Anfragen von Logger'n
# gewartet.
# Die übermittelten Sensor-Daten werden in einer Datenbank
# gespeichert.
#
# Detlev Ahlgrimm, 2017
#
# 30.03.2017  Umstellung auf UTC in insertDump()

import sys
import os
import socket
import select
import time
import datetime
import sqlite3

FILENAME_LOGFILE    ="/home/dede/logger_log.txt"
#DATABASE_FILENAME   ="/home/dede/daten/logger.sqlite"
DATABASE_FILENAME   ="/nfs/logger.sqlite"



class Database():
  def __init__(self, dbname):
    if os.path.isfile(dbname)==False:
      initDB=True
    else:
      initDB=False

    self.connection=sqlite3.connect(dbname)
    self.cursor=self.connection.cursor()

    if initDB==True:
      self.createDatabase()


  # ###########################################################
  # Legt die Tabellen an.
  def createDatabase(self):
    self.cursor.execute("CREATE TABLE  Host"                           \
                        " (id          INTEGER NOT NULL PRIMARY KEY,"  \
                        "  hostname    VARCHAR NOT NULL,"              \
                        "  ip_addr     VARCHAR NOT NULL,"              \
                        "  firmware    VARCHAR NOT NULL,"              \
                        "  sensor_cnt  INTEGER NOT NULL,"              \
                        "  last_reboot VARCHAR,"                       \
                        "  comment     VARCHAR)")
    self.cursor.execute("CREATE UNIQUE INDEX nodupe1 ON Host (hostname)")

    self.cursor.execute("CREATE TABLE  Location"                       \
                        " (id          INTEGER NOT NULL PRIMARY KEY,"  \
                        "  host        INTEGER NOT NULL,"              \
                        "  sensor      INTEGER NOT NULL,"              \
                        "  location    VARCHAR,"                       \
                        "  valid_from  VARCHAR,"                       \
                        "  valid_till  VARCHAR,"                       \
                        "  FOREIGN KEY(host) REFERENCES Host(id))")

    self.cursor.execute("CREATE TABLE  Dump"                           \
                        " (id          INTEGER NOT NULL PRIMARY KEY,"  \
                        "  location    INTEGER NOT NULL,"              \
                        "  timestamp   INTEGER,"                       \
                        "  temperature REAL,"                          \
                        "  light       INTEGER,"                       \
                        "  FOREIGN KEY(location) REFERENCES Location(id))")
    self.cursor.execute("CREATE UNIQUE INDEX nodupe2 ON Dump (location, timestamp)")
    self.cursor.execute("CREATE INDEX speed1 ON Dump (timestamp)")


    self.cursor.execute("CREATE VIEW HostLoc AS" \
                        " SELECT l.id AS locid, h.hostname, l.sensor, l.location" \
                        " FROM Host as h, Location as l WHERE h.id=l.host")

    self.cursor.execute("CREATE VIEW DumpD AS" \
                        " SELECT id, location, datetime(timestamp, 'unixepoch', 'localtime') AS timestamp, temperature, light" \
                        " FROM Dump" \
                        " ORDER BY timestamp, location")

    self.cursor.execute("CREATE VIEW Hours AS" \
                        " SELECT hl.hostname, hl.sensor, hl.location, substr(d.timestamp, 1, 13) || ':00' AS timestamp, count(*) AS minutes" \
                        " FROM DumpD AS d, HostLoc AS hl" \
                        " WHERE hl.locid=d.location" \
                        " GROUP BY hl.hostname, hl.sensor, hl.location, substr(d.timestamp, 1, 13)")

    self.connection.commit()


  # ######################################################################
  # Änderungen festschreiben.
  def commit(self):
    self.connection.commit()


  # ######################################################################
  # Nimmt einen Host auf oder aktualisiert ihn, wenn er schon in der DB ist.
  # Liefert True, wenn ein vorhandener Satz aktualisiert wurde.
  # Liefert False, wenn ein neuer Satz angelegt wurde.
  def insertHost(self, hostname, ip_addr, firmware, sensor_cnt, last_reboot, comment=""):
    self.cursor.execute('SELECT id FROM Host WHERE hostname=?', (hostname,))
    fs=self.cursor.fetchone()
    if fs is not None:  # wenn host schon in der DB enthalten ist
      update=True
      self.cursor.execute('UPDATE Host SET ip_addr=?, firmware=?, sensor_cnt=?, last_reboot=?, comment=? WHERE hostname=?', (ip_addr, firmware, sensor_cnt, last_reboot, comment, hostname))
    else:
      update=False
      self.cursor.execute('INSERT INTO Host (hostname, ip_addr, firmware, sensor_cnt, last_reboot, comment) VALUES (?, ?, ?, ?, ?, ?)', (hostname, ip_addr, firmware, sensor_cnt, last_reboot, comment))
    return(update)


  # ######################################################################
  # Liefert das Attribut Host.sensor_cnt von "hostname" als Integer oder
  # -1, wenn der Hostname nicht gefunden wurde.
  def getHost_sensor_cnt(self, hostname):
    self.cursor.execute('SELECT sensor_cnt FROM Host WHERE hostname=?', (hostname,))
    fs=self.cursor.fetchone()
    if fs is not None:
      return(int(fs[0]))
    return(-1)


  # ######################################################################
  # Fügt in die Tabelle Location einen Sensor ein. Der "hostname" muss
  # bereits in der Tabelle Host existieren.
  # Bei force_new==False wird der Sensor nur dann eingefügt, wenn er nicht
  # schon vorhanden ist.
  # Bei force_new==True wird auch dann ein neuer Satz eingefügt, wenn der
  # Sensor bereits vorhanden ist.
  #   0 für "Satz eingefügt"
  #   1 für "Satz war schon vorhanden"
  #   2 für "Host existiert nicht"
  def insertLocation(self, hostname, sensor, force_new=False):
    self.cursor.execute('SELECT id FROM Host WHERE hostname=?', (hostname,))
    host_id=self.cursor.fetchone()
    if host_id is None:
      return(2)
    host_id=host_id[0]
    if force_new:
      n=datetime.datetime.now()
      self.cursor.execute('INSERT INTO Location (host, sensor, valid_from) VALUES (?, ?, ?)', 
                          (host_id, sensor, n.strftime("%Y.%m.%d %H:%M")))
      return(0)
    else:
      self.cursor.execute('SELECT id FROM Location WHERE host=? AND sensor=?', (host_id, sensor))
      fs=self.cursor.fetchone()
      if fs is None:
        self.cursor.execute('INSERT INTO Location (host, sensor) VALUES (?, ?)', (host_id, sensor))
        return(0)
    return(1)


  # ###########################################################
  # Fügt die übergebenen Daten in die Tabelle Dump ein.
  # Liefert
  #   0 für "Satz eingefügt"
  #   1 für "Satz war schon vorhanden"
  #   2 für "Host existiert nicht"
  def insertDump(self, hostname, sensor, year, month, day, hour, minute, temp, light):
    self.cursor.execute('SELECT max(locid) FROM HostLoc WHERE hostname=? AND sensor=?', (hostname, sensor))
    locid=self.cursor.fetchone()
    if locid is None:
      return(2)
    locid=locid[0]
    try:
      d=int((datetime.datetime(year, month, day, hour, minute)-datetime.datetime(1970, 1, 1)).total_seconds())-3600
      self.cursor.execute('INSERT INTO Dump (location, timestamp, temperature, light) VALUES (?, ?, ?, ?)', (locid, d, temp, light))
      #d=datetime.datetime(year, month, day, hour, minute, 0)
      #print int(d.strftime("%s")), d, hour, minute
      #self.cursor.execute('INSERT INTO Dump (location, timestamp, temperature, light) VALUES (?, ?, ?, ?)', (locid, int(d.strftime("%s")), temp, light))
    except:
      return(1)
    return(0)





# ###########################################################
# Der Socket-Server.
# Also Socket öffnen, lauschen, Verbindungen annehmen, 
# verarbeiten und Verbindungen wieder schließen.
class SocketServer:
  def __init__(self, port):
    self.port=port;
    self.srvsock=socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    self.srvsock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    self.srvsock.bind(("", port))
    self.srvsock.listen(5)
    self.descriptors=[self.srvsock]

    #self.hostnames=dict()
    self.hostinfo=dict()

    self.db=Database(DATABASE_FILENAME)

    self.log=Logging()


  # ###########################################################
  # Hauptschleife des SocketServers.
  def run(self):
    timeout=5  # 5 Sekunden ohne Kommunikation führt zu disconnect

    while 1:
      (sread, swrite, sexc)=select.select(self.descriptors, [], [], timeout)
      if sread==[]:   # wenn sich "timeout" Sekunden lang niemand gemeldet hat
        if len(self.descriptors)>1:         # und wenn Clients verbunden
          for sock in self.descriptors:     # über alle Clients
            if sock!=self.srvsock:          # ohne den Server selbst
              try:
                host, port=sock.getpeername() # identifizieren
                self.close_connection(sock, host) # rausschmeißen
                del self.hostinfo[sock]
                self.log.write_warning("timeout")
              except:
                pass
        continue                            # zurück zu select.select

      for sock in sread:
        if sock==self.srvsock:
          self.accept_new_connection()
        else:
          try:
            strg=sock.recv(2048)
          except:
            strg=""
            self.log.write("Verbindung wird geschlossen\n")

          try:
            host, port=sock.getpeername()
          except:
            host, port=("?", "?")

          if strg=="":
            try:
              self.log.write("disconnect")
              self.close_connection(sock, host)
              del self.hostinfo[sock]
            except:
              pass
          else:
            status, data=parseReceivedData(strg)
            if status==True:                          # wenn Checksum gepasst hat
              self.process_command(sock, host, data)  # Kommando verarbeiten
            else:
              self.sock_send_with_checksum(sock, "BAD\x00")


  # ###########################################################
  # Nimmt einen neuen Client an und in die Liste der
  # verbundenen Clients auf.
  def accept_new_connection(self):
    newsock, (host, port)=self.srvsock.accept()
    self.descriptors.append(newsock)
    #if host not in self.hostnames:
    #  self.hostnames.update({host:self.get_hostname(host)})
    #print "Hostname:", self.hostnames[host], host


  # ###########################################################
  # Sendet "strg" samt Checksum an "sock".
  def sock_send_with_checksum(self, sock, strg):
    try:
      sock.send(buildDataWithBinChecksum(strg))
    except:
      pass
      self.log.write("Fehler bei sock_send_with_checksum()")


  # ###########################################################
  # Schließt die Verbindung zum Client gemäß "sock".
  def close_connection(self, sock, host=""):
    try:
      sock.close()
      self.descriptors.remove(sock)
    except:
      pass


  # ###########################################################
  # Liefert den Hostname zur IP-Adresse "ipaddr".
  def get_hostname(self, ipaddr):
    try:
      fqdn=socket.gethostbyaddr(ipaddr)
    except:
      return("")
    if len(fqdn)>1:
      hn=fqdn[0].split('.')
      return(hn[0])
    return("")


  # ###########################################################
  # Verarbeitet die Anweisungen des Clients.
  def process_command(self, sock, host, cmd):
    if cmd.startswith("GET_TIME"):
      t=getNormTime(True)
      self.sock_send_with_checksum(sock, t)
      #self.sock_send_with_checksum(sock, "2017.01.01 18:55:41")
      self.log.write("GET_TIME (%s)"%(t,))

    elif cmd.startswith("IDENT="):
      tmp=cmd[6:].split(",")
      if len(tmp)==6:
        hostname, firmware, sensor_cnt, last_reboot, warn, volts=tmp
      else:
        self.sock_send_with_checksum(sock, "BAD\x00")
        return
      self.log.write("-"*60)
      self.log.write("hostname   = %s"%(hostname,))
      self.log.write("firmware   = %s"%(firmware,))
      self.log.write("last_reboot= %s"%(last_reboot,))
      self.log.write("warn       = %s"%(warn,))
      self.log.write("volts      = %s"%(volts,))
      self.log.write("-"*60)
      self.hostinfo.update({sock:{"hostname":hostname, "firmware":firmware, "sensor_cnt":int(sensor_cnt)}})
      last_sensor_cnt=self.db.getHost_sensor_cnt(hostname)
      self.db.insertHost(hostname, host, firmware, int(sensor_cnt), last_reboot)
      if last_sensor_cnt!=int(sensor_cnt) or (int(warn)&0x2)>0:
        # Sensor-Anzahl hat sich geändert oder Standort-Änderungs-Wunsch
        force_new=True
      else:
        force_new=False
      for s in range(int(sensor_cnt)):
        rc=self.db.insertLocation(hostname, s+1, force_new)
        if rc==2:
          self.log.write_warning("insertLocation(): der Host wurde nicht gefunden!")
          self.sock_send_with_checksum(sock, "BAD\x00")
          return
      self.db.commit()
      self.sock_send_with_checksum(sock, "OK\x00")

    elif cmd.startswith("SHOUR.1=") or cmd.startswith("SHOUR.2="):
      data=cmd[8:]    # "SHOUR.x=" abtrennen
      rc=isValidChecksumForHourBlock(data)
      if rc:
        #print self.hostinfo[sock]
        #hexdump(cmd, len(cmd))
        try:
          hostname=self.hostinfo[sock]["hostname"]
        except:
          # da ist wohl der "IDENT"-Block nicht sauber angekommen...
          self.sock_send_with_checksum(sock, "BAD\x00")
          return
        year, month, day, hour, minute, second=convertTimestamp(data[:4])
        lst=getSensorDataFromHourBlock(data)
        #print lst
        good_cnt=bad_cnt=0    # Zähler für legale bzw. illegale Sensor-Daten
        ic=dc=0               # Zähler für eingefügte Sätze bzw. für Duplikate
        for minute2 in range(len(lst)):
          if minute2>=minute:
            # erst ab der Minute wegschreiben, ab der auch Daten da sind
            temp, light, bad=lst[minute2]
            if bad==0:
              good_cnt+=1
              rc=self.db.insertDump(hostname, int(cmd[6]), year, month, day, hour, minute2, temp, light)
              if rc==2:
                self.log.write_warning("insertDump(): der Satz in HostLoc wurde nicht gefunden!")
                break   # hostname nicht in DB
              elif rc==1:
                dc+=1
              elif rc==0:
                ic+=1
            else:
              bad_cnt+=1
        self.log.write("%s s=%d - %04d.%02d.%02d %02d:%02d - ins=%2d, dup=%2d (good=%2d, bad=%2d)"%(hostname, int(cmd[6]), year, month, day, hour, minute, ic, dc, good_cnt, bad_cnt))
        self.db.commit()
      else:
        self.log.write_warning("Block ist illegal. EEPROM defekt !?")
      # OK auch dann senden, wenn die interne Checksum nicht passt. Schließlich hat die
      # Übertragungs-Checksum ja gepasst, wenn er hier angekommen ist.
      self.sock_send_with_checksum(sock, "OK\x00")  # "OK" als C-String senden

    elif cmd.startswith("INFO="):
      v=cmd.split("=", 1)
      if len(v)>1:
        self.log.write("Info: %s"%(v[1]))
      self.sock_send_with_checksum(sock, "OK")

    else:
      self.sock_send_with_checksum(sock, "ECHO:"+cmd)
      self.log.write_warning("unbekannte Anforderung: %s"%(cmd,))






# ###########################################################
# Kapselung der Funktionen zum Logging.
class Logging():
  def __init__(self):
    pass


  # ###########################################################
  # Schreibt Ortszeit und "strg" mit vorangestelltem und farblich
  # hervorgehobenem Text " Warnung : " aufs Terminal und in die
  # Log-Datei.
  def write_warning(self, strg):
    with open(FILENAME_LOGFILE, 'a') as fl:
      fl.write("%s \x1b[1;32;41m Warnung \x1b[0m : %s\n"%(curTime(), strg))
    print "%s \x1b[1;32;41m Warnung \x1b[0m : %s"%(curTime(), strg,)

    
  # ###########################################################
  # Schreibt Ortszeit und "strg" aufs Terminal und in die
  # Log-Datei.
  def write(self, strg):
    with open(FILENAME_LOGFILE, 'a') as fl:
      fl.write(curTime() + " " + strg + "\n")
    print curTime(), strg




# ###########################################################
# Die drei Byte Sensor-Daten enthalten drei Werte. Der erste
# Wert belegt 12 Bit, der zweite 10 Bit und der dritte 1 Bit.
# Sind dies die Nibbles der drei Byte: ab cd ef
# bildet sich der Temeraturwert aus:   d ab  (&0xFFF)
# der Helligkeitswert aus:             ef c  (&0x3FF)
# und das BAD-Flag aus:                e     (&0x4)
# oder auch als:
#   temp=0x6bc   -> 0xbc 0x06 0x00
#   ldr =0x345   -> 0x00 0x50 0x34
#   bad =1       -> 0x00 0x00 0x40
def getSensorDataFromHourBlock(strg):
  lst=list()
  for m in range(60):
    p=4+3*m
    elem=strg[p:p+3]
    temp=ord(elem[0]) + ((ord(elem[1])&0x0F)<<8)
    if (temp&0x800)>0:  # Vorzeichen-Bit
      temp=-(0x7FF-(temp&0x7FF))
    ldr=((ord(elem[1])&0xF0)>>4) + ((ord(elem[2])&0x3F)<<4)
    bad=(ord(elem[2])&0x40)>>6
    #print hex(ord(elem[0])), hex(ord(elem[1])), hex(ord(elem[2])), (round(temp/16.0, 1), ldr, bad)
    lst.append((round(temp/16.0, 1), ldr, bad))
  return(lst)


# ###########################################################
# Die Nutzdaten eines ein-Stunden-Blocks sind 4+3*60=184=0xB8
# Byte lang. Zusammen mit der Prüfsumme von zwei Byte ist der
# ankommende Block 186=0xBA Byte lang. Die Prüfsumme wurde
# aus den 184 Byte Nutzdaten gebildet.
def isValidChecksumForHourBlock(strg):
  cs=0
  for c in strg[:-2]:
    cs+=ord(c)
    cs+=1
  return((cs&0xFFFF)==(ord(strg[-2:][0])+ord(strg[-2:][1])*256))


# ###########################################################
# Zerlegt einen vier Byte langen Timestamp in seine
# Komponenten.
# Eine long-Variable mit dem Inhalt 0x01234567 wird
# übertragen als: 67 45 23 01
# Somit sieht der Aufbau ziemlich durcheinander aus:
#    MMYYYYYY  hDDDDDMM  mmmmhhhh  ssssssmm
def convertTimestamp(strg):
  YY=  ord(strg[0])&0b00111111
  MM=((ord(strg[0])&0b11000000)>>6) + ((ord(strg[1])&0b00000011)<<2)
  DD= (ord(strg[1])&0b01111100)>>2
  hh=((ord(strg[1])&0b10000000)>>7) + ((ord(strg[2])&0b00001111)<<1)
  mm=((ord(strg[2])&0b11110000)>>4) + ((ord(strg[3])&0b00000011)<<4)
  ss= (ord(strg[3])&0b11111100)>>2
  return(YY+2000, MM, DD, hh, mm, ss)


# ###########################################################
# Liefert die Ortszeit als String.
# Wird "precise" mit True übergeben, wartet die Funktion vor
# der Rückgabe der Zeit bis zum Beginn einer neuen Sekunde.
def curTime(precise=False):
  if precise:
    n=datetime.datetime.now().second
    while datetime.datetime.now().second==n:
      time.sleep(0.00001)
    # jetzt hat die neue Sekunde gerade frisch angefangen
  return(datetime.datetime.now().strftime("%Y.%m.%d %H:%M:%S"))


# ###########################################################
# Liefert Datum+Uhrzeit in Winter- bzw. Normalzeit als String.
# Wird "precise" mit True übergeben, wartet die Funktion vor
# der Rückgabe der Zeit bis zum Beginn einer neuen Sekunde.
def getNormTime(precise=False):
  if precise==True:
    n=datetime.datetime.now().second
    while datetime.datetime.now().second==n:
      time.sleep(0.00001)
    # jetzt hat die neue Sekunde gerade frisch angefangen

  now=datetime.datetime.now()
  if time.localtime().tm_isdst==1:
    now=now+datetime.timedelta(hours=-1)
  return(now.strftime("%Y.%m.%d %H:%M:%S"))


# ###########################################################
# Trennt die Checksum von "strg" ab und liefert ein Tupel
# (True, strg), sofern die Checksum passte. Wenn nicht, wird
# (False, "") geliefert.
def parseReceivedData(strg):
  if len(strg)<3:
    return((False, ""))
  cs=ord(strg[-2:][0])+ord(strg[-2:][1])*256
  strg=strg[:-2]
  if getBinChecksum(strg)!=cs:
    return((False, ""))
  return((True, strg))


# ###########################################################
# Liefert "strg" mit angehängter Checksum.
def buildDataWithBinChecksum(strg):
  cst=getBinChecksum(strg)
  cs=chr(cst&0xFF) + chr((cst&0xFF00)/256)
  return(strg+cs)


# ###########################################################
# Liefert die Checksumme zu "strg" als 2-Byte Integer.
def getBinChecksum(strg):
  cs=0
  for c in strg:
    cs+=ord(c)
  return(cs&0xFFFF)


# ###########################################################
# Gibt einen Hexdump von "buf" in der Länge "lng" aus.
def hexdump(buf, lng):
  p=0
  while p<lng:
    sys.stdout.write("%04X  "%(p))
    for i in range(min(16, (lng-p))):
      sys.stdout.write("%02x "%ord(buf[p+i]))
    sys.stdout.write(" ")
    if (lng-p)<16:
      sys.stdout.write(" "*(3*(16-(lng-p))))
    for i in range(min(16, (lng-p))):
      c=ord(buf[p+i])
      if c>31 and c<128:
        sys.stdout.write(chr(c))
      else:
        sys.stdout.write('.')
    p+=16
    print


if __name__=='__main__':
  myServer=SocketServer(2627)
  myServer.run()
