#!/usr/bin/env python
# -*- coding: utf-8 -*-

import time
#import datetime
import threading
import numpy as np

from LoggerViewDataTypes  import Rect
from LoggerViewDB         import Database

# ###########################################################
#
class Data():
  def __init__(self, filename):
    self.filename=filename

    self.db=Database(filename)
    self.datacols=self.db.getColCount()               # Integer
    self.datarows=self.db.getRowCount()               # Integer
    self.coldesc=self.db.getColDescription()          # Liste
    self.minMaxTimestamp=self.db.getMinMaxTimestamp() # Tupel

    self.colorder=dict()  # definiert den Index der Spalten in self.arr
    for i in range(len(self.coldesc)):
      self.colorder.update({self.coldesc[i][0]:i+1})
    # Beispiel: {'2T': 3, '1H': 2, '2H': 4, '1T': 1}

    self.factor=self.db.getFactors()

    self.cache=dict()
    self.maxrows=1+(self.minMaxTimestamp[1]-self.minMaxTimestamp[0])/60
    self.baseIdx=self.minMaxTimestamp[0]
    self.arr=np.zeros(shape=(self.maxrows, self.datacols+1), dtype=np.int64)  # Array für die Daten
    self.flg=np.zeros(shape=(self.maxrows, self.datacols/2), dtype=bool)      # Array, ob Daten in arr valide sind
    # self.datacols/2 kennzeichnet die Anzahl der Sensor-Paare

    self.load()

    worker1=threading.Thread(target=self.loadStage2, name="load DB asynchronous")
    worker1.setDaemon(True)
    worker1.start()


  # ######################################################################
  # Lädt die Daten ins numpy-Array. Die jüngsten sieben Tage.
  def load(self):
    #minTimestamp=datetime.datetime.fromtimestamp(self.minMaxTimestamp[0])
    #maxTimestamp=datetime.datetime.fromtimestamp(self.minMaxTimestamp[1])
    #self.min_new=maxTimestamp-datetime.timedelta(days=7)
    self.min_new=self.minMaxTimestamp[1]-7*24*60*60

    #self.datadict=self.db.getData((int(self.min_new.strftime("%s")), self.minMaxTimestamp[1]))
    self.datadict=self.db.getData((self.min_new, self.minMaxTimestamp[1]))
    for rid in self.datadict.keys():      # über alle timestamps
      idx=(rid-self.baseIdx)/60           # timestamp nach Index wandeln
      self.arr[idx][0]=rid                # timestamp ablegen
      c=self.datadict[rid]                # Datenspalten holen
      for i in c:
        self.arr[idx][self.colorder[i[0]]]=i[1]   # Daten ablegen
        self.setValid(idx, self.colorder[i[0]])   # Daten als valide kennzeichnen


  # ######################################################################
  # Lädt die restlichen Daten asynchron nach.
  def loadStage2(self):
    self.dbt=Database(self.filename)  # es braucht ein eigenes DB-Objekt innerhalb eines Threads
    #datadict=self.dbt.getData((self.minMaxTimestamp[0], int(self.min_new.strftime("%s"))))
    datadict=self.dbt.getData((self.minMaxTimestamp[0], self.min_new))
    for rid in datadict.keys():           # über alle timestamps
      idx=(rid-self.baseIdx)/60           # timestamp nach Index wandeln
      self.arr[idx][0]=rid                # timestamp ablegen
      c=datadict[rid]                     # Datenspalten holen
      for i in c:
        self.arr[idx][self.colorder[i[0]]]=i[1]   # Daten ablegen
        self.setValid(idx, self.colorder[i[0]])   # Daten als valide kennzeichnen
    self.datadict.update(datadict)


  # ######################################################################
  # Liefert True, wenn zum Timestamp ts in der Spalte col Daten in
  # self.arr geladen wurden.
  def isIndexWithData(self, ts, col):
    row=(ts-self.baseIdx)/60
    if 0<=row<self.maxrows and 0<col<=self.datacols:
      return(self.flg[row][(col-1)/2])
    return(False)


  # ######################################################################
  # Liefert zu einem Timestamp ein Dictionary mit allen gültigen Werten
  # im Array. Key ist der Kurvenname, Value der Y-Wert.
  # z.B.:
  #  {'2T': -28, '3H': 0, '1T': 214, '1H': 0, '2H': 41, '3T': 200}
  def getValuesForTimestamp(self, ts):
    rd=dict()
    for cn, idx in self.colorder.items():
      if self.isIndexWithData(ts, idx):
        #rd.update({cn:float(self.arr[(ts-self.baseIdx)/60][idx])/self.factor[cn[-1]]})
        rd.update({cn:self.arr[(ts-self.baseIdx)/60][idx]})
    return(rd)


  # ######################################################################
  # Setzt self.flg an den Koordinaten row und col auf True und mappt
  # diese dabei korrekt auf die kleineren Dimensionen von self.flg 
  # bzgl. self.arr.
  def setValid(self, row, col):
    self.flg[row][(col-1)/2]=True


  # ######################################################################
  # Liefert ein Dictionary mit der Skala als key ("T" oder "H") und dem
  # Faktor, der darauf angewandt wurde, die den Messwert ganzzahlig zu
  # machen.
  def getFactors(self):
    return(self.factor)
    #return(self.db.getFactors())


  # ######################################################################
  # siehe DB-Klasse
  def getDataBoundaries(self):
    return(self.db.getDataBoundaries())
    # alte Version ... hat erst Daten liefern können, wenn das numpy-Array
    # komplett geladen war.
    dmin=self.arr.min(0)  # z.B.: [1483622580    199      0    -91      0]
    dmax=self.arr.max(0)  # z.B.: [1484287140    294    794     49    984]
    rd=dict()
    for n, i in self.colorder.items():
      rd.update({n:Rect(dmin[0], dmin[i], dmax[0], dmax[i])})
    return(rd)


  # ######################################################################
  # Liefert die Minimal- und Maximal-Werte aller Kurven als Dictionary,
  # aufgeteilt nach Seite der Skala.
  # Der key ist der Name der Kurve, value ist ein Rect().
  # Beispiel:
  #   { 'H': (x1=1483622580, y1=0,   x2=1484805540, y2=984, width=1182960, height=984),
  #     'T': (x1=1483622580, y1=-91, x2=1484805540, y2=294, width=1182960, height=385)  }
  def getFullDataBoundaries(self):
    b=self.getDataBoundaries()
    l_min=list()
    l_max=list()
    r_min=list()
    r_max=list()
    x_min=list()
    x_max=list()
    for k, v in b.items():
      if k.endswith("T"):   # linke Seite
        l_min.append(v.y1)
        l_max.append(v.y2)
        x_min.append(v.x1)
        x_max.append(v.x2)
      else:
        r_min.append(v.y1)
        r_max.append(v.y2)
        x_min.append(v.x1)
        x_max.append(v.x2)
    return({"T":Rect(min(x_min), min(l_min), max(x_max), max(l_max)),
            "H":Rect(min(x_min), min(r_min), max(x_max), max(r_max))})


  # ######################################################################
  # Wie __getLineList(), nur mit einem Cache für den letzten Satz
  # Kurven drüber.
  def getLineList(self, colnam, pix_rect, val_rect, win_h):
    #return(self.__getLineList(colnam, pix_rect, val_rect, win_h))
    if colnam in self.cache:
      opix_rect, oval_rect, owin_h, retlist=self.cache[colnam]
      if opix_rect==pix_rect and oval_rect==val_rect and owin_h==win_h:
        #print "from cache"
        return(retlist)
    retlist=self.__getLineList(colnam, pix_rect, val_rect, win_h)
    self.cache.update({colnam:(pix_rect, val_rect, win_h, retlist)})
    return(retlist)


  # ######################################################################
  # Liefert eine Liste von Listen für wx.dc.DrawLine()
  def __getLineList(self, colnam, pix_rect, val_rect, win_h):
    colidx=self.colorder[colnam]

    vw=float(val_rect.x2-val_rect.x1)/60  # Breite der Werte
    vh=val_rect.y2-val_rect.y1            # Höhe der Werte
    xs=float(pix_rect.width)/vw           # Skalierung für X
    ys=float(pix_rect.height)/vh          # Skalierung für Y

    retlist=list()
    linelist=list()
    xp=0
    if vw/pix_rect.width<=1: # wenn weniger als ein timestamp-Wert auf jedem Pixel liegt
      #print "direkt"
      for xl in xrange(val_rect.x1, val_rect.x2+60, 60):  # über die Zeit
        if self.isIndexWithData(xl, colidx):              # wenn zur Zeit Daten vorhanden sind
          # (Messwert[timestamp] - SkalaUnterkante) * SkalierungsFaktor
          yd=(self.arr[(xl-self.baseIdx)/60][colidx]-val_rect.y1)*ys
          linelist.append((pix_rect.x1+int(round(xp*xs)), win_h-int(round(pix_rect.y1+yd))))
        else:                         # wenn eine Lücke in den Daten ist...
          if len(linelist)>0:
            retlist.append(linelist)  # ...neue Liste anfangen
            linelist=list()
        xp+=1
      retlist.append(linelist)
    else:
      #print "unique ts"
      last_x=None
      # erstmal Listen bilden, in denen jeder timestamp-Wert nur einmal
      # vorkommt und den Mittelwert der entspr. Daten enthält
      auts=list()
      uts=list()
      for xl in xrange(val_rect.x1, val_rect.x2+60, 60):  # über die Zeit
        if self.isIndexWithData(xl, colidx):              # wenn zur Zeit Daten vorhanden sind
          x=pix_rect.x1+int(round(xp*xs))                 # X-Koordinate bestimmen
          if x==last_x:                                   # wenn selbe X-Koordinate wie zuvor
            y+=self.arr[(xl-self.baseIdx)/60][colidx]     # Mittelwert für Y-Koordinate bilden
            y/=2
          else:                                           # wenn neue X-Koordinate
            y=self.arr[(xl-self.baseIdx)/60][colidx]      # Y-Koordinate bestimmen
            if last_x is not None:                        # wenn schon Koordinaten berechnet wurden
              uts.append((last_x, y))                     # Koordinate in Liste ablegen
            last_x=x                                      # X-Koordinate für Dupe-Test merken
        else:                                             # wenn zur Zeit keine Daten vorhanden sind
          if last_x is not None:                          # wenn zuvor Koordinaten berechnet wurden
            uts.append((last_x, y))                       # Koordinate in Liste ablegen
          if len(uts)>0:                                  # wenn zuvor in eine Koordinaten-Liste geschrieben wurde
            auts.append(uts)                              # Koordinaten-Liste der Ergebnis-Liste als Sub-Liste zufügen 
            uts=list()                                    # neue/leere Koordinaten-Liste erzeugen
            last_x=None                                   # X-Koordinate für Dupe-Test als ungültig kennzeichen
        xp+=1                                             # nächstes Pixel auf der X-Achse ansteuern
      if last_x is not None:                              # wenn noch eine Koordinate ungespeichert ist
        uts.append((last_x, y))                           # auch diese Koordinate in Liste ablegen
      auts.append(uts)                                    # letzte Koordinaten-Liste der Ergebnis-Liste zufügen 
      xp=0
      for uts in auts:
        for x, y in uts:
          yd=(y-val_rect.y1)*ys
          linelist.append((x, win_h-int(round(pix_rect.y1+yd))))
          xp+=1
        retlist.append(linelist)
        linelist=list()
    return(retlist)
