#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ###########################################################
# Ein Programm zur Visualisierung der Daten von Logger.
#
# Detlev Ahlgrimm, 2017
#
# 05.02.2017   V1.2     Fix in Data.getFullDataBoundaries()
# 06.02.2017   V1.2.1   Zoomstufe in Statusbar, Zoom auf Grundstellung zugefügt,
#                       Menüpunkt "springe zu Datum" zugefügt, Tastatursteuerung zugefügt
# 11.02.2017   V1.3     Hervorheben der angeklickten Kurve zugefügt
# 20.02.2017   V1.3.1   Anzeige des Wochentages in der Skala und Parameter "-c" zugefügt
# 24.02.2017   V1.3.2   Wochen-Sprung mit Ctrl statt Shift, "vertikal zurücksetzen" zugefügt,
#                       An/Ausschalten aller Helligkeits-/Temperatur-Kurven zugefügt
# 25.02.2017   V1.3.3   copy der DB nach /ramdisk, wenn auf i5 ausgeführt - lokal auf 11w
# 28.02.2017   V1.3.4   Kurvenname in Statusbar wird nur bei hervorgehobener Kurve angezeigt
# 01.03.2017   V1.3.5   Anzeige der Temperatur-Nulllinie zugefügt
# 30.03.2017   V1.3.6   ein paar Timestamp-Berechnungen Ortszeit-unabhängig gemacht
#

import wx
import os
import sys
import time
import datetime
import subprocess

from LoggerViewDataTypes  import Rect
from LoggerViewData       import Data
from LoggerViewArea       import Area
from LoggerViewDialogs    import CurveSettingsFrame

DATABASE_FILENAME =os.path.join("/nfs/vms/ss", "logger.sqlite") # Pfad für 11w, wird in main() ggf. geändert
#DATABASE_FILENAME =os.path.join(os.path.expanduser('~'), "logger.sqlite") # temporär / für Tests
CONFIG_FILENAME   =os.path.join(os.path.expanduser('~'), ".logger.conf")
VERSION="1.3.6"

DAYS=["Mo", "Di", "Mi", "Do", "Fr", "Sa", "So"]

# ###########################################################
# Das Fenster.
class LoggerViewWindow(wx.Window):
  def __init__(self, parent):
    wx.Window.__init__(self, parent)
    self.parent=parent
    self.parent.SetMinSize(wx.Size(175, 110))

    self.data=Data(DATABASE_FILENAME)
    self.boundaries=self.data.getFullDataBoundaries()
    #self.boundaries.update({"H":Rect(self.boundaries["H"].x1, 0, self.boundaries["H"].x2, 1023)})
    self.cur_val_boundaries=self.boundaries.copy()
    self.factor=self.data.getFactors()

    # Das letzte "00:00 Uhr" aus den Daten bestimmen, nach denen noch
    # ein voller Tag (=24h) folgt.
    end_date=datetime.datetime.fromtimestamp(self.boundaries["T"].x2)
    if end_date.minute!=59 or end_date.hour!=23:
      end_date-=datetime.timedelta(hours=end_date.hour, minutes=end_date.minute)
    end_date-=datetime.timedelta(days=1)

    t1=int(end_date.strftime("%s"))
    #t2=int((end_date+datetime.timedelta(days=1)).strftime("%s"))
    t2=t1+24*60*60  # auch an einem Zeitumstellungstag 24 Stunden darstellen
    vH=self.cur_val_boundaries["H"]
    vT=self.cur_val_boundaries["T"]
    self.cur_val_boundaries.update({"H":Rect(t1, vH.y1, t2, vH.y2)})
    self.cur_val_boundaries.update({"T":Rect(t1, vT.y1, t2, vT.y2)})

    self.history=list()

    self.color_background="#000000"
    self.color_grid=      "#CCCCCC"
    self.color_text=      "#CCCCCC"

    self.color_background="#FFFFFF"
    self.color_grid=      "#000000"
    self.color_text=      "#000000"

    self.color_selection= "#FFFFFF" # invertiert
    #self.color_curve={"1T":"#FF3800", "1H":"#5CF509", "2T":"#FC12AE", "2H":"#2503F9" }

    # Settings für die Kurven einstellen
    self.curve_name=list()
    self.curve_name_dict=dict()
    fc=wx.FileConfig(localFilename=CONFIG_FILENAME)
    for cn, ct in self.data.coldesc:
      is_on=fc.ReadInt("curve_isOn_%s"%(cn,), 1)
      col=fc.Read("curve_col_%s"%(cn,), "#FFFFFF")
      self.curve_name.append([cn, ct, is_on, col])
      self.curve_name_dict.update({cn:ct})
    self.curveSettings_screenPos=(fc.ReadInt("cs_pos_x",  -1), fc.ReadInt("cs_pos_y",  -1))
    self.curveSettings_sizeTuple=(fc.ReadInt("cs_size_x", -1), fc.ReadInt("cs_size_y", -1))
    open_cs=fc.ReadInt("cs_is_open",  0)

    self.createMenu()
    self.statusbar=self.parent.CreateStatusBar(4)

    self.a=Area(parent)
    self.a.setBorders(4, 30, 30, 30)  # wird in onSize() noch korrigiert
    self.leftMouseIsDown=False
    self.ctrlIsDown=False
    self.shiftIsDown=False

    self.hilite=""              # der Name der hervorzuhebenden Kurve (1T, 1H, ...)
    self.last_click_pos=None    # die letzte Klick-Position
    self.last_click_cnt=0       # die wievielte Kurve soll hervorgehoben werden
    self.click_selectivity=20   # der Abstand, wie genau eine Kurve getroffen werden muss

    # hier wird ein unsichtbares TextCtrl() zur Kommunikation mit dem KurvenSettings-Fenster
    # angelegt und eine Event-Funktion beim Event "change" definiert
    pan=wx.Panel(self, size=(0, 0))
    self.curveSettingsFeedback=wx.TextCtrl(pan, -1, pos=(0,0), size=(0,0), style=wx.DEFAULT)
    self.Bind(wx.EVT_TEXT, self.processCurveSettingsFeedback, self.curveSettingsFeedback)
    self.curveSettingsFrame=None

    self.SetBackgroundStyle(wx.BG_STYLE_CUSTOM)

    self.winMoveTimer=wx.Timer(self)
    self.Bind(wx.EVT_TIMER, self.onWinMoveTimer, self.winMoveTimer)

    self.parent.Bind(wx.EVT_CLOSE,  self.onClose)
    self.parent.Bind(wx.EVT_MOVE,   self.onMove)
    self.Bind(wx.EVT_CONTEXT_MENU,  self.onContextMenu)
    self.Bind(wx.EVT_PAINT,         self.onPaint)
    self.Bind(wx.EVT_SIZE,          self.onSize)
    self.Bind(wx.EVT_LEFT_UP,       self.onLeftUp)
    self.Bind(wx.EVT_LEFT_DOWN,     self.onLeftDown)
    self.Bind(wx.EVT_MOTION,        self.onMotion)
    self.Bind(wx.EVT_KEY_DOWN,      self.onKeyDown)
    self.Bind(wx.EVT_KEY_UP,        self.onKeyUp)
    self.Bind(wx.EVT_MOUSEWHEEL,    self.onMouseWheel)

    self.timeWasPrinted=False

    # ohne SetFocus() kommen EVT_SET_FOCUS, EVT_KEY_UP und EVT_KEY_DOWN nicht
    wx.FutureCall(10, self.SetFocus)
    if open_cs==1:
      self.men_open_curveSettings()

  # ###########################################################
  # Schließt bei Bedarf das KurvenSettings-Fenster, bevor das
  # eigene Fenster geschlossen wird.
  def onClose(self, event):
    if self.curveSettingsFrame is not None:
      self.curveSettingsFrame.Close()
    self.parent.Destroy()


  # ###########################################################
  # Korrigiert die von der Maus gelieferten Koordinaten, wenn
  # die Mauscursor-Spitze nicht genau passt.
  def fixMouseCursorPos(self, p):
    p.x-=1
    p.y-=2
    return(p)


  # ###########################################################
  # Start einer Bereichs-Auswahl.
  def onLeftDown(self, event):
    self.sel_end_point=self.sel_start_point=self.fixMouseCursorPos(event.GetPosition())
    self.SetCursor(wx.StockCursor(wx.CURSOR_CROSS))
    self.dc.SetLogicalFunction(wx.XOR)
    self.dc.SetBrush(wx.Brush(self.color_selection, wx.TRANSPARENT))
    self.dc.SetPen(wx.Pen(self.color_selection, 1, wx.SOLID))
    self.setColor(self.color_selection, style=wx.DOT) # wx.LONG_DASH wx.DOT
    self.leftMouseIsDown=True
    self.ctrlIsDown=False

    # Skalenwerte zur aktuell angeklickten Position in die StatusBar eintragen
    self.leftDownPos=self.getScalaValueForPoint(self.sel_start_point)
    tval=float(self.leftDownPos[1])/self.factor["T"]
    hval=self.leftDownPos[2]
    txt="%s, %.1f°C, %d"%(datetime.datetime.fromtimestamp(self.leftDownPos[0]).strftime("%d.%m.%Y %H:%M:%S"), tval, hval)
    self.statusbar.SetStatusText(txt, 0)
    
    tval=float(tval)*self.factor["T"]
    hval=float(hval)*self.factor["H"]
    if (self.leftDownPos[0]%60)>30:
      xa=60*(self.leftDownPos[0]//60+1)
    else:
      xa=60*(self.leftDownPos[0]//60)
    
    vft=self.data.getValuesForTimestamp(xa)
    self.click_dist=list()
    for cn, v in vft.items():
      if cn[-1]=="T":
        self.click_dist.append((cn, int(abs(v-tval))))
      else:
        self.click_dist.append((cn, int(abs(v-hval))))
    f=lambda x: x[1]
    self.click_dist_srt=sorted(self.click_dist, key=f)
    self.hilite=""
    #print self.click_dist_srt
    if len(self.click_dist_srt)>0:  # wenn Kurven an X-Pos sind
      if self.last_click_pos==self.leftDownPos: # wenn selbe Pos wie zuvor geklickt wurde
        if (self.last_click_cnt+1)<len(self.click_dist_srt):  # und noch weitere Kurven da sind
          self.last_click_cnt+=1  # nächste Kurve auswählen
          if self.click_dist_srt[self.last_click_cnt][1]>=self.click_selectivity: # wenn nicht nah genug dran
            self.last_click_cnt=0 # wieder erste Kurve auswählen
      else:
        self.last_click_cnt=0 # an anderer Pos geklickt -> erste Kurve auswählen
      if self.click_dist_srt[0][1]<self.click_selectivity:  # wenn Kurve nah genug an Klick-Pos dran
        self.hilite=self.click_dist_srt[self.last_click_cnt][0] # diese Kurve hervorheben
        self.statusbar.SetStatusText(self.curve_name_dict[self.hilite], 2)  # und ihren Namen in Statusbar setzen
        self.last_click_pos=self.leftDownPos
      else:
        self.statusbar.SetStatusText("", 2)
    else:
      self.statusbar.SetStatusText("", 2)

  # ###########################################################
  # Dragging bzw. Maus-Bewegung zwischen LeftDown und LeftUp.
  def onMotion(self, event):
    if self.leftMouseIsDown==True:
      # altes Auswahl-Objekt löschen bzw. weg-XOR'n
      self.drawSelectionLines(self.sel_start_point, self.sel_end_point)
      # neues Auswahl-Objekt darstellen
      self.sel_end_point=self.fixMouseCursorPos(event.GetPosition())
      self.drawSelectionLines(self.sel_start_point, self.sel_end_point)


  # ###########################################################
  # Zeichnet ein Rechteck mit den Eckpunkten p1 und p2.
  def drawSelectionLines(self, p1, p2):
    self.dc.DrawLine(p1.x, p1.y, p2.x, p1.y)
    self.dc.DrawLine(p2.x, p1.y, p2.x, p2.y)
    self.dc.DrawLine(p2.x, p2.y, p1.x, p2.y)
    self.dc.DrawLine(p1.x, p2.y, p1.x, p1.y)


  # ###########################################################
  # Ende einer Bereichs-Auswahl.
  # Das heißt Zoomen oder Bereich verschieben...
  def onLeftUp(self, event):
    if self.leftMouseIsDown==False:
      return
    self.leftMouseIsDown=False
    self.SetCursor(wx.StockCursor(wx.CURSOR_ARROW)) 
    self.dc.SetLogicalFunction(wx.COPY)
    self.sel_end_point=self.fixMouseCursorPos(event.GetPosition())

    self.leftUpPos=self.getScalaValueForPoint(self.sel_end_point)
    px1=self.leftDownPos[0]
    if px1%60>0:  px1=px1-px1%60+60
    px2=self.leftUpPos[0]
    if px2%60>0:  px2=px2-px2%60+60

    td=Rect(px1, self.leftDownPos[1], px2, self.leftUpPos[1])
    hd=Rect(px1, self.leftDownPos[2], px2, self.leftUpPos[2])
    if td.width>120 and td.height>2 and hd.height>2:
      self.history.append(self.cur_val_boundaries.copy())
      self.cur_val_boundaries.update({"T":td})
      self.cur_val_boundaries.update({"H":hd})
      self.paint()
    else:
      pass
      #print "kein Rechteck ausgewählt"
      self.paint()
    self.statusbar.SetStatusText("Zooms:%d"%(len(self.history),), 1)


  # ###########################################################
  # Liefert zu einem Punkt im Fenster die Skalen-Werte.
  def getScalaValueForPoint(self, (x, y)):
    secs_per_pix=self.cur_val_boundaries["T"].width/float(self.a.boundaries.width)
    px=int((x-self.a.boundaries.x1)*secs_per_pix)+self.cur_val_boundaries["T"].x1

    val_per_pix=self.cur_val_boundaries["T"].height/float(self.a.boundaries.height)
    pyT=(((self.a.screen_height-y)-self.a.boundaries.y1)*val_per_pix+self.cur_val_boundaries["T"].y1)/self.factor["T"]

    val_per_pix=self.cur_val_boundaries["H"].height/float(self.a.boundaries.height)
    pyH=(((self.a.screen_height-y)-self.a.boundaries.y1)*val_per_pix+self.cur_val_boundaries["H"].y1)/self.factor["H"]
    return(int(datetime.datetime.fromtimestamp(px).strftime("%s")), int(round(pyT*self.factor["T"])), int(pyH))


  # ###########################################################
  # Stellt das Kontext-Menue dar.
  def onContextMenu(self, event):
    self.menue.Enable(90, len(self.history)>0)
    self.menue.Enable(92, len(self.history)>0)
    self.cancelSelection()
    self.PopupMenu(self.menue)
    self.Refresh()
    # ohne Refresh ist nur noch der Bereich, an dem das
    # Kontextmenü dargestellt wurde, schreibbar


  # ###########################################################
  # Bricht eine onLeftDown-onMotion-onLeftUp-Folge ab, wenn
  # mittendrin ein onContextMenu auftritt.
  def cancelSelection(self):
    self.leftMouseIsDown=False
    self.SetCursor(wx.StockCursor(wx.CURSOR_ARROW)) 
    self.dc.SetLogicalFunction(wx.COPY)
    self.paint()


  # ###########################################################
  # Stellt die jeweils vorige Zoom-Stufe wieder her.
  def men_undoZoom(self, event):
    self.ctrlIsDown=False
    if len(self.history)>0:
      self.cur_val_boundaries=self.history.pop()
      self.paint()
    self.statusbar.SetStatusText("Zooms:%d"%(len(self.history),), 1)

  
  # ###########################################################
  # Löscht alle Zoom-Stufen.
  def men_delZoom(self, event):
    self.ctrlIsDown=False
    while len(self.history)>0:
      self.cur_val_boundaries=self.history.pop()
    self.paint()
    self.statusbar.SetStatusText("Zooms:%d"%(len(self.history),), 1)


  # ###########################################################
  # Wird bei KeyDown aufgerufen und setzt self.ctrlIsDown gemäß
  # Gedrückt-Zustand der Strg-Taste.
  # Quelle KeyCodes: https://wxpython.org/docs/api/wx.KeyEvent-class.html
  def onKeyDown(self, event):
    if event.GetKeyCode()==wx.WXK_CONTROL:  self.ctrlIsDown=True
    if event.GetKeyCode()==wx.WXK_SHIFT:    self.shiftIsDown=True
    #print "down", event.GetKeyCode(), self.shiftIsDown
    if event.GetKeyCode()==ord("H"):
      self.men_jumpToDate_sub(120)
    if event.GetKeyCode()==ord("G"):
      self.men_jumpToDate_sub(121)
    if event.GetKeyCode()==ord("S"):
      self.men_jumpToDate_sub(122)
    if event.GetKeyCode()==wx.WXK_LEFT and not self.ctrlIsDown:
      self.men_jumpToDate_sub(123)
    if event.GetKeyCode()==wx.WXK_RIGHT and not self.ctrlIsDown:
      self.men_jumpToDate_sub(124)
    if event.GetKeyCode()==wx.WXK_LEFT and self.ctrlIsDown:
      self.men_jumpToDate_sub(125)
    if event.GetKeyCode()==wx.WXK_RIGHT and self.ctrlIsDown:
      self.men_jumpToDate_sub(126)
    if event.GetKeyCode()==wx.WXK_HOME:
      self.men_jumpToDate_sub(127)
    if event.GetKeyCode()==wx.WXK_END:
      self.men_jumpToDate_sub(128)
    if event.GetKeyCode()==ord("T"):
      self.men_setDur_sub(100)
      self.paint()
    if event.GetKeyCode()==ord("W"):
      self.men_setDur_sub(101)
      self.paint()
    if event.GetKeyCode()==ord("M"):
      self.men_setDur_sub(102)
      self.paint()
    if event.GetKeyCode()==ord("A"):
      self.men_setDur_sub(103)
      self.paint()
    if event.GetKeyCode()==ord("V"):
      self.men_setDur_sub(104)
      self.paint()
    if event.GetKeyCode()==ord("K"):
      if self.curveSettingsFrame is None: # wenn SubFenster nicht offen
        self.men_open_curveSettings()     # öffen
      else:                               # sonst
        self.curveSettingsFrame.Close()   # schließen

    if event.GetKeyCode()==wx.WXK_F10:
      self.onContextMenu(event)
    event.Skip(True)

  # ###########################################################
  # Wird bei KeyUp aufgerufen und setzt self.ctrlIsDown auf
  # False.
  def onKeyUp(self, event):
    if event.GetKeyCode()==wx.WXK_CONTROL:  self.ctrlIsDown=False
    if event.GetKeyCode()==wx.WXK_SHIFT:    self.shiftIsDown=False
    #print "up  ", event.GetKeyCode(), self.shiftIsDown
    event.Skip(True)

  # ###########################################################
  # Scrollen mit dem MouseWheel verarbeiten.
  def onMouseWheel(self, event):
    if self.ctrlIsDown:
      bh_steps=float(self.cur_val_boundaries["H"].height)/self.a.gridY_cnt
      bt_steps=float(self.cur_val_boundaries["T"].height)/self.a.gridY_cnt
      t=self.cur_val_boundaries["H"]
      if event.GetWheelRotation()>0:
        bh1=self.cur_val_boundaries["H"].y1+bh_steps
        bh2=self.cur_val_boundaries["H"].y2+bh_steps
        bt1=self.cur_val_boundaries["T"].y1+bt_steps
        bt2=self.cur_val_boundaries["T"].y2+bt_steps
      else:
        bh1=self.cur_val_boundaries["H"].y1-bh_steps
        bh2=self.cur_val_boundaries["H"].y2-bh_steps
        bt1=self.cur_val_boundaries["T"].y1-bt_steps
        bt2=self.cur_val_boundaries["T"].y2-bt_steps
      self.cur_val_boundaries.update({"H":Rect(t.x1, bh1, t.x2, bh2)})
      self.cur_val_boundaries.update({"T":Rect(t.x1, bt1, t.x2, bt2)})
    else:
      b_steps=self.cur_val_boundaries["T"].width/self.a.gridX_cnt
      b_steps-=b_steps%60   # dafür sorgen, dass "b_steps" immer ganzzahlig durch 60 teilbar ist
      bh=self.cur_val_boundaries["H"]
      bt=self.cur_val_boundaries["T"]
      if event.GetWheelRotation()<0:
        t1=self.cur_val_boundaries["H"].x1+b_steps
        t2=self.cur_val_boundaries["H"].x2+b_steps
      else:
        t1=self.cur_val_boundaries["H"].x1-b_steps
        t2=self.cur_val_boundaries["H"].x2-b_steps
      self.cur_val_boundaries.update({"H":Rect(t1, bh.y1, t2, bh.y2)})
      self.cur_val_boundaries.update({"T":Rect(t1, bt.y1, t2, bt.y2)})
    self.paint()


  # ###########################################################
  # Aktualisiert das Fenster.
  def onPaint(self, event):
    self.paint()


  # ###########################################################
  # Aktualisiert das Fenster.
  def paint(self):
    wx.BeginBusyCursor()
    self.dc=wx.AutoBufferedPaintDC(self)
    self.dc.SetBackground(wx.Brush(self.color_background))
    self.dc.SetTextForeground(self.color_text)
    self.dc.Clear()
    self.a.setDC(self.dc)

    minutes=self.cur_val_boundaries["T"].width/60
    self.a.recalculate(minutes)

    self.scale_x=float(self.a.boundaries.width-1)/float(minutes)

    self.drawGrid()
    self.drawCurve()
    wx.EndBusyCursor()


  # ###########################################################
  # Resize des Fensters.
  def onSize(self, event):
    # wenn der self.statusbar.GetSize() bereits in __init__
    # aufgerufen wird, liefert er zu kleine Werte.
    self.a.setBorders(4, 30+self.statusbar.GetSize()[1], 30, 40)
    self.paint()
    self.Refresh()


  # ###########################################################
  # Fenter wird/wurde verschoben.
  # Da während des Verschiebens nicht immer Refreshed werden
  # soll, wird hier nur ein Timer restartet, so dass erst dann
  # der Refresh erfolgt, wenn 200ms lang keine Bewegung erkannt
  # wurde.
  def onMove(self, event):
    self.winMoveTimer.Start(200, oneShot=True)  # [Re]start timer


  # ###########################################################
  # Hier landet er, wenn nach einer Fenster-Verschiebung 200ms
  # keine weitere Verschiebung erkannt wurde.
  def onWinMoveTimer(self, event):
    # der Refresh ist nötig, da sonst nach einer Verschiebung
    # drawSelectionLines() nicht mehr korrekt zeichnet.
    self.paint()
    self.Refresh()


  # ###########################################################
  # Stellt die Skala dar.
  def drawGrid(self):
    l_steps=float(self.cur_val_boundaries["T"].height)/self.a.gridY_cnt
    r_steps=float(self.cur_val_boundaries["H"].height)/self.a.gridY_cnt

    self.setColor(self.color_grid, style=wx.DOT)
    ymin=float(self.cur_val_boundaries["T"].y1)/self.factor["T"]
    ymax=float(self.cur_val_boundaries["T"].y2)/self.factor["T"]
    zl=self.a.boundaries.height/(ymax-ymin)*abs(ymin)
    zl=self.a.screen_height-(self.a.boundaries.y1+zl)
    self.dc.DrawLine(self.a.boundaries.x1, zl, self.a.boundaries.x2, zl)

    self.setColor(self.color_grid, style=wx.LONG_DASH)
    for y in range(0, self.a.gridY_cnt+1):
      yp=self.a.screen_height-(self.a.boundaries.y1+y*self.a.gridY_pix_per_step)
      a=(10 if y<self.a.gridY_cnt else 0) # beim letzten Strich ganz rechts nicht verlängern
      self.dc.DrawLine(self.a.boundaries.x1-a, yp, self.a.boundaries.x2+a, yp)
      if y<self.a.gridY_cnt:  # kein Text über der obersten Linie
        txt="%4.1f"%((float(self.cur_val_boundaries["T"][1])+y*l_steps)/self.factor["T"],)
        w, h=self.dc.GetTextExtent(txt)
        self.dc.DrawText(txt, self.a.boundaries.x1-w-2, yp-h)    # linke Seite (Temperatur)
        txt="%.1f"%((self.cur_val_boundaries["H"][1]+y*r_steps)/self.factor["H"],)
        w, h=self.dc.GetTextExtent(txt)
        self.dc.DrawText(txt, self.a.boundaries.x2+2, yp-h)      # rechte Seite (Prozent)
    txt="°C"
    w, h=self.dc.GetTextExtent(txt)
    self.dc.DrawText(txt, self.a.boundaries.x1-w-2, 8)

    last_day=""
    last_minute=""
    for x in range(0, self.a.gridX_cnt+1):
      xp=self.a.boundaries.x1+x*self.a.gridX_pix_per_step
      a=(10 if x<self.a.gridX_cnt else 0)
      self.dc.DrawLine(xp, self.a.screen_height-self.a.boundaries.y1+a, xp, self.a.screen_height-self.a.boundaries.y2)
      if int(xp)<self.a.boundaries.x2-1:  # kein Text nach der letzten Linie
      #if x<self.a.gridX_cnt:
        dt=datetime.datetime.fromtimestamp(self.cur_val_boundaries["H"].x1+x*self.a.minutes_per_step*60)
        minute=dt.strftime("%H:%M")
        if last_minute!=minute:
          self.dc.DrawText(dt.strftime("%H:%M"), xp+2, self.a.screen_height-(self.a.boundaries.y1-3))
          last_minute=minute
        day=DAYS[dt.weekday()]+" "+dt.strftime("%d.%m.%y")
        if last_day!=day:
          self.dc.DrawText(day, xp+2, self.a.screen_height-(self.a.boundaries.y1-3-h))
          last_day=day


  # ###########################################################
  # Zeichnet die eigentliche Kurve entsprechend der Daten.
  def drawCurve(self):
    for cn, descr, is_on, col in self.curve_name:
      if is_on:
        if cn==self.hilite:
          self.setColor(col, 2)
        else:
          self.setColor(col)
        ll=self.data.getLineList(cn, self.a.boundaries, self.cur_val_boundaries[cn[-1]], self.a.screen_height)
        for l in ll:
          self.dc.DrawLines(l)
    return


  # ###########################################################
  # Stellt die Farbe für Linien und Polygone ein.
  def setColor(self, color, width=1, style=wx.SOLID):
    self.dc.SetPen(wx.Pen(color, width=width, style=style))
    self.dc.SetBrush(wx.Brush(color))


  # ###########################################################
  # Liefert zum Timestamp "ts" Datum+Zeit als String.
  def TimestampToStr(self, ts):
    return(datetime.datetime.fromtimestamp(ts).strftime("%Y.%m.%d %H:%M:%S"))


  # ###########################################################
  # Legt das Kontext-Menue an.
  def createMenu(self):
    self.menue=wx.Menu()
    self.submenueCols=wx.Menu()
    self.submenueDur=wx.Menu()
    self.submenueJumpTo=wx.Menu()
    self.menue.Append(90, 'letzten Zoom zurücknehmen')
    self.menue.Append(92, 'Zoom auf Grundstellung')
    self.menue.AppendSeparator()
    self.menue.AppendSubMenu(self.submenueJumpTo, 'springe zu Datum')
    self.menue.AppendSeparator()
    self.menue.AppendSubMenu(self.submenueDur, 'angezeigte Zeitdauer setzen')
    self.menue.AppendSubMenu(self.submenueCols, 'Kurven anzeigen/verbergen')
    self.menue.Append(130, 'Kurven-Settings\tK')
    self.menue.AppendSeparator()
    self.menue.Append(110, 'Settings speichern')
    self.menue.AppendSeparator()
    self.menue.Append(200, 'Über LoggerView')

    self.submenueJumpTo.Append(120, '&heute - 00:00 Uhr\tH')
    self.submenueJumpTo.Append(121, 'gestern - 00:00 Uhr\tG')
    self.submenueJumpTo.Append(122, 'letzter Sonntag - 00:00 Uhr\tS')
    self.submenueJumpTo.Append(123, 'einen Tag zurück\tLEFT')
    self.submenueJumpTo.Append(124, 'einen Tag vorwärts\tRIGHT')
    self.submenueJumpTo.Append(125, 'eine Woche zurück\tCTRL+LEFT')
    self.submenueJumpTo.Append(126, 'eine Woche vorwärts\tCTRL+RIGHT')
    self.submenueJumpTo.Append(127, 'Anfang der Aufzeichnung\tHOME')
    self.submenueJumpTo.Append(128, 'Ende der Aufzeichnung\tEND')

    self.submenueDur.Append(100, 'ein Tag\tT')
    self.submenueDur.Append(101, 'eine Woche\tW')
    self.submenueDur.Append(102, 'ein Monat\tM')
    self.submenueDur.Append(103, 'alles\tA')
    self.submenueDur.AppendSeparator()
    self.submenueDur.Append(104, 'vertikal zurücksetzen\tV')

    for i in range(len(self.curve_name)):
      self.submenueCols.Append(300+i, self.curve_name[i][1], "", True)
      self.Bind(wx.EVT_MENU, self.showOrHideCols, id=300+i)
      self.submenueCols.Check(300+i, self.curve_name[i][2])

    for i in range(120, 129):
      self.Bind(wx.EVT_MENU, self.men_jumpToDate,       id=i)

    self.Bind(wx.EVT_MENU, self.men_undoZoom,           id=90)
    self.Bind(wx.EVT_MENU, self.men_delZoom,            id=92)
    self.Bind(wx.EVT_MENU, self.men_setDur,             id=100)
    self.Bind(wx.EVT_MENU, self.men_setDur,             id=101)
    self.Bind(wx.EVT_MENU, self.men_setDur,             id=102)
    self.Bind(wx.EVT_MENU, self.men_setDur,             id=103)
    self.Bind(wx.EVT_MENU, self.men_setDur,             id=104)
    self.Bind(wx.EVT_MENU, self.men_saveConfig,         id=110)
    self.Bind(wx.EVT_MENU, self.men_open_curveSettings, id=130)
    self.Bind(wx.EVT_MENU, self.aboutDialog,            id=200)


  # ###########################################################
  # Schaltet gemäß Menüauswahl Kurven an oder aus.
  def showOrHideCols(self, event):
    col=event.GetId()-300   # Nummer der umzuschaltenden Kurve
    self.curve_name[col][2]=self.submenueCols.IsChecked(event.GetId())
    if self.curveSettingsFrame is not None: # wenn SubFenster offen -> auch darin ändern
      self.curveSettingsFrame.cols[col].SetValue(self.curve_name[col][2])


  # ###########################################################
  # Ändert gemäß Menüauswahl den angezeigten Zeitbereich.
  def men_setDur(self, event):
    self.men_setDur_sub(event.GetId())
  def men_setDur_sub(self, e):
    if e==100:
      dur=60*60*24
    elif e==101:
      dur=60*60*24*7
    elif e==102:
      dur=60*60*24*31
    elif e==103:
      self.cur_val_boundaries=self.boundaries.copy()
      return
    elif e==104:
      sH=self.boundaries["H"]
      sT=self.boundaries["T"]
      vH=self.cur_val_boundaries["H"]
      vT=self.cur_val_boundaries["T"]
      self.cur_val_boundaries.update({"H":Rect(vH.x1, sH.y1, vH.x2, sH.y2)})
      self.cur_val_boundaries.update({"T":Rect(vT.x1, sT.y1, vT.x2, sT.y2)})
      return

    for k, v in self.cur_val_boundaries.items():
      x1=v.x1
      if self.boundaries[k].x2<v.x1:          # Anzeigebereich würde hinter den Daten liegen
        x1=self.boundaries[k].x2-dur+60
      elif (v.x1+dur)<self.boundaries[k].x1:  # Anzeigebereich würde vor den Daten liegen
        x1=self.boundaries[k].x1
      self.cur_val_boundaries.update({k:Rect(x1, v.y1, x1+dur, v.y2)})


  # ###########################################################
  # About-Box darstellen.
  def aboutDialog(self, event):
    info=wx.AboutDialogInfo()
    info.SetName("LoggerView")
    info.SetVersion(VERSION)
    info.SetCopyright("D.A.  (01.2017)")
    info.SetDescription("Ein Programm zur Anzeige von Logger-Messwerten.")
    info.SetLicence("""
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.""")
    info.AddDeveloper("Detlev Ahlgrimm")
    wx.AboutBox(info)


  # ###########################################################
  # Speichert Fenster-Größe und Position sowie die Einstellung
  # der anzuzeigenden Kurven.
  def men_saveConfig(self, event):
    fc=wx.FileConfig(localFilename=CONFIG_FILENAME)
    sp=self.parent.GetScreenPosition()
    ss=self.parent.GetSizeTuple()
    fc.WriteInt("pos_x",    sp[0])
    fc.WriteInt("pos_y",    sp[1])
    fc.WriteInt("size_x" ,  ss[0])
    fc.WriteInt("size_y" ,  ss[1])
    for i in range(len(self.curve_name)):
      fc.WriteInt("curve_isOn_%s"%(self.curve_name[i][0],), self.curve_name[i][2])
      fc.Write("curve_col_%s"%(self.curve_name[i][0],), self.curve_name[i][3])
    if self.curveSettingsFrame is None: # wenn SubFenster nicht offen ist
      fc.WriteInt("cs_is_open",  0)
      fc.WriteInt("cs_pos_x",    self.curveSettings_screenPos[0])
      fc.WriteInt("cs_pos_y",    self.curveSettings_screenPos[1])
      fc.WriteInt("cs_size_x" ,  self.curveSettings_sizeTuple[0])
      fc.WriteInt("cs_size_y" ,  self.curveSettings_sizeTuple[1])
    else: # SubFenster ist offen -> aktuelle Daten holen
      fc.WriteInt("cs_is_open",  1)
      fc.WriteInt("cs_pos_x",    self.curveSettingsFrame.GetScreenPosition()[0])
      fc.WriteInt("cs_pos_y",    self.curveSettingsFrame.GetScreenPosition()[1])
      fc.WriteInt("cs_size_x" ,  self.curveSettingsFrame.GetSizeTuple()[0])
      fc.WriteInt("cs_size_y" ,  self.curveSettingsFrame.GetSizeTuple()[1])
    fc.Flush()


  # ###########################################################
  # Setzt für den angezeigten Bereich den neuen linken
  # Startpunkt auf "new_x1".
  def setNewVisibleStart(self, new_x1):
    new_x1-=new_x1%60
    h=self.cur_val_boundaries["H"]
    t=self.cur_val_boundaries["T"]

    self.cur_val_boundaries.update({"H":Rect(new_x1, h.y1, new_x1+h.width, h.y2)})
    self.cur_val_boundaries.update({"T":Rect(new_x1, t.y1, new_x1+t.width, t.y2)})
    self.paint()


  # ###########################################################
  # Menüpunkt "springe zu Datum" verarbeiten.
  def men_jumpToDate(self, event):
    self.men_jumpToDate_sub(event.GetId())
  def men_jumpToDate_sub(self, fid):
    if fid==120:     # heute - 00:00 Uhr
      n=datetime.datetime.now()
      new_x1=int(datetime.datetime(n.year, n.month, n.day, 0, 0, 0).strftime("%s"))
      self.setNewVisibleStart(new_x1)
    elif fid==121:   # gestern - 00:00 Uhr
      n=datetime.datetime.now()
      new_x1=int(datetime.datetime(n.year, n.month, n.day, 0, 0, 0).strftime("%s"))
      new_x1-=(24*60*60)
      self.setNewVisibleStart(new_x1)
    elif fid==122:   # letzter Sonntag - 00:00 Uhr
      n=datetime.datetime.now()
      new_x1=int(datetime.datetime(n.year, n.month, n.day, 0, 0, 0).strftime("%s"))
      if n.weekday()<6:   # 0=Montag, ..., 6=Sonntag -> t=6->d=0, t=0->d=1, t=1->d=2
        new_x1-=(n.weekday()+1)*(24*60*60)
      self.setNewVisibleStart(new_x1)
    elif fid==123:   # einen Tag zurück
      new_x1=self.cur_val_boundaries["H"].x1-(24*60*60)
      self.setNewVisibleStart(new_x1)
    elif fid==124:   # einen Tag vorwärts
      new_x1=self.cur_val_boundaries["H"].x1+(24*60*60)
      self.setNewVisibleStart(new_x1)
    elif fid==125:   # eine Woche zurück
      new_x1=self.cur_val_boundaries["H"].x1-7*(24*60*60)
      self.setNewVisibleStart(new_x1)
    elif fid==126:   # eine Woche vorwärts
      new_x1=self.cur_val_boundaries["H"].x1+7*(24*60*60)
      self.setNewVisibleStart(new_x1)
    elif fid==127:   # Anfang der Aufzeichnung
      self.setNewVisibleStart(self.boundaries["H"].x1)
    elif fid==128:   # Ende der Aufzeichnung
      new_x1=self.boundaries["H"].x2-self.cur_val_boundaries["H"].width
      self.setNewVisibleStart(new_x1)


  # ###########################################################
  # Öffnet das Fenster mit den KurvenSettings nicht-modal.
  def men_open_curveSettings(self, event=None):
    if self.curveSettingsFrame is not None:   # wenn schon offen
      self.curveSettingsFrame.Iconize(False)  # ...nur in den Vordergrund holen
      return
    self.curveSettingsFrame=CurveSettingsFrame(self, self.curve_name, 
                                              pos=self.curveSettings_screenPos, 
                                              size=self.curveSettings_sizeTuple)


  # ###########################################################
  # Verarbeitet Änderungen im Kurven-Settings-Fenster.
  def processCurveSettingsFeedback(self, event):
    v=self.curveSettingsFeedback.GetValue()
    cmd=v.split(",")
    if cmd[0]=="close":     # close,746,501,417,139 (ScreenPosition, SizeTuple)
      self.curveSettingsFrame=None
      self.curveSettings_screenPos=(int(cmd[1]), int(cmd[2]))
      self.curveSettings_sizeTuple=(int(cmd[3]), int(cmd[4]))
    elif cmd[0]=="on/off":  # on/off,2T,1
      for i in range(len(self.curve_name)):
        if self.curve_name[i][0]==cmd[1]:
          self.curve_name[i][2]=int(cmd[2])
          self.submenueCols.Check(300+i, self.curve_name[i][2])
    elif cmd[0]=="color":   # color,2H,#2503F9
      for i in range(len(self.curve_name)):
        if self.curve_name[i][0]==cmd[1]:
          self.curve_name[i][3]=str(cmd[2])
    self.paint()




# ###########################################################
# Der Fenster-Rahmen für das Hauptfenster.
class LoggerViewFrame(wx.Frame):
  def __init__(self, parent, pos=wx.DefaultPosition, size=wx.DefaultSize):
    style=wx.DEFAULT_FRAME_STYLE
    wx.Frame.__init__(self, None, wx.ID_ANY, "LoggerView "+VERSION,
                      pos=pos, size=size, style=style)
    self.panel=LoggerViewWindow(self)



# ###########################################################
# Führt das Kommando "cmd_list" im Betriebssystem aus und
# liefert True, wenn dessen Returncode 0 ist.
# Bei Übergabe von "returncode_only" mit False wird
# stattdessen ein Tupel aus Returncode und Textausgabe
# geliefert.
def exec_cmds(cmd_list, returncode_only=True):
  process=subprocess.Popen(cmd_list,  shell=False, 
                                      stdout=subprocess.PIPE,
                                      stderr=subprocess.STDOUT)
  rc=process.communicate()[0]
  if(returncode_only):
    return(process.returncode==0)
  return(process.returncode, rc.decode().split("\n"))



# ###########################################################
# Main-Loop
if __name__=='__main__':
  fc=wx.FileConfig(localFilename=CONFIG_FILENAME)
  spx=fc.ReadInt("pos_x", -1)
  spy=fc.ReadInt("pos_y", -1)
  ssx=fc.ReadInt("size_x", 1450)
  ssy=fc.ReadInt("size_y", 250)
  sp=(spx, spy) # (-1, -1) entspricht wx.DefaultPosition
  ss=(ssx, ssy) # (-1, -1) entspricht wx.DefaultSize
  del fc

  if len(sys.argv)>1:
    if sys.argv[1]=="-c":
      exec_cmds(["scp", "11w:/nfs/vms/ss/logger.sqlite", "/ramdisk"])
      DATABASE_FILENAME=os.path.join("/ramdisk", "logger.sqlite")
  elif os.path.exists("/ramdisk"):
    exec_cmds(["scp", "11w:/nfs/vms/ss/logger.sqlite", "/ramdisk"])
    DATABASE_FILENAME=os.path.join("/ramdisk", "logger.sqlite")

  app=wx.App(False)
  frame=LoggerViewFrame(None, pos=sp, size=ss)
  frame.Show(True)
  app.MainLoop()
