#include <Time.h>
#include <ESP8266WiFi.h>

#include "globals.h"
#include "wlan_comm.h"
#include "ds18b20.h"
#include "sensors.h"
#include "debug_helper.h"

#define HOSTNAME "Logger-02"
#define FIRMWARE "2.0"

#define LED_ON  LOW
#define LED_OFF HIGH

int sync_hour_g=11;     // zu Beginn dieser Stunde werden die gesammelten Daten zum SocketServer gesendet

int error;              // zur Steuerung der Blink-Frequenz der Lebenszeichen-LED
//  alles 0 => Normalzustand
//  Bit0==1 => letzter Sync ist fehlgeschlagen
//  Bit1==1 => Kennung für Minuten-Wechsel

#define REC_BUF_SIZE 200
uint8_t rec_buf[REC_BUF_SIZE];



/* -------------------------------------------------------------------
 *  Init nach Restart.
 */
void setup() {
  time_t cur;
  int got_time;

  #if DEBUG==1
  Serial.begin(115200);
  Serial.println();
  #endif

  pinMode(ALIVE_LED_PIN, OUTPUT);
  pinMode(SPECIAL_SYNC_BUTTON_PIN, INPUT_PULLUP);
  pinMode(DS18B20_DATA_PIN, INPUT);
  digitalWrite(ALIVE_LED_PIN, HIGH);  // LED aus

  error=0;          // Lebenszeichen-LED normal blinken lassen
  warnings_g=1;     // Restart vermerken
  if(digitalRead(SPECIAL_SYNC_BUTTON_PIN)==0) {
    warnings_g|=2;  // vermerken, dass das Python-Script einen neuen Standort-Satz anlegen soll
  }

  init_sensorMinute();  // Speicher auf ungültig setzen
  sensorMinute_idx_g=0;
  init_sensorHour();
  init_sensorDay();

  while(discoverOneWireDevices()!=1) {
    digitalWrite(ALIVE_LED_PIN, LED_ON);
    delay(100);
    digitalWrite(ALIVE_LED_PIN, LED_OFF);
  }
  readSensors();  // Dummy-Read

  got_time=0;
  while(got_time==0) {  // ohne Zeit macht es keinen Sinn, setup() zu verlassen
    digitalWrite(ALIVE_LED_PIN, LED_ON);
    if(connectWLAN()) {   // Zeit vom SocketServer holen und die Uhr entsprechend stellen
      if(connectSocketServer()) {
        sendSocketServer((uint8_t*)"GET_TIME", 8);
        if(receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000)>=19) {  // z.B.: 2017.02.02 14:14:14
          stringToTime(rec_buf);
          power_on_timestamp_g=now();
          got_time=1;
          error=0;
        }
        delay(100);
        disconnectSocketServer();
      } else {
        error=1;
      }
      delay(100);
      disconnectWLAN();
    } else {
      error=1;
    }
    digitalWrite(ALIVE_LED_PIN, LED_OFF);
    delay(5000);
  }

  if(second()>52) { // mindestens zwei Messwerte für den ersten Minuten-Mittelwert
    while(second()!=0) {
      blinkAliveLED(0);
      delay(10);
    }
  }
  
  cur=now();
  done_second_g=-1;           // "diese Sekunde wurde bereits bearbeitet" auf "nicht diese" setzen -> also sofort Messwert holen
  cur_minute_g=minute(cur);   // "diese Minute wird gerade bearbeitet"
  cur_hour_g=hour(cur);       // "diese Stunde wird gerade bearbeitet"
  done_day_g=-1;              // "dieser Tag wurde bereits bearbeitet" auf "nicht heute" setzen
  done_sync_minute_g=-1;      // "in dieser Minute wurde schon synchronisiert" auf "nicht diese Minute" setzen
  done_sync_day_g=-1;         // "ursprünglicher Tag eines fehlgeschlagenen Sync's" auf "kein Sync-Fehler" setzen

  if(hour(cur)==sync_hour_g) {  // wenn die Sync-Stunde gerade läuft
    // sind noch keine Daten da, die gesendet werden könnten
    done_day_g=day(cur);        // daher braucht es heute keinen Sync mehr
  }

  sensorHour_g.timestamp=buildTimestamp(cur);  // für aktuelle Stunde initialisieren
}


/* -------------------------------------------------------------------
 *  Hauptschleife.
 */
void loop() {
  static time_t cur;
  int rc;

  cur=now();

  // wenn es Zeit ist, einen Messwert zu holen, die Minute noch nicht abgelaufen ist und er nicht schon geholt wurde
  if((second(cur)-2)%5==0 && minute(cur)==cur_minute_g && done_second_g!=second(cur)) {
    #if DEBUG==1
    printTimestamp(cur);  Serial.print("   ");  Serial.print(warnings_g, DEC);  Serial.print("   ");
    #endif
    sensorMinute_g[sensorMinute_idx_g]=readSensors();   // Messwerte holen und ablegen
    #if DEBUG==1
    Serial.print("Messwerte holen und ablegen in sensorMinute_g[");
    Serial.print(sensorMinute_idx_g, DEC);
    Serial.println("]");
    Serial.print(sensorMinute_g[sensorMinute_idx_g].temp/16.0);
    Serial.print("  ");
    Serial.print(sensorMinute_g[sensorMinute_idx_g].ldr);
    Serial.print("  ");
    Serial.println(sensorMinute_g[sensorMinute_idx_g].bad, DEC);
    #endif
    sensorMinute_idx_g++;
    done_second_g=second(cur);  // diese Sekunde wurde bereits bearbeitet
    blinkAliveLED(error);
  }

  // wenn die Minute um ist
  if(cur_minute_g!=minute(cur)) {
    #if DEBUG==1
    printTimestamp(cur);  Serial.print("   ");  Serial.print(warnings_g, DEC);  Serial.print("   ");
    #endif
    sensorHour_g.data[cur_minute_g]=getSensorMinuteAvg();   // Mittelwerte über sensorMinute_g bilden und ablegen
    #if DEBUG==1
    Serial.print("Mittelwerte über sensorMinute_g bilden und ablegen in sensorHour_g.data[");
    Serial.print(cur_minute_g, DEC);
    Serial.println("]");
    Serial.print(sensorHour_g.data[cur_minute_g].temp/16.0);
    Serial.print("  ");
    Serial.print(sensorHour_g.data[cur_minute_g].ldr);
    Serial.print("  ");
    Serial.println(sensorHour_g.data[cur_minute_g].bad, DEC);
    hexdump(&sensorMinute_g, sizeof(sensorMinute_g));
    #endif
    init_sensorMinute();
    sensorMinute_idx_g=0;
    cur_minute_g=minute(cur);   // diese Minute wird gerade bearbeitet
    blinkAliveLED(error | 2);
  }

  // wenn die Stunde um ist
  if(cur_hour_g!=hour(cur)) {
    sensorDay_g[cur_hour_g]=sensorHour_g; // abgelaufene Stunde umkopieren
    sensorDay_g[cur_hour_g].checksum=getSensorHourChecksum((uint8_t*)&sensorDay_g[cur_hour_g]);
    #if DEBUG==1
    hexdump(&sensorHour_g, SIZEOF_SensorHour);
    Serial.print("sensorHour_g kopieren nach sensorDay_g[");
    Serial.print(cur_hour_g, DEC);
    Serial.println("]");
    #endif
    init_sensorHour();
    sensorHour_g.timestamp=buildTimestamp(cur); // für neue Stunde initialisieren
    cur_hour_g=hour(cur);       // diese Stunde wird gerade bearbeitet
  }

  // wenn der Tag noch nicht bearbeitet wurde und (die Sync-Stunde läuft oder ein voriger Sync noch nicht erfolgreich war)
  // und (für den Fall, das nicht erfolgreich gesendet werden konnte) die Minute ganzzahlig durch 5 teilbar ist
  // und diese Minute hier noch nicht bearbeitet wurde
  if(done_day_g!=day(cur) && (hour(cur)==sync_hour_g || done_sync_minute_g!=-1) && (minute(cur)%5)==0 && done_sync_minute_g!=minute(cur)) {
    #if DEBUG==1
    Serial.println("Daten senden");
    #endif
    rc=sendAllHours(0);
    if(rc==1) {
      if(hour(cur)!=hour()) {   // wenn die Zeit in die vorige Stunde zurück-korrigiert wurde
        if(now()<cur) {         // wenn tatsächlich zurück-korrigiert wurde
          while(minute()!=0 || second()!=0) {   // auf Beginn der neuen Stunde warten
            blinkAliveLED(0);
            delay(10);
          }
        }
      }
      init_sensorDay();
      if(done_sync_day_g!=-1) {       // wenn es ein Sync-Neuversuch war
        done_day_g=done_sync_day_g;   // den Tag für Sync blocken, für den der wiederholte Sync gedacht war
        done_sync_day_g=-1;           // gemerkten Sync-Tag löschen
      } else {
        done_day_g=day(cur);          // heute kein Sync mehr
      }
      done_sync_minute_g=-1;          // beim nächsten Sync sofort Wiederholung erlauben
      error&=0xfe;                    // Bit0 (Sync-Error) ausschalten
    } else {                          // wenn der Sync fehlgeschlagen ist
      if(done_sync_day_g==-1) {       // wenn sich bisher noch kein Sync-Tag gemerkt wurde
        done_sync_day_g=day(cur);     // ursprünglichen Tag des fehlgeschlagenen Sync's merken
      }
      done_sync_minute_g=minute(cur); // erst in 5 Minuten wieder einen Sync-Versuch starten
      error|=1;                       // Bit0 (Sync-Error) einschalten
    }
  }

  if(digitalRead(SPECIAL_SYNC_BUTTON_PIN)==0) {
    #if DEBUG==1
    printTimestamp(cur);  Serial.print("   ");  Serial.print(warnings_g, DEC);  Serial.print("   ");
    Serial.println("Sonder-Sync: Daten senden");
    #endif
    rc=sendAllHours(1);
    #if DEBUG==1
    if(rc==1) {
      Serial.print("zeit:");
      Serial.println((char*)rec_buf);
    }
    #endif
  }

  delay(100);
}


/* -------------------------------------------------------------------
 *  Lässt sie Lebenszeichen-LED abhängig von "error" kurz blinken.
 */
void blinkAliveLED(int error) {
  if(error==0) {                  // normales Blinken (1x kurz)
    digitalWrite(ALIVE_LED_PIN, LED_ON);
    delayMicroseconds(100);
    digitalWrite(ALIVE_LED_PIN, LED_OFF);
  }
  if((error&1)>0) {               // Sync-Fehler (1x lang)
    digitalWrite(ALIVE_LED_PIN, LED_ON);
    delay(10);
    digitalWrite(ALIVE_LED_PIN, LED_OFF);
  }
  if((error&2)>0) {               // Minutenwechsel (2x kurz)
    blinkAliveLED(0);
    delay(100);
    blinkAliveLED(0);
  }
}


/* -------------------------------------------------------------------
 *  Stellt eine Verbindung zum WLAN und zum SocketServer her und
 *  - bei Übergabe von 0 wird sensorDay_g gesendet und die Zeit neu
 *    gestellt.
 *  - bei Übergabe von 1 wird sensorDay_g und sensorHour_g gesendet.
 *  Danach werden die Verbindungen zum SocketServer und WLAN getrennt.
 *  Liefert 1, wenn alle Daten erfolgreich gesendet wurden. Sonst 0.
 */
int sendAllHours(int send_cur_hour) {
  int flg, retries, all_good, h;
  SensorHour *ptr;

  if(!connectWLAN()) {
    return(0);
  }
  if(!connectSocketServer()) {
    disconnectWLAN();
    return(0);
  }
  all_good=1;

  // Identifizierungs-Daten senden
  for(flg=retries=0; retries<3; retries++) {
    snprintf((char*)rec_buf, REC_BUF_SIZE, "IDENT=%s,%s,1,%04d.%02d.%02d %02d:%02d:%02d,%d,5.0", HOSTNAME, FIRMWARE, 
      (int)year(power_on_timestamp_g), (int)month(power_on_timestamp_g), (int)day(power_on_timestamp_g), 
      (int)hour(power_on_timestamp_g), (int)minute(power_on_timestamp_g), (int)second(power_on_timestamp_g), warnings_g);
    sendSocketServer(rec_buf, strlen((char*)rec_buf));
    if(receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000)>=2) {  // OK oder BAD
      if(strcmp((char*)rec_buf, "OK")==0) {
        flg=1;          // Kennung für "IDENT erfolgreich"
        warnings_g=0;   // wenn erfolgreich gemeldet wurde, ist erstmal wieder alles gut
        break;
      }
    }
  }
  if(flg==0) {  all_good=0; } // das Senden hat "retries" mal nicht geklappt

  // Sensor-Daten aus sensorDay_g senden
  for(h=0; h<24 && all_good==1; h++) {
    for(flg=retries=0; retries<3; retries++) {
      blinkAliveLED(1);
      if(sensorDay_g[h].checksum==getSensorHourChecksum((uint8_t*)&sensorDay_g[h])) {
        memcpy(rec_buf, "SHOUR.1=", 8);
        memcpy((char*)rec_buf+8, &sensorDay_g[h], SIZEOF_SensorHour);
        #if DEBUG==1
        hexdump((char*)rec_buf, SIZEOF_SensorHour+8);
        #endif
        sendSocketServer(rec_buf, SIZEOF_SensorHour+8);
        if(receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000)>=2) {  // OK oder BAD
          if(strcmp((char*)rec_buf, "OK")==0) {
            flg=1;
            break;
          }
        }
      } else {
        flg=1;  // nicht senden ist immer erfolgreich
        break;
      }
    }
    if(flg==0) {  all_good=0; }
  }

  if(send_cur_hour==1) {
    // Sensor-Daten aus sensorHour_g senden
    for(flg=retries=0; retries<3 && all_good==1; retries++) {
      blinkAliveLED(1);
      sensorHour_g.checksum=getSensorHourChecksum((uint8_t*)&sensorHour_g);
      memcpy(rec_buf, "SHOUR.1=", 8);
      memcpy((char*)rec_buf+8, &sensorHour_g, SIZEOF_SensorHour);
      #if DEBUG==1
      hexdump((char*)rec_buf, SIZEOF_SensorHour+8);
      #endif
      sendSocketServer(rec_buf, SIZEOF_SensorHour+8);
      if(receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000)>=2) {  // OK oder BAD
        if(strcmp((char*)rec_buf, "OK")==0) {
          flg=1;
          break;
        }
      }
    }
    if(flg==0) {  all_good=0; }
  }

  if(send_cur_hour==0) {
    for(flg=retries=0; retries<3 && all_good==1; retries++) {
      blinkAliveLED(1);
      sendSocketServer((uint8_t*)"GET_TIME", 8);
      if(receiveSocketServer(rec_buf, REC_BUF_SIZE, 2000)>=19) {  // z.B.: 2017.02.02 14:14:14
        stringToTime(rec_buf);    // Zeit neu setzten
        flg=1;
        break;
      }
    }
    if(flg==0) {  all_good=0; }
  }

  delay(90);
  disconnectSocketServer();
  delay(10);
  disconnectWLAN();

  return(all_good);
}

